#!/usr/bin/perl

use strict;
use warnings FATAL => 'all';
use utf8;
use open qw(:std :utf8);

use Test::Deep qw(cmp_deeply);
use Test::More;

use JSON::XS;
use File::Slurp qw(read_file write_file);

use lib::abs qw(../lib/);

use CreateChecksAndRules;
use Partner2::Juggler::API;

use Util qw(get_partner2_juggler_api);
use Constants qw($PARTNER2_ENDPOINTS);

my $DATA_PATH = lib::abs::path('/data/');


main();

sub main {
    my $coder = JSON::XS->new->utf8->canonical->pretty->allow_nonref;

    my $pja = get_partner2_juggler_api();
    my $checks = CreateChecksAndRules->new();

    my %checks = ();

    {
        no warnings 'redefine';
        no strict 'refs';

        *{'Partner2::Juggler::API::call'} = sub {
            my ($self, $method, $url, %opts) = @_;

            my $uniq_name = _get_uniq_name($opts{'data'});

            ok(!exists($checks{$uniq_name}), "unique $uniq_name");

            $checks{$uniq_name} = $opts{'data'};

            return { success => 1 };
        };
    }

    my $result = $checks->create($pja, $CreateChecksAndRules::CHECKS_DIR, 'Check');

    ok($result, 'no errors');

    my %files = map {my ($uniq_name) = $_ =~ /$DATA_PATH\/(.+?)\.json$/;
        ($uniq_name => $_)} glob("$DATA_PATH/*");

    if ($ENV{'SELF_UPDATE'}) {
        my @not_exist_files = grep {!exists($checks{$_})} keys(%files);

        if (@not_exist_files) {
            unlink @files{@not_exist_files};

            delete(@files{@not_exist_files});
        }

        foreach my $uniq_name (sort keys(%checks)) {
            my $json = eval {$coder->encode($checks{$uniq_name})};
            if ($@) {
                warn "ERROR $uniq_name";
                die $@;
            }

            write_file("$DATA_PATH/$uniq_name.json", $json);

            $files{$uniq_name} = "$DATA_PATH/$uniq_name.json";
        }
    }

    ok(!(grep {!exists($files{$_})} keys(%checks)), 'all checks have test data');
    ok(!(grep {!exists($checks{$_})} keys(%files)), 'test data is up to date');

    foreach my $uniq_name (sort keys(%checks)) {
        next unless exists($files{$uniq_name});

        my $expected_json = read_file("$DATA_PATH/$uniq_name.json");

        cmp_deeply($checks{$uniq_name}, $coder->decode($expected_json), "Check $uniq_name");
    }

    done_testing();
}

sub _get_uniq_name {
    my ($check) = @_;

    return "$check->{'host'}__$check->{'service'}";
}
