use Mojolicious::Lite;

use Business::PFR;
use Business::IBAN;
use Business::SWIFT;
use CHI;
use Carp;
use Cpanel::JSON::XS;
use File::Basename;
use File::Slurp;
use Data::Dumper;
use HTTP::Tiny;
use Hash::Util qw(lock_keys);
use Path::Tiny;

use lib::abs qw(
  ../lib/
  );

use Branches;
use FormConstants qw(
  @API_TYPES_LIST
  $CONTRACT_OFFER_ERROR_MAIL
  $DEFAULT_PROJECT
  $SAVE_ATTACHMENT_MAX_FILE_SIZE
  $SAVE_ATTACHMENT_MIN_FILE_SIZE
  $SENTRY_URL
  $PARTNER2_CRON_MAIL
  $PARTNERS_MAIL
  $PARTNERS_TEST_MAIL
  $TVM_BLACKBOX_ID
  );
use Monitoring;
use Partner2;
use Submit;
use Type::inn_ph;
use Type::inn_ur;
use Type::ogrnip;
use Utils;
use Yandex::Blackbox;
use Yandex::TVM qw(get_x_ya_service_ticket);

# Add args info into trace frames while Mojo::Exception is called (used by send_to_sentry)
Utils::hack_mojo_exception_trace();

hook before_dispatch => sub {
    my ($c) = @_;

    my $project = ($c->param('project') // $DEFAULT_PROJECT);
    return $c->render(
        json => {
            success       => JSON::XS::false,
            error_message => "unknown project",
        },
        status => 400,
    ) unless (is_valid_project($project));
    $c->stash(project => $project);

    my $yb = Yandex::Blackbox->new(
        url                 => $ENV{FORM_BLACKBOX},
        x_ya_service_ticket => get_x_ya_service_ticket($TVM_BLACKBOX_ID),
    );

    my $host = $c->{tx}->{req}->{content}->{headers}->header('X-Real-Host') // '';
    $host =~ s/(.*)(:[0-9]+)/$1/;

    my $data;
    my $is_logged;
    eval {
        $data = $yb->sessionid(
            sessionid    => $c->cookie('Session_id'),
            sslsessionid => $c->cookie('sessionid2'),
            userip       => $c->{tx}->{req}->{content}->{headers}->header('X-Real-IP'),
            host         => $host,
        );
        $is_logged = ($data->{status}->{id} // '') eq '0';
    };
    $c->stash(user_data => $data);

    if ($is_logged) {
        my $canonical_login = lc($data->{login});
        $canonical_login =~ s/\./-/g;

        # Правильнее использовать библиотеку lang-detect
        my $language = ($data->{dbfields}->{'userinfo.lang.uid'} // '') eq 'en' ? 'en' : 'ru';

        $c->stash(is_logged => 1);
        $c->stash(
            user => {
                user_id  => $data->{uid}->{value},
                language => $language,

                # в случае если пользователь вошел через социальноую авторизацию и еще не создал логин,
                # то в этих полях будет по пустой строке
                display_login   => $data->{login},
                canonical_login => $canonical_login,
            }
        );

        $c->stash(version => ($c->cookie('form_version') || get_form_version($data->{uid}{value}) || 1));
    } else {
        $c->stash(is_logged => 0);

        my @enpoints_without_auth = qw(
          /form/api/0/current_user
          /form/api/0/alive
          /form/api/0/die
          );

        if (!grep {$c->req->url->path->to_string() eq $_} @enpoints_without_auth) {
            return $c->render(
                json   => {success => JSON::XS::false,},
                status => 403,
            );
        }
    }
};

hook before_render => sub {
    my ($c, $args) = @_;

    my $status = $args->{status} || '200';
    my $method = (($c->req->url->to_string() =~ m|/form/api/0/(.+?)\b|)[0] // 'unknown');

    send_to_solomon(path => join(' ', 'status', $method, $status),);

    if ($status eq '404') {
        $args->{json} = {
            success           => JSON::XS::false,
            error_description => 'not found',
        };
    }

    if ($status eq '500') {
        my $id;
        my $err;
        my $user = $args->{snapshot}->{user};
        my $exception = $args->{exception} // $c->stash->{exception};
        unless (eval {$id = send_to_sentry(controller => $c, exception => $exception, user => $user,);}) {
            $err = $@ // '';
            print STDERR join(
                "\n",
                'sentry error: ' . $err,
                '',
                ($exception->{type} // '<nt>') . ': ' . $exception->to_string,
                '    CallStack:',
                (
                    map {
                        sprintf(q[        %s() called at '%s' line %s], map {$_ // 'undef'} $_->[3], $_->[1], $_->[2])
                      } @{$exception->{frames}}
                ),
                Dumper($exception->{extra})
            );
        }

        unless (
            eval {
                local $Data::Dumper::Sortkeys = 1;
                send_email(
                    to => (_is_production_stage() ? $CONTRACT_OFFER_ERROR_MAIL : $PARTNERS_TEST_MAIL),
                    (_is_production_stage() ? (bcc => $PARTNER2_CRON_MAIL) : ()),
                    from => $PARTNERS_MAIL,
                    subject =>
                      sprintf('PI form failed - login=%s', ($user->{canonical_login} // $user->{user_id} // '')),
                    body => join(
                        "\n", '<pre>',
                        sprintf(
'Пользователь не смог зарегистрироваться в анкете ПИ (%s)',
                            $c->stash('project')),
                        '-' x 40,
                        sprintf('Error: %s', $exception->to_string),
                        sprintf('Login: %s', ($user->{canonical_login} // '')),
                        sprintf('ID: %s',    ($user->{user_id}         // '')),
                        (
                            $exception->{extra} && $exception->{extra}{api_url}
                            ? (
                                sprintf('Была вызвана ручка %s', $exception->{extra}{api_url}),
                                (
                                    $exception->{extra}{api_response}
                                    ? (
                                        'которая вернула ответ:',
                                        Dumper($exception->{extra}{api_response})
                                      )
                                    : ()
                                ),
                              )
                            : ()
                        ),
                        '',
                        'Технические детали:',
                        '-' x 40,
                        (
                            $id
                            ? sprintf('Событие в сентри: %s?query=%s', $SENTRY_URL, $id)
                            : ()
                        ),
                        ($err ? sprintf('Ошибка сентри: %s', $err) : ()),
                        '',
                        sprintf('%s: %s', $exception->{type}, $exception->to_string),
                        '    CallStack:',
                        (
                            map {
                                sprintf(
                                    q[        %s() called at '%s' line %s],
                                    map {$_ // 'undef'} $_->[3],
                                    $_->[1], $_->[2]
                                  )
                              } @{$exception->{frames}}
                        ),
                        Dumper($exception->{extra}),
                        '</pre>',
                    )
                ) if ($method eq 'submit');

                return 1;
            }
          )
        {
            my $err = $@ // '';
            print STDERR "send_email error: $err\n";
        }

        $args->{json} = {
            success           => JSON::XS::false,
            error_description => 'internal error',
            (
                _is_dev_stage()
                ? (($id ? (sentry_url => "$SENTRY_URL?query=" . $id,) : ()), ($err ? (sentry_error => $err,) : ()),)
                : ()
            ),
        };
    }
};

get '/form/api/0/alive' => sub {
    my ($c) = @_;

    $c->render(json => {success => JSON::XS::true,},);
};

get '/form/api/swagger.json' => sub {
    my ($c) = @_;

    return $c->render(status => 404) if not _is_dev_stage();

    my $data = read_file '/data/swagger.json';

    $c->render(json => decode_json $data );
};

get '/form/api/0/dispatcher' => sub {
    my ($c) = @_;

    my $project   = $c->stash('project');
    my $countries = Countries->new(project => $project);
    my $branches  = Branches->new(
        version => $c->stash('version'),
        project => $project,
    );

    my $result = {};

    my $user = $c->stash('user');

    my %country_id2name =
      map {$_->{country_id} => $_->{country_name}} $countries->get_working_countries(lang => $user->{language},);

    my $branch_ids2country_ids = $branches->branch_ids2country_ids();

    for my $branch_id (keys %$branch_ids2country_ids) {
        delete $branch_ids2country_ids->{$branch_id} if $branch_id =~ /part2$/;
    }

    my $oferta = get_data_from_json_file dirname(__FILE__) . '/../data/oferta.json';

    foreach my $branch_id (keys %{$branch_ids2country_ids}) {
        foreach my $country_id (@{$branch_ids2country_ids->{$branch_id}}) {
            $result->{$country_id}->{country_name} = $country_id2name{$country_id};

            my $is_oferta = $branches->is_oferta($branch_id);
            $result->{$country_id}->{branches}->{$branch_id} = {
                branch_name => $branches->get_branch_name($branch_id, $user->{language}),
                oferta => ($is_oferta ? JSON::XS::true : JSON::XS::false),
                ($is_oferta ? (oferta_id => $branches->get_oferta_id($branch_id) . '') : ()),
                (
                    $is_oferta
                    ? (oferta_url => $oferta->{$branches->get_oferta_id($branch_id)}->{$user->{language}})
                    : ()
                ),
            };
        }
    }

    foreach my $country_id (keys %{$result}) {
        $result->{$country_id}->{order} = [sort {_sort_branch_ids($a, $b)} keys %{$result->{$country_id}->{branches}}];
    }

    $c->render(json => $result);
};

get '/form/api/0/current_user' => sub {
    my ($c) = @_;

    if ($c->stash('is_logged')) {
        my $user = $c->stash('user');

        # В метрике есть штука под названием "Аналитика форм". Эта штука открывает по настоящему сайт
        # и сверху рисует дополнительные данные. Анкета редиректит пользователей на /v2/dashbaord
        # если у них есть роли в ПИ2. В такой ситуации посмотреть "Аналитику форм" не получается.
        # Поэтому сделан специальный хак. Если установлена особая кука, то на фронт передаются такие данные, чтобы фронт
        # не делал редирект, а показывал анкету (но submit анкеты все равно выдаст ошибку, так как
        # в ручке сабмита этого хака нет)

        if ($c->cookie('can_fill_form')) {

            $c->render(
                json => {
                    is_logged            => JSON::XS::true,
                    user_id              => $user->{user_id},
                    display_login        => $user->{display_login},
                    canonical_login      => $user->{canonical_login},
                    can_fill_form        => JSON::XS::true,
                    has_role_in_partner2 => JSON::XS::false,
                    language             => $user->{language},
                    version              => $c->stash('version'),
                },
            );

        } else {

            my $user_data = get_partner2_user_data($user->{user_id});

            my ($can_fill_form, $reason, $message) = can_fill_form(
                canonical_login => $user->{canonical_login},
                can_fill_part2  => $user_data->{can_fill_part2},
                language        => $user->{language},
                partner2_db     => $user_data->{db},
                representative  => $user_data->{representative},
                version         => $c->stash('version'),
            );

            my $project;
            my $part2;
            my $country;
            if ($user_data->{'db'}) {
                if ($user_data->{'db'}{'user'} && $user_data->{'db'}{'user'}{'is_mobile_mediation'}) {
                    $project = 'mobile_mediation';
                } elsif ($user_data->{'db'}{'user'} && $user_data->{'db'}{'user'}{'is_efir_blogger'}) {
                    $project = 'efir_blogger';
                } elsif ($user_data->{'db'}{'user'} && $user_data->{'db'}{'user'}{'is_video_blogger'}) {
                    $project = 'video_blogger';
                } elsif ($user_data->{'db'}{'user'} && $user_data->{'db'}{'user'}{'is_games'}) {
                    $project = 'games';
                } elsif ($user_data->{'db'}{'form_data'} && $user_data->{'db'}{'form_data'}{'data'}) {
                    my $data = Cpanel::JSON::XS->new->decode($user_data->{'db'}{'form_data'}{'data'});
                    $project = $data->{'project'};
                }
                if (   $user_data->{'db'}{'form_data'}
                    && $user_data->{'db'}{'form_data'}{'branch_id'}
                    && ($part2 = Branches->get_branch_part2($user_data->{'db'}{'form_data'}{'branch_id'})))
                {
                    $country = $user_data->{'db'}{'form_data'}{'country_id'};
                }
            }

            $c->render(
                json => {
                    is_logged       => JSON::XS::true,
                    user_id         => $user->{user_id},
                    display_login   => $user->{display_login},
                    canonical_login => $user->{canonical_login},
                    can_fill_form   => ($can_fill_form > 0 ? JSON::XS::true : JSON::XS::false),
                    ($can_fill_form <= 0 && $reason  ? (can_fill_form_reason  => $reason)  : ()),
                    ($can_fill_form <= 0 && $message ? (can_fill_form_message => $message) : ()),
                    has_role_in_partner2 => ($user_data->{has_role_in_partner2} ? JSON::XS::true : JSON::XS::false),
                    (
                             $user_data->{can_fill_part2}
                          && $part2 ? (force => {branch_id => $part2, country_id => $country}) : ()
                    ),
                    language => $user->{language},
                    ($project ? (project => $project) : ()),
                    version => $c->stash('version'),
                },
            );

        }

    } else {
        $c->render(json => {is_logged => JSON::XS::false,},);
    }
};

get '/form/api/0/pfr' => sub {
    my ($c) = @_;

    my $value = $c->param('value');

    if ($value) {
        my $bp = Business::PFR->new(value => $value,);

        $c->render(json => {is_valid => ($bp->is_valid() ? JSON::XS::true : JSON::XS::false),},);
    } else {
        $c->render(
            json => {
                success       => JSON::XS::false,
                error_message => "must specify parameter 'value'",
            },
            status => 400,
        );
    }
};

get '/form/api/0/iban' => sub {
    my ($c) = @_;

    my $value = $c->param('value');

    if ($value) {
        if ($value !~ / /) {
            my $bi = Business::IBAN->new();
            $c->render(json => {is_valid => ($bi->valid($value) ? JSON::XS::true : JSON::XS::false),},);
        } else {
            $c->render(json => {is_valid => JSON::XS::false,},);
        }
    } else {
        $c->render(
            json => {
                success       => JSON::XS::false,
                error_message => "must specify parameter 'value'",
            },
            status => 400,
        );
    }
};

get '/form/api/0/swift' => sub {
    my ($c) = @_;

    my $value = $c->param('value');

    if (defined($value) && length($value) > 0) {

        my $bank_data = get_bank_by_swift($value);

        $c->render(json => {is_valid => $bank_data->{found} ? JSON::XS::true : JSON::XS::false,},);

    } else {
        $c->render(
            json => {
                success       => JSON::XS::false,
                error_message => "must specify parameter 'value'",
            },
            status => 400,
        );
    }
};

get '/form/api/0/bik' => sub {
    my ($c) = @_;

    my $value = $c->param('value');

    if (defined($value) && length($value) > 0) {

        my $bank_data = get_bank_by_bik($value);

        $c->render(json => {is_valid => $bank_data->{found} ? JSON::XS::true : JSON::XS::false,},);

    } else {
        $c->render(
            json => {
                success       => JSON::XS::false,
                error_message => "must specify parameter 'value'",
            },
            status => 400,
        );
    }

};

get '/form/api/0/ogrnip' => sub {
    my ($c) = @_;

    my $value = $c->param('value');

    if ($value) {
        $c->render(
            json => {
                is_valid => (
                      Type::ogrnip->new(language => 'ru')->is_valid($value)->{is_valid}
                    ? JSON::XS::true
                    : JSON::XS::false
                ),
            },
        );
    } else {
        $c->render(
            json => {
                success       => JSON::XS::false,
                error_message => "must specify parameter 'value'",
            },
            status => 400,
        );
    }
};

get '/form/api/0/inn_ur' => sub {
    my ($c) = @_;

    my $value = $c->param('value');

    if ($value) {
        $c->render(
            json => {
                is_valid => (
                      Type::inn_ur->new(language => 'ru')->is_valid($value)->{is_valid}
                    ? JSON::XS::true
                    : JSON::XS::false
                ),
            },
        );
    } else {
        $c->render(
            json => {
                success       => JSON::XS::false,
                error_message => "must specify parameter 'value'",
            },
            status => 400,
        );
    }
};

get '/form/api/0/inn_ph' => sub {
    my ($c) = @_;

    my $value = $c->param('value');

    if ($value) {
        $c->render(
            json => {
                is_valid => (
                      Type::inn_ph->new(language => 'ru')->is_valid($value)->{is_valid}
                    ? JSON::XS::true
                    : JSON::XS::false
                ),
            },
        );
    } else {
        $c->render(
            json => {
                success       => JSON::XS::false,
                error_message => "must specify parameter 'value'",
            },
            status => 400,
        );
    }
};

post '/form/api/0/adfox_account' => sub {
    my ($c) = @_;

    my $data;
    eval {$data = decode_json $c->req->body;};

    my $user        = $c->stash('user');
    my $login       = $data->{login};
    my $password    = $data->{password};
    my $adfox_offer = $data->{adfox_offer} // '';
    my $branch_id   = $data->{branch_id} // '';

    if ($login && $password) {
        my $result = check_pi_adfox_contracts(
            adfox_login         => $login,
            adfox_password      => $password,
            allow_paid_services => $adfox_offer,
            branch_id           => $branch_id,
            lang                => $user->{language},
            user_id             => $user->{user_id},
        );

        if ($result->{contracts}) {
            $c->render(json => {is_valid => JSON::XS::true,},);
        } elsif ($result->{error_message}) {
            $c->render(
                json => {
                    is_valid      => JSON::XS::false,
                    error_message => $result->{error_message},
                    error_token   => $result->{error_token},
                },
            );
        } else {
            croak 'Wrong result from check_pi_adfox_contracts. This should not happen.';
        }
    } else {
        $c->render(
            json => {
                is_valid      => JSON::XS::false,
                error_message => gettext("Пустой логин или пароль", $user->{language}),
            },
            status => 200,
        );
    }
};

get '/form/api/0/branch' => sub {
    my ($c) = @_;

    my $id      = $c->param('id');
    my $user    = $c->stash('user');
    my $project = $c->stash('project');
    my $version = $c->stash('version');

    my $branches = Branches->new(
        version => $version,
        project => $project,
    );

    if ($id) {
        if ($branches->is_valid_branch_id($id)) {

            my @fields;

            foreach my $field_id ($branches->get_fields($id)) {
                my $p = "Field::$field_id";
                my $f = $p->new(
                    language => $user->{language},
                    project  => $project,
                    version  => $c->stash('version'),
                );

                my $field_data = $f->get_api_description($c->stash('user_data'));

                push @fields, $field_data;
            }

            $c->render(json => {fields => \@fields,},);
        } else {
            $c->render(
                json => {
                    success       => JSON::XS::false,
                    error_message => "unknown branch id",
                },
                status => 400,
            );
        }
    } else {
        $c->render(
            json => {
                success       => JSON::XS::false,
                error_message => "must specify parameter 'id'",
            },
            status => 400,
        );
    }
};

get '/form/api/0/type' => sub {
    my ($c) = @_;

    return $c->render(status => 404) if not _is_dev_stage();

    my $id = $c->param('id');

    if ($id) {
        if (-e "/app/lib/Type/" . $id . ".pm") {

            require "Type/${id}.pm";
            my $p = "Type::$id";
            my $t = $p->new(language => 'ru');

            $c->render(text => $t->get_description(),);
        } else {
            $c->render(
                json => {
                    success       => JSON::XS::false,
                    error_message => "unknown type id",
                },
                status => 400,
            );
        }
    } else {
        $c->render(
            json => {
                success       => JSON::XS::false,
                error_message => "must specify parameter 'id'",
            },
            status => 400,
        );
    }
};

get '/form/api/0/types' => sub {
    my ($c) = @_;

    return $c->render(status => 404) if not _is_dev_stage();

    $c->render(json => \@API_TYPES_LIST,);
};

post '/form/api/0/rm_user_in_db' => sub {
    my ($c) = @_;

    return $c->render(status => 404) if not _is_dev_stage();

    my $user_id = $c->param('user_id');

    return $c->render(_get_400_error("must specify user_id")) unless $user_id;
    return $c->render(_get_400_error("incorrect user_id")) unless $user_id =~ /^[0-9]+\z/;

    rm_user_in_db($user_id);

    return $c->render(json => {success => JSON::XS::true,},);
};

post '/form/api/0/submit' => sub {
    my ($c) = @_;

    my $data;

    eval {$data = decode_json $c->req->body;};

    return $c->render(
        json => {
            success       => JSON::XS::false,
            error_message => "Expected valid json as a body",
        },
        status => 400,
    ) unless $data;

    return $c->render(
        json => {
            success       => JSON::XS::false,
            error_message => "Incorrect data",
        },
        status => 400,
    ) if exists $data->{user_id};

    my $project = delete $data->{'project'};
    $project //= $DEFAULT_PROJECT;
    return $c->render(
        json => {
            success       => JSON::XS::false,
            error_message => "unknown project",
        },
        status => 400,
    ) unless (is_valid_project($project));
    $c->stash(project => $project);

    my $user      = $c->stash('user');
    my $user_data = get_partner2_user_data($user->{user_id});
    my $version   = $c->stash('version');

    my ($can_fill_branch, $reason, $message) = can_fill_branch(
        branch_id       => $data->{branch_id},
        canonical_login => $user->{canonical_login},
        can_fill_part2  => $user_data->{can_fill_part2},
        language        => $user->{language},
        partner2_db     => $user_data->{db},
        representative  => $user_data->{representative},
        version         => $version,
    );

    return $c->render(
        json => {
            success => JSON::XS::false,
            errors  => {
                global => [
                    {
                        description => (
                            $message || sprintf(
                                gettext(
"Пользователь с логином %s не может быть зарегистрирован. Пожалуйста, выберите другой логин",
                                    $user->{language}
                                ),
                                $user->{display_login}
                            )
                        ),
                    },
                ],
            },
        },
        status => 400,
    ) if !$can_fill_branch;

    # При повторной отправке формы после ошибки при условии, что пользователь всё же зарегистрировался
    return $c->render(json => {success => JSON::XS::true,},) if $can_fill_branch < 0;

    my $s = Submit->new(
        {
            %{$data},
            user_id       => $user->{user_id},
            display_login => $user->{display_login},
        },
        $user->{language},
        $project, $version,
    );

    my ($is_valid, $errors) = $s->submit();

    if ($is_valid) {
        return $c->render(json => {success => JSON::XS::true,},);
    } else {
        return $c->render(
            json => {
                errors  => $errors,
                success => JSON::XS::false,
            },
            status => 400,
        );
    }
};

post '/form/api/0/save_attachment' => sub {
    my ($c) = @_;

    my $upload = $c->req->upload('upload');
    return $c->render(_get_400_error("must specify upload")) unless $upload;

    my $size = $upload->size();
    return $c->render(
        json   => {success => JSON::XS::false,},
        status => 400,
    ) if ($size < $SAVE_ATTACHMENT_MIN_FILE_SIZE || $size > $SAVE_ATTACHMENT_MAX_FILE_SIZE);

    my $file_name = $upload->filename();
    my $data      = $upload->slurp();

    my $user = $c->stash('user');

    my $key = save_attachment($user->{user_id}, 'application/json', $data, $file_name);

    return $c->render(
        json => {
            success => JSON::XS::true,
            key     => $key,
        },
    );
};

get '/form/api/0/query_fias' => sub {
    my ($c) = @_;

    return $c->render(
        json => {
            success => JSON::XS::true,
            data    => query_fias(parent_guid => $c->param('parent_guid')),
        },
    );
};

get '/form/api/0/die' => sub {
    my ($c) = @_;

    my $sleep = $c->param('sleep');

    sleep $sleep if $sleep;

    die "Test die";
};

sub _is_dev_stage {
    return (($ENV{FORM_STAGE} // '') eq 'dev');
}

sub _is_production_stage {
    return (($ENV{FORM_STAGE} // '') eq 'production');
}

sub _get_400_error {
    my ($message) = @_;

    return (
        json => {
            success       => JSON::XS::false,
            error_message => $message,
        },
        status => 400,
    );
}

sub _sort_branch_ids {
    my ($one, $two) = @_;

    my @order = qw(
      russia_ph
      russia_ph_part1
      russia_ph_part2
      russia_ur
      russia_ur_part1
      russia_ur_part2
      russia_ip
      russia_ip_part1
      russia_ip_part2
      belarus_ph_part1
      belarus_ph_part2
      belarus_ur_part1
      belarus_ur_part2
      belarus_ip
      belarus_ip_part1
      belarus_ip_part2
      );

    my %id2position;
    for (my $i = 0; $i < @order; $i++) {
        $id2position{$order[$i]} = $i;
    }

    return (($id2position{$one} // 0) <=> ($id2position{$two} // 0))
      || $one cmp $two;
}

app->start;
