package Branches;

use strict;
use warnings FATAL => 'all';
use feature 'say';
use utf8;
use open qw(:std :utf8);

use Carp;

use Countries;
use Utils;
use File::Basename;

my $BRANCH_FOR_ASSESSORS = 'assessor';

=head2 new

    my $branches = Branches->new();

=cut

sub new {
    my ($class, %opts) = @_;

    $opts{version} ||= 1;
    my $path = $opts{version} == 2 ? '/v2' : '';

    my %branch_ids =
      map {m'^.*/(.*)\.json\z'; $1 => $_} glob dirname(__FILE__) . '/../data/branches' . $path . '/*.json';

    my $countries = Countries->new(%opts);

    if ($opts{project} && $opts{project} eq 'assessor') {
        my $branch = $branch_ids{$BRANCH_FOR_ASSESSORS};
        %branch_ids = ($BRANCH_FOR_ASSESSORS => $branch);
    } else {
        delete $branch_ids{$BRANCH_FOR_ASSESSORS};
    }
    my $self = {
        _branch_ids  => \%branch_ids,
        _country_ids => {map {$_->{country_id} => 1} $countries->get_working_countries(lang => 'ru')},
        %opts,
    };
    bless $self, $class;

    return $self;
}

sub get_all_branches {
    my ($self) = @_;

    return [sort keys %{$self->{_branch_ids}}];
}

=head2 is_valid_branch_id

    my $bool = $branches->is_valid_branch_id('ag_ph');

=cut

sub is_valid_branch_id {
    my ($self, $branch_id) = @_;

    return $self->get_branch_path($branch_id) ? 1 : 0;
}

sub get_branch_path {
    my ($self, $branch_id) = @_;

    return exists($self->{_branch_ids}->{$branch_id}) && $self->{_branch_ids}->{$branch_id};
}

=head2 get_fields

    my @fields = $branches->get_fields('ag_ph');

=cut

sub get_fields {
    my ($self, $branch_id, $fields_key) = @_;

    $self->_check_branch_id_and_get_data($branch_id);

    return @{$self->{'_branches'}->{$branch_id}->{$fields_key // 'fields'}};
}

=head2 is_oferta

    my $bool = $branches->get_fields('ag_ph');

=cut

sub is_oferta {
    my ($self, $branch_id) = @_;

    $self->_check_branch_id_and_get_data($branch_id);

    return !!$self->{'_branches'}->{$branch_id}->{oferta};
}

=head2 get_oferta_id

    my $oferta_id = $branches->get_fields('ag_ph');

=cut

sub get_oferta_id {
    my ($self, $branch_id) = @_;

    $self->_check_branch_id_and_get_data($branch_id);

    croak "Branch does not work with oferta" unless $self->is_oferta($branch_id);

    return $self->{'_branches'}->{$branch_id}->{oferta_id};
}

=head2 branch_ids2country_ids

    my $b2c = $branches->branch_ids2country_ids();

=cut

sub branch_ids2country_ids {
    my ($self) = @_;

    my $result = {};

    foreach my $branch_id (keys %{$self->{_branch_ids}}) {
        my $data = get_data_from_json_file $self->{_branch_ids}{$branch_id};
        if (exists $data->{country_ids}) {
            $result->{$branch_id} = $data->{country_ids};
        } elsif (exists $data->{except_country_ids}) {

            my %except = map {$_ => 1} @{$data->{except_country_ids}};
            my @ids;
            foreach my $country_id (keys %{$self->{_country_ids}}) {
                if (!exists $except{$country_id}) {
                    push @ids, $country_id;
                }
            }

            $result->{$branch_id} = \@ids;
        } else {
            croak 'Incorrect data.';
        }
    }

    return $result;
}

=head2 get_branch_name

    my $name = $branches->get_branch_name('ag_ph', 'ru');

=cut

sub get_branch_name {
    my ($self, $branch_id, $language) = @_;

    $self->_check_branch_id_and_get_data($branch_id);

    croak "Must specify language" unless $language;

    return gettext($self->{'_branches'}->{$branch_id}->{name_ru}, $language);
}

=head2 working_with_balance

    my $bool = $branches->working_with_balance('ag_ph');

=cut

sub working_with_balance {
    my ($self, $branch_id) = @_;

    $self->_check_branch_id_and_get_data($branch_id);

    return !!$self->{'_branches'}->{$branch_id}->{working_with_balance};
}

=head2 working_with_contract

    my $bool = $branches->working_with_contract('ag_ph');

=cut

sub working_with_contract {
    my ($self, $branch_id) = @_;

    $self->_check_branch_id_and_get_data($branch_id);

    return !!$self->{'_branches'}{$branch_id}{working_with_contract};
}

=head2 get_balance_person_type

    my $person_type = $branches->get_balance_person_type('ag_ph');

=cut

sub get_balance_person_type {
    my ($self, $branch_id) = @_;

    $self->_check_branch_id_and_get_data($branch_id);

    croak "branch_id '$branch_id' is not working with balance" unless $self->working_with_balance($branch_id);

    return $self->{'_branches'}->{$branch_id}->{balance_person_type};
}

=head2 get_balance_contract_currency

    my $person_type = $branches->get_balance_contract_currency('ag_ph');

=cut

sub get_balance_contract_currency {
    my ($self, $branch_id) = @_;

    $self->_check_branch_id_and_get_data($branch_id);

    croak "branch_id '$branch_id' is not working with balance and contract"
      unless $self->working_with_contract($branch_id) || $self->working_with_balance($branch_id);

    return $self->{'_branches'}{$branch_id}{balance_contract_currency};
}

=head2 get_balance_contract_firm_id

    my $person_type = $branches->get_balance_contract_firm_id('ag_ph');

=cut

sub get_balance_contract_firm_id {
    my ($self, $branch_id) = @_;

    $self->_check_branch_id_and_get_data($branch_id);

    croak "branch_id '$branch_id' is not working with balance and contract"
      unless $self->working_with_contract($branch_id) || $self->working_with_balance($branch_id);

    return $self->{'_branches'}{$branch_id}{balance_contract_firm_id};
}

=head2 get_balance_contract_nds

    my $person_type = $branches->get_balance_contract_nds('ag_ph');

=cut

sub get_balance_contract_nds {
    my ($self, $branch_id) = @_;

    $self->_check_branch_id_and_get_data($branch_id);

    croak "branch_id '$branch_id' is not working with balance and contract"
      unless $self->working_with_contract($branch_id) || $self->working_with_balance($branch_id);

    return $self->{'_branches'}{$branch_id}{balance_contract_nds};
}

=head2 get_balance_contract_testmode

    my $person_type = $branches->get_balance_contract_testmode('ag_ph');

=cut

sub get_balance_contract_testmode {
    my ($self, $branch_id) = @_;

    $self->_check_branch_id_and_get_data($branch_id);

    croak "branch_id '$branch_id' is not working with balance and contract"
      unless $self->working_with_contract($branch_id) || $self->working_with_balance($branch_id);

    return $self->{'_branches'}{$branch_id}{balance_contract_testmode};
}

=head2 get_converter_balance

    my $hash = $branches->get_converter_balance('ag_ph');

=cut

sub get_converter_balance {
    my ($self, $branch_id) = @_;

    $self->_check_branch_id_and_get_data($branch_id);

    return $self->{'_branches'}->{$branch_id}->{converter_balance};
}

sub get_converter_requisites {
    my ($self, $branch_id) = @_;

    $self->_check_branch_id_and_get_data($branch_id);

    return $self->{'_branches'}->{$branch_id}->{converter_requisites};
}

=head2 get_converter_partner2

    my $hash = $branches->get_converter_partner2('ag_ph');

=cut

sub get_converter_partner2 {
    my ($self, $branch_id) = @_;

    $self->_check_branch_id_and_get_data($branch_id);

    return $self->{'_branches'}{$branch_id}{converter_partner2};
}

=head2 get_converter_form

    my $hash = $branches->get_converter_form('ag_ph');

=cut

sub get_converter_form {
    my ($self, $branch_id) = @_;

    $self->_check_branch_id_and_get_data($branch_id);

    return $self->{'_branches'}->{$branch_id}->{converter_form};
}

=head2 get_wiki_link

=cut

sub get_wiki_link {
    my ($self, $branch_id) = @_;

    $self->_check_branch_id_and_get_data($branch_id);

    return $self->{'_branches'}->{$branch_id}->{wiki_link};
}

sub _check_branch_id_and_get_data {
    my ($self, $branch_id) = @_;

    croak "Must specify branch_id" unless $branch_id;
    croak "Unknown branch_id $branch_id" unless $self->is_valid_branch_id($branch_id);

    unless ($self->{'_branches'}->{$branch_id}) {
        $self->{'_branches'}->{$branch_id} = get_data_from_json_file $self->get_branch_path($branch_id);
    }

    # Стуктура project_specific_fields позволяет для разных проектов подменять
    # некоторые поля формы или заменять всю их структуру.
    # Пример использования (формат perl, но в конфиге должен быть json):
    # project_specific_fields => {          # Если не указано, то для всех проектов одинаково
    #     project_name => {                 # Список замены для проекта project_name
    #         some_field => other_field,    # Заменяет в списке полей поле some_field на other_field
    #                                       # Если не найдено, игнорируется
    #         some_field => [               # Заменяет в списке полей поле some_field на список
    #             other_field,              # из other_field и one_more_field
    #             one_more_field,
    #         ],
    #         extra_field => undef,         # Удаляет поле extra_field (undef или пустой массив)
    #         extra_field => [ ],
    #     },
    #     other_project => [                # Список замены для проекта other_project
    #         new_field_1,                  # Полностью заменяет дефолтный список полей на новый
    #         new_field_2,
    #     ],
    # },
    if ($self->{'project'}) {    # Если задан специфичный проект
        my $psf = delete $self->{'_branches'}{$branch_id}{'project_specific_fields'};
        if (ref($psf) eq 'HASH') {
            my $conv = $psf->{$self->{'project'}};
            if (ref($conv) eq 'HASH')
            {                    # Если для этого проекта заданы списки замены
                my $fields = $self->{'_branches'}{$branch_id}{'fields'};
                # Бежим по всем дефолтным полям и заменяем их при необходимости
                # Пробежка с конца до начала, чтобы не зависеть от потенциально изменяющейся длины списка
                for (my $i = $#$fields; $i >= 0; $i--) {
                    my $name = $fields->[$i];
                    if (exists $conv->{$name}) {    # Для этого поля есть замена
                        my $value = $conv->{$name};
                        if (defined $value) {
                            my $ref = ref($value);
                            if ($ref eq 'ARRAY') {    # Замена списком
                                splice @$fields, $i, 1, @$value;
                            } elsif (!$ref) {         # Замена значением
                                $fields->[$i] = $value;
                            }
                        } else {    # Удаление
                            splice @$fields, $i, 1;
                        }
                    }
                }
            } elsif (ref($conv) eq 'ARRAY')
            {    # Если для этого проекта задан свой/новый список полей
                $self->{'_branches'}{$branch_id}{'fields'} = $conv;
            }
        }
    }

    return 1;
}

sub is_branch_part1 {
    my ($self, $branch_id) = @_;
    return $branch_id =~ /_part1$/;
}

sub is_branch_part2 {
    my ($self, $branch_id) = @_;
    return $branch_id =~ /_part2$/;
}

sub get_branch_part2 {
    my ($self, $branch_id) = @_;
    if ($branch_id =~ /(.*)_part1$/) {
        return "${1}_part2";
    } else {
        return undef;
    }
}

1;
