package Countries;

use strict;
use warnings FATAL => 'all';
use feature 'say';
use utf8;
use open qw(:std :utf8);

use Carp;
use File::Basename;

use Utils;

my $COUNTRY_FOR_ASSESSORS = 226;

=head2 new

    my $countries = Countries->new();

=cut

sub new {
    my ($class, %opts) = @_;

    my $data        = get_data_from_json_file(dirname(__FILE__) . '/../data/countries.json');
    my @known_langs = keys %{$data->[0]->{name}};

    my %h = map {$_->{country_id} => $_} @{$data};
    if ($opts{project} && $opts{project} eq 'assessor') {
        my $country = $h{$COUNTRY_FOR_ASSESSORS};
        $country->{can_work} = 1;
        %h = ($COUNTRY_FOR_ASSESSORS => $country);
        $data = [$country];
    }

    my $self = {
        _countries_by_id => \%h,
        _data            => $data,
        _known_langs     => \@known_langs,
        %opts,
    };
    bless $self, $class;

    return $self;
}

=head2 is_valid_working_country_id

    my $bool = $countries->is_valid_working_country_id(117);

=cut

sub is_valid_working_country_id {
    my ($self, $country_id) = @_;

    return exists($self->{_countries_by_id}->{$country_id}) && $self->{_countries_by_id}->{$country_id}->{can_work};
}

=head2 get_working_countries

    my @countries = $countries->get_working_countries(
        lang => 'ru',
    );

=cut

sub get_working_countries {
    my ($self, %opts) = @_;

    croak "Must specify lang" if not exists $opts{lang};
    croak "Incorrect lang" if not grep {$opts{lang} eq $_} @{$self->{_known_langs}};

    my @countries;
    foreach my $el (@{$self->{_data}}) {
        if ($el->{can_work}) {
            push @countries,
              {
                country_id   => $el->{country_id},
                country_name => $el->{name}->{$opts{lang}},
              };
        }
    }

    return @countries;
}

=head2 get_country_name

    my $name = $countries->get_country_name(225, 'ru');

=cut

sub get_country_name {
    my ($self, $country_id, $lang) = @_;

    croak "Must specify country_id" unless $country_id;
    croak "Must specify lang"       unless $lang;

    # TODO - validate $lang

    if (exists $self->{_countries_by_id}->{$country_id}) {
        return $self->{_countries_by_id}->{$country_id}->{name}->{$lang};
    } else {
        croak "Unknown country_id " . $country_id;
    }

}

1;
