package Partner2;

use strict;
use warnings FATAL => 'all';
use feature 'say';
use utf8;
use open qw(:std :utf8);

use Carp;
use Cpanel::JSON::XS;
use Encode;
use Exporter;
use HTTP::Tiny;
use LWP::UserAgent;
use Scalar::Util qw(blessed);
use URI;
use URI::Escape;
use URI::QueryParam;

use Utils;

our @ISA       = qw(Exporter);
our @EXPORT_OK = qw(
  check_bik_and_account
  check_pi_adfox_contracts
  create_offer
  create_or_update_partner
  create_person
  fias_hierarchy
  get_bank_by_bik
  get_bank_by_swift
  get_client_id
  get_currency_code
  get_form_version
  get_partner2_user_data
  get_person
  is_yamoney_account_identificated
  list_attachments
  query_fias
  remove_user_client_association
  rm_user_in_db
  save_attachment
  save_data
  send_email
  update_contract
  );
our @EXPORT = @EXPORT_OK;

$ENV{FORM_PARTNER2_SERVER} //= 'localhost';

sub MODIFY_CODE_ATTRIBUTES {
    my ($package, $sub, @attrs) = @_;

    return Utils::setup_sensitive_data_filters($package, $sub, @attrs);
}

sub check_api_result : SENSITIVE_DATA_FILTER_ALL() {
    my ($url, $response, $request, $throw_mojo) = @_;

    $throw_mojo //= 1;

    my $status;
    if (blessed($response) && $response->isa('HTTP::Response')) {
        $status = $response->code();
    } else {
        $status = ($response->{status} // '');
    }
    if ($status ne '200') {
        throw_mojo_exception(
            {
                api_data => {
                    request  => $request,
                    response => $response,
                    subtype  => 'status ' . $status,
                    url      => $url,
                },
            }
        );
    }

    my $data;
    my $error_xml;
    eval {
        my $content;
        if (blessed($response) && $response->isa('HTTP::Response')) {
            $content = $response->content();
        } else {
            $content = $response->{content};
        }
        $data      = decode_json $content;
        $error_xml = $data->{data}{error_message};
    };

    if ($throw_mojo && $error_xml) {
        throw_mojo_exception(
            {
                error_xml => $error_xml,
                api_data  => {
                    request  => $request,
                    response => $response,
                    subtype  => 'error_xml',
                    url      => $url,
                },
            }
        );
    }
    return $data;
}

=head2 get_form_version

    my $version = get_client_id($user_id);

Получает результат из ручки ПИ2 о текущей версии анкеты

L<https://wiki.yandex-team.ru/partner/w/dev/infrastructure/systems/applications/partner/partner2-intapi/form/form-version/>

=cut

sub get_form_version {
    my ($user_id) = @_;
    croak "Must specify user_id" unless $user_id;

    my $url = $ENV{FORM_PARTNER2_SERVER} . '/intapi/form/form_version.json?lang=en&user_id=' . $user_id;

    my $response = HTTP::Tiny->new()->get($url);

    my $result = check_api_result($url, $response);
    my $version = $result->{data}{version} // 1;

    return $version;
}

=head2 get_client_id

    my $client_id = get_client_id($user_id);

Саба ходит в ручку ПИ2 L<https://wiki.yandex-team.ru/partner/w/partner2-intapi-form-client-id/>, получает из нее Client ID и отдает его.

=cut

sub get_client_id {
    my ($user_id) = @_;

    croak "Must specify user_id" unless $user_id;

    my $url = $ENV{FORM_PARTNER2_SERVER} . '/intapi/form/client_id.json?lang=en&user_id=' . $user_id;

    my $response = HTTP::Tiny->new()->get($url);

    my $result = check_api_result($url, $response);
    if (my $client_id = $result->{data}{client_id}) {
        return $client_id;
    } else {
        my $err = $result->{data}{error};
        if (    ref $err eq 'HASH'
            and $err->{txt}
            and $err->{txt} =~ /^Passport .+? \(\d+\) is already linked as representative of client \d+$/)
        {
            return (undef, $err);
        } else {
            throw_mojo_exception(
                {
                    api_data => {
                        response => $response,
                        url      => $url,
                    },
                    type    => 'Partner2::get_client_id',
                    message => sprintf(qq[Can't find out client_id for user_id %s], $user_id),
                }
            );
        }
    }
}

=head2 create_person

    my $person_id = create_person(
        client_id => $client_id,
        operator_uid => $user_id,
        person_id => -1,
        type => 'sw_ytph',
        fields => {
            phone => 123,
            email => 'aa@yandex.ru',
            ...
        },
    );

Саба ходит в ручку ПИ2 L<https://wiki.yandex-team.ru/partner/w/partner2-intapi-form-create-person/>

=cut

sub create_person {
    my (%opts) = @_;

    my $fields = delete $opts{fields};

    my $url = $ENV{FORM_PARTNER2_SERVER} . '/intapi/form/create_person.json';

    my $content = {%opts, %{$fields},};
    my $response = HTTP::Tiny->new()->request(
        'POST', $url,
        {
            headers => {
                'Content-Type'    => 'application/json',
                'Accept-Language' => 'en',
            },
            content => encode_json $content,
        }
    );

    my $result = check_api_result($url, $response, $content, 0);
    if (my $person_id = $result->{data}{person_id}) {
        return $person_id;
    } else {
        my $err = _contract_error($result, $response);
        if ($err) {
            return (undef, $err);
        } else {
            throw_mojo_exception(
                {
                    api_data => {
                        response => $response,
                        url      => $url,
                    },
                    type    => 'Partner2::create_person',
                    message => "Can't find out person_id",
                }
            );
        }
    }
}

=head2 create_or_update_partner

    create_or_update_partner(
        user_id => $user_id,
    );

=cut

sub create_or_update_partner {
    my (%opts) = @_;

    my $user_id = delete $opts{user_id};
    croak "Must specify user_id" unless $user_id;

    my $api_mobile_mediation_data = delete $opts{api_mobile_mediation_data};
    my $adfox_link                = delete($opts{adfox_link}) // {type => 'none'};
    my $url                       = $ENV{FORM_PARTNER2_SERVER} . '/intapi/form/create_or_update_partner.json';
    my $content                   = {
        user_id => $user_id,
        data    => \%opts,
        (
            $api_mobile_mediation_data
            ? (api_mobile_mediation_data => $api_mobile_mediation_data,)
            : (adfox_link => $adfox_link,)
        ),
    };
    my $response = HTTP::Tiny->new()->request(
        'POST', $url,
        {
            headers => {
                'Content-Type'    => 'application/json',
                'Accept-Language' => 'en',
            },
            content => encode_json $content,
        }
    );
    my $result = check_api_result($url, $response, $content);

    if (ref($result) eq 'HASH' && ref($result->{data}) eq 'HASH' && $result->{data}{error}) {
        my $err = $result->{data}{error};
        utf8::decode($err);
        throw_mojo_exception(
            {
                api_data => {
                    response => $response,
                    url      => $url,
                },
                type    => 'Partner2::create_or_update_partner',
                message => $err,
            }
        );
    }

    return 1;
}

=head2 list_attachments

    list_attachments($user_id, $data);

Саба ходит в ручку ПИ2 L<https://wiki.yandex-team.ru/partner/w/partner2-intapi-form-list-attachments/>

=cut

sub list_attachments {
    my ($user_id) = @_;

    croak "Must specify user_id" unless $user_id;

    my $url      = $ENV{FORM_PARTNER2_SERVER} . "/intapi/form/list_attachments.json?lang=en&uid=$user_id";
    my $response = HTTP::Tiny->new()->get($url);
    my $result   = check_api_result($url, $response);

    return $result->{data};
}

=head2 save_attachments

    save_attachment($user_id, $type, $data, $file_name);

Саба ходит в ручку ПИ2 L<https://wiki.yandex-team.ru/partner/w/partner2-intapi-form-save-attachment/>

=cut

sub save_attachment {
    my ($user_id, $type, $data, $file_name) = @_;

    croak "Must specify user_id"   unless $user_id;
    croak "Must specify type"      unless $type;
    croak "Must specify data"      unless $data;
    croak "Must specify file_name" unless $file_name;

    my $url = $ENV{FORM_PARTNER2_SERVER} . '/intapi/form/save_attachment.json';
    $file_name = encode_utf8($file_name);
    my $lwp = LWP::UserAgent->new();
    $lwp->ssl_opts(SSL_ca_path => '/etc/ssl/certs/',);
    my $response = $lwp->post(
        $url,
        'Content-Type' => 'form-data',
        Content        => [
            uid       => $user_id,
            file_name => $file_name,
            data      => [
                undef,
                $file_name,
                'Content-Type' => $type,
                Content        => $data,
            ],
        ],
    );
    my $result = check_api_result(
        $url,
        $response,
        {
            data_length => length($data),
            data_type   => $type,
            file_name   => $file_name,
            uid         => $user_id,
        }
    );

    return $result->{data}{key};
}

=head2 save_data

Саба ходит в ручку L<https://wiki.yandex-team.ru/partner/w/partner2-intapi-form-save_data/>

=cut

sub save_data {
    my ($data) = @_;

    my $url      = $ENV{FORM_PARTNER2_SERVER} . '/intapi/form/save_data.json';
    my $response = HTTP::Tiny->new()->request(
        'POST', $url,
        {
            headers => {
                'Content-Type'    => 'application/json',
                'Accept-Language' => 'en',
            },
            content => encode_json $data,
        }
    );
    my $result = check_api_result($url, $response, $data);

    return 1;
}

=head2 query_fias

Саба ходит в ручку L<https://wiki.yandex-team.ru/partner/w/partner2-intapi-form-query-fias/>

=cut

sub query_fias {
    my (%opts) = @_;

    if ($Application::APPLICATION) {
        require Utils::Fias;
        return Utils::Fias::query_fias($Application::APPLICATION, map {$_ => $opts{$_}} qw(guid parent_guid));
    }

    my $url = $ENV{FORM_PARTNER2_SERVER} . '/intapi/form/query_fias.json?';
    $url .= join(
        '&',
        (
              $opts{guid}        ? "guid=$opts{guid}"
            : $opts{parent_guid} ? "parent_guid=$opts{parent_guid}"
            : ()
        ),
        'lang=en'
    );
    my $response = HTTP::Tiny->new()->get($url);
    my $result = check_api_result($url, $response);

    return $result->{data};
}

=head2 fias_hierarchy

Саба ходит в ручку L<https://wiki.yandex-team.ru/partner/w/partner2-intapi-form-fias-hierarchy/>

=cut

sub fias_hierarchy {
    my ($guid) = @_;

    if ($Application::APPLICATION) {
        require Utils::Fias;
        return Utils::Fias::fias_hierarchy($Application::APPLICATION, guid => $guid);
    }

    my $url      = $ENV{FORM_PARTNER2_SERVER} . '/intapi/form/fias_hierarchy.json?lang=en&guid=' . $guid;
    my $response = HTTP::Tiny->new()->get($url);
    my $result   = check_api_result($url, $response);

    return $result->{data};
}

=head2 get_partner2_user_data

Саба ходит в ручку L<https://wiki.yandex-team.ru/partner/w/partner2-intapi-form-user/>

Пример (такого user_id в базе ПИ2 нет):

    my $user_data = get_partner2_user_data(123);

    {
        can_fill_part2 => '',
        db => {
            form_data => undef,
            user => undef,
        },
        has_role_in_partner2 => '',
    }

Еще один пример (такой user_id в базе ПИ2 есть:

    my $user_data = get_partner2_user_data(35309619);

    {
        can_fill_part2 => '',
        db => {
            form_data => undef,
            user => {
                accountant_email => undef,
                allowed_constructed_formats => 0,
                business_unit => 0,
                client_id => 1304746,
                country_id => 225,
                create_date => '1970-01-01 03:00:00',
                email => 'ivan@bessarabov.ru',
                has_approved => 1,
                has_tutby_agreement => 0,
                id => 35309619,
                is_tutby => 0,
                lastname => 'Бессарабов',
                login => 'ivanbessarabov',
                midname => 'Михайлович',
                multistate => 1,
                name => 'Иван',
                need_to_email_processing => 0,
                newsletter => 1,
                no_stat_monitoring_emails => 0,
            },
        },
        has_role_in_partner2 => 1,
    }

=cut

sub get_partner2_user_data {
    my ($user_id, $flag) = @_;

    croak "Incorrect user_id" unless $user_id =~ /^\d+\z/;
    croak "flag is deprecated" if $flag;

    my $url      = $ENV{FORM_PARTNER2_SERVER} . '/intapi/form/user.json?lang=en&user_id=' . $user_id . '&db=1&rep=1';
    my $response = HTTP::Tiny->new()->get($url);
    my $result   = check_api_result($url, $response);

    return $result->{data};
}

=head2 rm_user_in_db

=cut

sub rm_user_in_db {
    my ($user_id) = @_;

    croak "Incorrect user_id" unless $user_id =~ /^\d+\z/;

    my $url      = $ENV{FORM_PARTNER2_SERVER} . '/intapi/form/rm_user_in_db.json?lang=en&user_id=' . $user_id;
    my $response = HTTP::Tiny->new()->get($url);
    my $result   = check_api_result($url, $response);

    return 1;
}

=head2 send_email

=cut

sub send_email {
    my (%opts) = @_;

    my $url      = $ENV{FORM_PARTNER2_SERVER} . '/intapi/form/send_email.json';
    my $content  = \%opts;
    my $response = HTTP::Tiny->new()->request(
        'POST', $url,
        {
            headers => {
                'Content-Type'    => 'application/json',
                'Accept-Language' => 'en',
            },
            content => encode_json $content,
        }
    );
    my $result = check_api_result($url, $response, $content) || {};

    if ($result->{data} && $result->{data} eq "ok") {
        return 1;
    } else {
        throw_mojo_exception(
            {
                api_data => {
                    response => $response,
                    url      => $url,
                },
                type    => 'Partner2::send_email',
                message => 'Does not return success',
            }
        );
    }
}

sub _contract_error {
    my ($result, $response) = @_;

    if ($result && $result->{data}{error_token} && (my $err = $result->{data}{error_message})) {
        return {global => $err,};
    } else {
        my $content = $response->{content} // '';
        utf8::decode($content);
        if ($content =~
/Договор на плательщика с ИНН \d+ на выбранный период уже существует/
           )
        {
            # Небольшой хак, пока не будет реализован тикет https://st.yandex-team.ru/BALANCE-28523
            return {inn => 1,};
        } elsif ($content =~ /Договор конфликтует по датам с договорами ID: \d+/) {
            return {contract => 1,};
        } elsif ($content =~ /Bank with BIC \w+ not found in DB/) {
            return {swift => 1,};
        }
    }
}

=head2 create_offer

=cut

sub create_offer : SENSITIVE_DATA_HASH_POINTERS('/adfox_account/password') {
    my (%opts) = @_;

    my $lang     = delete $opts{lang} // croak 'Must specify lang';
    my $url      = $ENV{FORM_PARTNER2_SERVER} . '/intapi/form/create_offer.json';
    my $content  = \%opts;
    my $response = HTTP::Tiny->new()->request(
        'POST', $url,
        {
            headers => {
                'Content-Type'    => 'application/json',
                'Accept-Language' => $lang,
            },
            content => encode_json $content,
        }
    );
    my $result = check_api_result($url, $response, $content, 0);
    if ($result && (my $contract = $result->{data}{contract})) {
        return $contract;
    } else {
        my $err = _contract_error($result, $response);
        if ($err) {
            return (undef, $err);
        } else {
            throw_mojo_exception(
                {
                    api_data => {
                        response => $response,
                        url      => $url,
                    },
                    type    => 'Partner2::create_offer',
                    message => "Can't find out contract",
                }
            );
        }
    }
}

=head2 update_contract

=cut

sub update_contract {
    my (%opts) = @_;

    my $lang     = delete $opts{lang} // croak 'Must specify lang';
    my $url      = $ENV{FORM_PARTNER2_SERVER} . '/intapi/form/update_contract.json';
    my $content  = \%opts;
    my $response = HTTP::Tiny->new()->request(
        'POST', $url,
        {
            headers => {
                'Content-Type'    => 'application/json',
                'Accept-Language' => $lang,
            },
            content => encode_json $content,
        }
    );
    my $result = check_api_result($url, $response, $content, 0);

    if ($result && (my $contract = $result->{data}{contract})) {
        return $contract;
    } else {
        my $err = _contract_error($result, $response);
        if ($err) {
            return (undef, $err);
        } else {
            throw_mojo_exception(
                {
                    api_data => {
                        response => $response,
                        url      => $url,
                    },
                    type    => 'Partner2::update_contract',
                    message => "Can't find out contract",
                }
            );
        }
    }
}

=head2 get_person

=cut

sub get_person {
    my ($client_id, $person_id) = @_;

    croak "Must specify client_id" unless $client_id;
    croak "Must specify person_id" unless $person_id;

    my $url = $ENV{FORM_PARTNER2_SERVER} . "/intapi/form/person.json?lang=en&client_id=$client_id&person_id=$person_id";
    my $response = HTTP::Tiny->new()->get($url);
    my $result = check_api_result($url, $response);

    if (my $person = $result->{data}) {
        return $person;
    } else {
        throw_mojo_exception(
            {
                api_data => {
                    response => $response,
                    url      => $url,
                },
                type    => 'Partner2::get_person',
                message => "Can't find out person for client_id $client_id person_id $person_id",
            }
        );
    }
}

=head2 get_bank_by_bik

L<https://wiki.yandex-team.ru/partner/w/partner2-intapi-form-bank/>

=cut

sub get_bank_by_bik {
    my ($bik) = @_;

    croak "Must specify bik" unless $bik;

    return $Application::APPLICATION->api_balance->check_bank(bik => $bik)
      if ($Application::APPLICATION);

    my $url      = $ENV{FORM_PARTNER2_SERVER} . '/intapi/form/bank.json?lang=en&bik=' . $bik;
    my $response = HTTP::Tiny->new()->get($url);
    my $result   = check_api_result($url, $response);

    return $result->{data} || {};
}

=head2 get_bank_by_swift

L<https://wiki.yandex-team.ru/partner/w/partner2-intapi-form-bank/>

=cut

sub get_bank_by_swift {
    my ($swift) = @_;

    croak "Must specify swift" unless $swift;

    return $Application::APPLICATION->api_balance->check_bank(swift => $swift)
      if ($Application::APPLICATION);

    my $url      = $ENV{FORM_PARTNER2_SERVER} . '/intapi/form/bank.json?lang=en&swift=' . $swift;
    my $response = HTTP::Tiny->new()->get($url);
    my $result   = check_api_result($url, $response);

    return $result->{data} || {};
}

=head2 is_yamoney_account_identificated

L<https://wiki.yandex-team.ru/partner/w/partner2-intapi-form-is-yamoney-account-identificated/>

=cut

sub is_yamoney_account_identificated {
    my (%opts) = @_;

    my @fields = (qw(account firstname lastname middlename passport));

    foreach (@fields) {
        croak "Must specify $_" unless defined $opts{$_};
    }

    return $Application::APPLICATION->api_yamoney->is_account_identificated(map {$_ => $opts{$_}} @fields)
      if ($Application::APPLICATION);

    my $query_args = join('&', map {$_ . '=' . uri_escape_utf8($opts{$_})} @fields);
    my $url = $ENV{FORM_PARTNER2_SERVER} . "/intapi/form/is_yamoney_account_identificated.json?lang=en&$query_args";
    my $response = HTTP::Tiny->new()->get($url);
    my $result = check_api_result($url, $response) || {};

    return $result->{data}{is_identificated};
}

=head2 remove_user_client_association

L<https://wiki.yandex-team.ru/partner/w/partner2-intapi-form-remove-user-client-association/>

=cut

sub remove_user_client_association {
    my (%opts) = @_;

    my $uri = URI->new($ENV{FORM_PARTNER2_SERVER} . '/intapi/form/remove_user_client_association.json');
    $uri->query_form(
        lang             => 'en',
        operator_user_id => $opts{operator_user_id},
        client_id        => $opts{client_id},
        user_id          => $opts{user_id},
    );
    my $url      = $uri->as_string();
    my $response = HTTP::Tiny->new()->get($url);
    my $result   = check_api_result($url, $response) || {};

    if ($result->{data} && $result->{data} eq "ok") {
        return 1;
    } else {
        throw_mojo_exception(
            {
                api_data => {
                    response => $response,
                    url      => $url,
                },
                type    => 'Partner2::remove_user_client_association',
                message => "Does not return success",
            }
        );
    }
}

# TODO: update link

sub get_currency_code {
    my ($currency_code) = @_;
    require Application::Model::Documents;
    return Application::Model::Documents::get_currency_for_offer($Application::APPLICATION, $currency_code, '');
}

=head2 check_pi_adfox_contracts

L<https://wiki.yandex-team.ru/partner/w/partner2-intapi-form-check-pi-adfox-contracts/>

=cut

sub check_pi_adfox_contracts : SENSITIVE_DATA_HASH_KEYS('adfox_password') {
    my (%opts) = @_;

    $opts{from_offer} = 0;
    my $branch_id = delete $opts{branch_id};
    undef $branch_id if (!$branch_id || $branch_id !~ /_ph(?:_|$)/);

    my $url = URI->new($ENV{FORM_PARTNER2_SERVER} . '/intapi/form/check_pi_adfox_contracts.json');
    my $content =
      {map {($_ => ($opts{$_} // croak "Must specify $_"))}
          (qw(user_id adfox_login adfox_password from_offer allow_paid_services))};
    $content->{allow_paid_services} = undef if ($branch_id);    # Физикам нельзя

    my $response = HTTP::Tiny->new()->request(
        'POST', $url,
        {
            headers => {
                'Content-Type'    => 'application/json',
                'Accept-Language' => ($opts{lang} // croak "Must specify lang"),
            },
            content => encode_json $content,
        }
    );
    my $result = check_api_result($url, $response, undef, 0);

    my $data = $result->{data};

    unless ($data->{contracts} || $data->{error_message}) {
        throw_mojo_exception(
            {
                api_data => {
                    response => $response,
                    url      => $url->as_string,
                },
                type    => 'Partner2::check_pi_adfox_contracts',
                message => 'Expected "contracts" or "error_message"',
            }
        );
    }

    $data->{error_token} = $branch_id . '_' . $data->{error_token}
      if ($data->{error_token} && $data->{error_token} eq 'adfox_has_paid' && $branch_id);

    return $result->{data};
}

sub check_bik_and_account {
    my ($bik, $account) = @_;

    croak "Must specify bik" unless $bik;

    return $Application::APPLICATION->api_balance->check_rubank_account(
        bik     => $bik,
        account => $account,
    ) if $Application::APPLICATION;

    my $url =
      $ENV{FORM_PARTNER2_SERVER} . '/intapi/form/rubank_account.json?lang=en&bik=' . $bik . '&account=' . $account;
    my $response = HTTP::Tiny->new()->get($url);
    my $result = check_api_result($url, $response);

    return $result->{data}{found};
}

1;
