package Submit;

use strict;
use warnings FATAL => 'all';
use utf8;
use open qw(:std :utf8);

use Carp;
use Cwd qw(abs_path);
use Path::Tiny;
use Cpanel::JSON::XS;
use Encode;
use JSV::Validator;
use JSON::Pointer;
use Moment;
use Clone qw(clone);

use Branches;
use Countries;
use File::Basename;
use FormConstants qw(
  @ADFOX_PRESET_ASSESSOR
  @ADFOX_PRESET_DEFAULT
  @ADFOX_PRESET_MOBILEMEDIATION
  @ADFOX_PRESET_PAIDPRODUCTS
  $BLOGER_NOTIFICATION_MAIL
  $PARTNER2_CRON_MAIL
  $PARTNERS_MAIL
  );
use Monitoring;
use Partner2;
use Utils;

BEGIN {
    my @field_files = glob dirname(abs_path(__FILE__)) . '/Field/*.pm';
    for my $field (@field_files) {
        require $field;
    }
}

=head2 new

В объекте никак не валидруется user_id — валидацию на это поле нужно делать снаружи.

    my $submit = Submit->new({
        user_id => 607369863,
        country_id => 149,
        branch_id => 'ag_ph',
        fields => {
            email => 'devnull@yandex.ru',
            iban_or_account => {
                iban => 'GB29NWBK60161331926819',
            },
            ...
        },
    });

=cut

sub MODIFY_CODE_ATTRIBUTES {
    my ($package, $sub, @attrs) = @_;

    return Utils::setup_sensitive_data_filters($package, $sub, @attrs);
}

sub new {
    my ($class, $data, $language, $project, $version) = @_;

    croak "Must specify language" unless defined $language;

    my $self = {
        _data     => $data,
        _language => $language,
        project   => $project,
        version   => ($version // 1),
    };
    bless $self, $class;

    return $self;
}

=head2 submit

    my ($is_valid, $errors) = $submit->submit();

=cut

sub submit {
    my ($self) = @_;

    JSV::Validator->load_environments("draft4");
    my $jv = JSV::Validator->new(environment => "draft4");

    my $data   = $self->{'_data'};
    my $result = $jv->validate(
        {
            type       => 'object',
            properties => {
                user_id       => {},
                display_login => {},
                country_id    => {},
                branch_id     => {
                    type      => 'string',
                    minLength => 1,
                },
                fields => {},
            },
            required             => [qw(user_id display_login country_id branch_id fields)],
            additionalProperties => JSON::XS::false,
        },
        $data,
    );

    return (0, $self->_get_global_error('Incorrect data structure')) unless $result;

    my $user_id    = $data->{user_id};
    my $country_id = $data->{country_id};
    my $branch_id  = $data->{branch_id};

    my @attachment_fields;

    my $countries = Countries->new($self->{'project'} ? ('project' => $self->{'project'}) : ());
    return (0, $self->_get_global_error('Incorrect country_id'))
      unless $countries->is_valid_working_country_id($country_id);

    my $branches = Branches->new(
        version => $self->{version},
        $self->{'project'} ? ('project' => $self->{'project'}) : ()
    );
    return (0, $self->_get_global_error('Unknown branch_id')) unless $branches->is_valid_branch_id($branch_id);

    my @country_id_for_branch_id = @{$branches->branch_ids2country_ids()->{$branch_id} // []};
    return (0, $self->_get_global_error('branch_id can\'t be used with country_id'))
      unless grep {$_ eq $country_id} @country_id_for_branch_id;

    my $contract;
    my $part1_data;
    my $fields                = $data->{'fields'};
    my $submitted_fields_list = clone($fields);
    my $is_part2              = $branches->is_branch_part2($branch_id);
    if ($is_part2) {
        my $partner2_user_data = get_partner2_user_data($user_id);
        my $json_data          = $partner2_user_data->{db}{form_data}{data};
        croak qq[Got "form_data.data" : null for user_id $user_id] unless $json_data;
        $part1_data = Cpanel::JSON::XS->new->decode($json_data);
        %$fields    = (%{$part1_data->{fields}}, %$fields);
        $contract   = $part1_data->{contract};
    }

    my %errors;
    my %deps;
    foreach my $field_name ($branches->get_fields($branch_id)) {
        my $p = "Field::$field_name";
        delete $submitted_fields_list->{$field_name};
        $deps{$field_name} = $p->get_dependencies();
    }
    delete $submitted_fields_list->{$_} foreach (qw(branch country));
    if (scalar keys %$submitted_fields_list) {
        $errors{global} //= [];
        push @{$errors{global}},
          {description => gettext('Указаны лишние поля: ', $self->{_language})
              . join(', ', sort keys %$submitted_fields_list),};
        $errors{$_} = [{description => gettext('Это поле лишнее', $self->{_language})}]
          foreach (keys %$submitted_fields_list);
    }

    my $sorted = _topological_sort(\%deps);

    foreach my $field_name (@$sorted) {
        if (exists($fields->{$field_name})) {
            my $f = "Field::$field_name"->new(
                language => $self->{_language},
                project  => $self->{'project'},
                version  => $self->{version},
            );

            my $is_valid = $f->is_valid(
                $fields->{$field_name},
                {
                    branch_id    => $branch_id,
                    user_id      => $user_id,
                    dependencies => {map {$_ => $fields->{$_}} @{$deps{$field_name}}},
                }
            );

            if ($is_valid->{is_valid}) {
                $fields->{$field_name} = $f->normalize($fields->{$field_name});
            } else {
                $errors{$field_name} = [
                    {
                        ($is_valid->{error_token} ? (error_token => $is_valid->{error_token},) : ()),
                        description => $is_valid->{description}
                          // gettext('Неверное значение', $self->{_language}),
                    },
                ];
            }

            if ($f->_get_type() eq 'attachments') {
                push @attachment_fields, $field_name;
            }

        } else {
            $errors{$field_name} =
              [{description => gettext('Поле должно быть заполнено', $self->{_language})}];
        }
    }
    if (   $errors{adfox_account}
        && $errors{adfox_account}[0]{error_token}
        && $errors{adfox_account}[0]{error_token} eq $branch_id . '_adfox_has_paid')
    {
        $errors{global} //= [];
        push @{$errors{global}},
          {
            description => gettext(
'Для связи с указанным пользователем AdFox анкету необходимо заполнять, как юридическое лицо',
                $self->{_language}
            ),
          };
    }
    return (0, \%errors) if %errors;

    my ($client_id, $id_err) = get_client_id($user_id);
    return (
        0,
        $self->_get_global_error(
            sprintf(
                gettext(
'Пользователь с логином %s не может быть зарегистрирован. Пожалуйста, выберите другой логин',
                    $self->{'_language'}
                ),
                $data->{display_login}
            )
        )
    ) if ($id_err);
    my $person_id;

    my $today_date = Moment->now()->get_d();

    my %contract_data;
    if ($branches->working_with_contract($branch_id) || $branches->working_with_balance($branch_id)) {
        my $form = _get_converted_fields(
            converter => $branches->get_converter_form($branch_id),
            fields    => $fields,
        );
        my $nds       = $branches->get_balance_contract_nds($branch_id);
        my $test_mode = $branches->get_balance_contract_testmode($branch_id);
        my $signed    = !$test_mode;
        my $currency  = $branches->get_balance_contract_currency($branch_id) // $form->{currency};
        if (defined $form->{'taxation_type'}) {
            # ОСН - с НДС
            # УСН - без НДС
            # с НДС - то, что в balance_contract_nds
            # без НДС - 0
            $nds = 0 if $form->{'taxation_type'} eq 'УСН';
        }
        %contract_data = (
            lang         => $self->{'_language'},
            login        => $data->{display_login},
            operator_uid => $user_id,
            client_id    => $client_id,

            # Всегда передаем такие значения
            currency     => $currency,
            ctype        => 'PARTNERS',
            firm_id      => $branches->get_balance_contract_firm_id($branch_id),
            payment_type => 1,
            partner_pct  => 43,

            # всегда сегодняшняя дата
            start_dt => $today_date,

            test_mode => ($test_mode ? 1 : 0),
            signed    => ($signed    ? 1 : 0),

            nds => $nds,

            (exists($fields->{adfox_account}) ? (adfox_account => $fields->{adfox_account}) : ()),
            _specific_fields_for_offer_creation($self->{project}),
        );
    }

    if ($branches->working_with_balance($branch_id)) {
        ($person_id, my $err) = create_person(
            client_id    => $client_id,
            operator_uid => $user_id,
            person_id    => -1,
            type         => $branches->get_balance_person_type($branch_id),
            fields       => {
                country_id => $country_id,
                is_partner => 1,
                %{
                    _get_converted_fields(
                        converter => $branches->get_converter_balance($branch_id),
                        fields    => $fields,
                    )
                  },
            },
        );
        if ($err) {
            return (0, _contract_error($err, $fields, $self->{'_language'}));
        }
        $contract_data{person_id} = $person_id;
        if ($is_part2) {
            my $need_sign;
            if ($self->{version} == 2) {
                my $test_mode = $branches->get_balance_contract_testmode($branch_id);
                $need_sign = !$test_mode;
            }
            ($contract, my $err) = update_contract(
                %contract_data,
                lang         => $self->{'_language'},
                operator_uid => $user_id,
                contract_id  => $part1_data->{contract}{ID},
                start_dt     => $today_date,
                (
                    $need_sign
                    ? (
                        test_mode => 0,
                        signed    => 1,
                      )
                    : ()
                ),
            );
            if ($err)
            { # Небольшой хак, пока не будет реализован тикет https://st.yandex-team.ru/BALANCE-28523
                return (0, _contract_error($err, $fields, $self->{'_language'}));
            }
        }
    }

    if ($branches->working_with_contract($branch_id)) {
        ($contract, my $err) = create_offer(%contract_data, service_start_dt => $today_date,);
        if ($err)
        { # Небольшой хак, пока не будет реализован тикет https://st.yandex-team.ru/BALANCE-28523
            return (0, _contract_error($err, $fields, $self->{'_language'}));
        }
    }

    save_data(
        {
            user_id    => $user_id,
            client_id  => $client_id,
            person_id  => $person_id,
            country_id => $country_id,
            branch_id  => $branch_id,
            data       => {
                ($contract          ? (contract => $contract)          : ()),
                ($self->{'project'} ? (project  => $self->{'project'}) : ()),
                country_name_ru   => $countries->get_country_name($country_id, 'ru'),
                language          => $self->{_language},
                fields            => _filter_sensitive_data($fields),
                attachment_fields => \@attachment_fields,
            },
        }
    );

    my %partner = (
        user_id => $user_id,
        ($self->{project} ne 'games' ? (features => ['editable_requisites']) : ()),
        _specific_fields_for_partner_creation($self->{project}, $branch_id, $country_id, $branches),
        %{
            _get_converted_fields(
                converter => $branches->get_converter_partner2($branch_id),
                fields    => $fields,
            )
          },
    );
    create_or_update_partner(%partner);

    if ($self->{'project'} && $self->{'project'} =~ '(video|efir)_blogger') {
        send_email(
            body => join(
                "\n", '<pre>',
                'Зарегистрирован новый пользователь.',
                '-' x 40,
                sprintf('login: %s',     $data->{display_login}),
                sprintf('user_id: %s',   $user_id),
                sprintf('client_id: %s', $client_id),
                sprintf('branch_id: %s', $branch_id),
                (
                    $self->{'project'} eq 'efir_blogger'
                    ? sprintf('video_channel_uri: %s', $fields->{video_channel_uri})
                    : ()
                ),
                '</pre>',
            ),
            content_type => 'text/plain',
            to           => $BLOGER_NOTIFICATION_MAIL,
            bcc          => $PARTNER2_CRON_MAIL,
            subject      => sprintf('%s - Зарегистрирован новый пользователь.',
                $data->{display_login}),
            from => $PARTNERS_MAIL,
        );
    }

    unless ($is_part2) {
        my %templates = (
            ru  => 'CDQDUR54-VZD1',
            en  => '',
            def => 'ru',
        );

        my $template = $templates{$self->{_language}} || $templates{$templates{'def'}};

        send_email(
            bcc       => $PARTNER2_CRON_MAIL,
            user_id   => $user_id,
            template  => $template,
            transport => 'sender',
        );
    }

    send_to_solomon(path => join(' ', 'submit', $branch_id),);

    return (1, {});
}

sub _get_global_error {
    my ($self, $description) = @_;

    return {global => [{description => $description,}],};
}

sub _contract_error {
    my ($err, $fields, $lang) = @_;

    if (my $global = delete $err->{global}) {
        $err->{global} = [{description => $global,},];
    } elsif (my $contract = delete $err->{contract}) {
        my $msg = gettext('У Вас уже есть договор', $lang);
        $err->{global} = [{description => $msg,},];
    } elsif (my $inn = delete $err->{'inn'}) {
        my $no_one = 1;
        my $msg    = gettext(
'Пользователь с таким ИНН уже зарегистрирован. Если это ваш ИНН и вы не можете самостоятельно восстановить свой аккаунт, обратитесь в службу поддержки',
            $lang
        );
        $inn = [{description => $msg,},];
        foreach (qw( inn_ph inn_ur inn_ip )) {
            if (exists($fields->{$_})) {
                $err->{$_} = $inn;
                undef $no_one;
            }
        }
        if ($no_one) {
            $err->{global} = $inn;
        }
    } elsif (my $swift = delete $err->{'swift'}) {
        my $msg = gettext('Указанный SWIFT не найден', $lang);
        $err->{'swift'} = [{description => $msg,},];
    }

    return $err;
}

=begin comment

Преобразует ввод пользователя в данные которые можно передавать в баланс.

    my $fields = _get_converted_fields(
        converter => branches->get_converter_balance($branch_id),
        fields => {
             email => 'devnull@yandex.ru',
             iban_or_account => {
                 iban => 'GB29NWBK60161331926819',
             },
             ...
        },
    ),

=end comment

=cut

sub _get_converted_fields : SENSITIVE_DATA_HASH_POINTERS('/fields/adfox_account/password') {
    my (%opts) = @_;

    my $converter = delete $opts{converter};
    my $fields    = delete $opts{fields};

    my $result = {};

    foreach my $pointer (keys(%{$converter})) {
        my $value = JSON::Pointer->get($fields, $pointer);
        if (defined $value) {
            my $output_field_name;

            if (ref($converter->{$pointer}) eq '') {
                $output_field_name = $converter->{$pointer};
            } elsif (ref($converter->{$pointer}) eq 'HASH') {
                $output_field_name = $converter->{$pointer}->{output_name};
                my $func = $converter->{$pointer}{function};
                if ($func eq 'boolean_to_10') {
                    $value = _boolean_to_10($value);
                } elsif ($func eq 'parse_dict') {
                    $value = _parse_dict(
                        fields => $fields,
                        args   => $converter->{$pointer}->{args},
                    );
                } elsif ($func eq 'space_join') {
                    $value = _space_join(
                        fields => $fields,
                        args   => $converter->{$pointer}->{args},
                    );
                } elsif ($func eq 'space_split') {
                    $value = _space_split(
                        fields => $fields,
                        args   => $converter->{$pointer}->{args},
                    );
                } elsif ($func eq 'fias_house') {
                    $value = _fias_house(
                        fields => $fields,
                        args   => $converter->{$pointer}->{args},
                    );
                } elsif ($func eq 'fias_values') {
                    $value = _fias_values(
                        fields => $fields,
                        args   => $converter->{$pointer}->{args},
                    );
                } elsif ($func eq 'fias_to_text') {
                    $value = _fias_to_text(
                        fields => $fields,
                        args   => $converter->{$pointer}->{args},
                    );
                } elsif ($func eq 'russia_ip_organization') {
                    $value = _russia_ip_organization(
                        fields => $fields,
                        args   => $converter->{$pointer}->{args},
                    );
                } elsif ($func eq 'adfox_link_type') {
                    $value = _adfox_link_type($value);
                } elsif ($func eq 'yamoney_wallet_bank_type') {
                    $value = _yamoney_wallet_bank_type($value);
                } elsif ($func eq 'yandex_money') {
                    $value = _yandex_money(
                        fields => $fields,
                        args   => $converter->{$pointer}->{args},
                    );
                } elsif ($func eq 'reverse_fias_part') {
                    $value = _reverse_fias_part(
                        fields      => $fields,
                        args        => $converter->{$pointer}->{args},
                        output_name => $converter->{$pointer}->{output_name},
                    );
                } elsif ($func eq 'reverse_legaladdress') {
                    $value = _reverse_legaladdress(
                        fields      => $fields,
                        args        => $converter->{$pointer}->{args},
                        output_name => $converter->{$pointer}->{output_name},
                    );
                } elsif ($func eq 'nospace') {
                    $value = _nospace(
                        fields      => $fields,
                        args        => $converter->{$pointer}->{args},
                        output_name => $converter->{$pointer}->{output_name},
                    );
                } elsif ($func eq 'currency') {
                    $value = _currency(
                        fields      => $fields,
                        args        => $converter->{$pointer}->{args},
                        output_name => $converter->{$pointer}->{output_name},
                    );
                } elsif ($func eq 'get_bank_name') {
                    $value = _get_bank_name($fields);
                } else {
                    croak "need to implement '$func'";
                }
            }

            if (ref($value) eq '') {
                $result->{$output_field_name} = $value;
            } elsif (ref($value) eq 'ARRAY') {
                $result->{$output_field_name} //= [];
                push @{$result->{$output_field_name}}, @$value;
            } elsif (ref($value) eq 'HASH') {
                @$result{keys %$value} = values %$value;
            } else {
                croak "Unexpected type of value: " . ref($value);
            }
        }
    }

    return $result;
}

sub _boolean_to_10 {
    my ($value) = @_;

    if (ref($value) eq 'JSON::PP::Boolean') {
        return $value ? 1 : 0;
    } else {
        croak "Expected JSON::PP::Boolean";
    }
}

sub _space_join {
    my (%opts) = @_;

    my @values;
    foreach my $pointer (@{$opts{args}}) {
        my $value = JSON::Pointer->get($opts{fields}, $pointer);
        push @values, $value if defined $value && $value ne '';
    }

    return join ' ', @values;
}

sub _space_split {
    my (%opts) = @_;

    my $result  = {};
    my @args    = @{$opts{args}};
    my $pointer = shift @args;
    my $value   = JSON::Pointer->get($opts{fields}, $pointer);
    my $key;
    foreach my $part (split ' ', $value) {

        if (!@args && $key) {
            $result->{$key} .= ' ' . $part;
            next;
        }

        $key = shift @args;
        $result->{$key} = $part;
    }

    return $result;
}

sub _fias_house {
    my (%opts) = @_;

    my %args = @{$opts{args}};

    my @order = qw(
      house
      structure
      building
      office
      apartment
      );

    # Не используем gettext, так как ФИАС адреса есть только в России
    my %names = (
        house     => 'дом',
        structure => 'корпус',
        building  => 'строение',
        office    => 'офис',
        apartment => 'квартира',
    );

    my @values;

    foreach my $id (@order) {
        if ($args{$id}) {
            my $pointer = $args{$id};
            my $value = JSON::Pointer->get($opts{fields}, $pointer);
            if ($value) {
                $value = $names{$id} . ' ' . $value if $value !~ /\Q$names{$id}\E/;
                push @values, $value;
            }
        }
    }

    return join ', ', @values;
}

sub _fias_values {
    my (%opts) = @_;

    my %args = @{$opts{args}};

    my $guid = JSON::Pointer->get($opts{fields}, $args{guid_pointer});

    my $hierarchy = fias_hierarchy($guid);

    my $street = $hierarchy->[0]->{short_name} . ' ' . $hierarchy->[0]->{formal_name};

    # 4 город
    # 5 внутригородская территория
    # 6 населенный пункт

    my $city = '';

    foreach my $el (reverse @{$hierarchy}) {
        if (grep {$el->{obj_level} eq $_} qw(4 5 6)) {
            $city = $el->{short_name} . ' ' . $el->{formal_name};
            last;
        }
    }

    return {
        city      => $city,
        street    => $street,
        fias_guid => $guid,
    };
}

sub _fias_to_text {
    my (%opts) = @_;

    my %args = @{$opts{args}};

    my @order = qw(
      house
      structure
      building
      office
      apartment
      );

    # Не используем gettext, так как ФИАС адреса есть только в России
    my %names = (
        house     => 'дом',
        structure => 'корпус',
        building  => 'строение',
        office    => 'офис',
        apartment => 'квартира',
    );

    my @values;

    foreach my $id (@order) {
        if ($args{$id}) {
            my $pointer = $args{$id};
            my $value = JSON::Pointer->get($opts{fields}, $pointer);
            if ($value) {
                push @values, $names{$id} . ' ' . $value;
            }
        }
    }

    my $zip_code = JSON::Pointer->get($opts{fields}, $args{zip_code});
    my $guid     = JSON::Pointer->get($opts{fields}, $args{guid});

    my $hierarchy = fias_hierarchy($guid);

    my $fias_address = '';

    foreach my $el (reverse @{$hierarchy}) {
        $fias_address .= $el->{short_name} . ' ' . $el->{formal_name} . ", ";
    }

    my $house = join ', ', @values;

    return {legaladdress => "$fias_address$house, $zip_code",};
}

sub _russia_ip_organization {
    my (%opts) = @_;

    my $name = JSON::Pointer->get($opts{fields}, '/organization_name');
    $name =~ s/^\s+//;
    $name =~ s/\s+$//;
    $name =~ s/^(?:ИП|Индивидуальный Предприниматель)\s+//i;
    my $longname = 'ИП ' . $name;

    return {
        name     => $name,
        longname => $longname,
    };
}

# bank_type: 1 - сбербанк; 2 - другой банк; 3 - Я.Деньги; 9 - Приват
sub _yandex_money {
    my (%opts) = @_;

    my %vals;
    foreach (keys %{$opts{args}}) {
        $vals{$_} = JSON::Pointer->get($opts{fields}, $opts{args}{$_}) // '';
    }

    return {
        'bank_type'      => 3,
        'ben-account'    => join(' ', @vals{qw(lname fname pname)}),
        'yamoney_wallet' => $vals{yandex_money},
    };
}

sub _topological_sort {
    my ($deps) = @_;

    my $visited = {};
    my $stack   = [];

    my $step;

    $step = sub {
        my ($node) = @_;

        $visited->{$node} = 1;

        for my $dep (@{$deps->{$node}}) {
            if (!$visited->{$dep}) {
                $step->($dep);
            }
        }

        push @$stack, $node;
    };

    for my $node (sort keys %$deps) {
        if (!$visited->{$node}) {
            $step->($node);
        }
    }

    return $stack;
}

sub _adfox_link_type : SENSITIVE_DATA_ARRAY_POINTERS('/0/password') {
    my ($account) = @_;

    if ($account) {
        my $presets = [@ADFOX_PRESET_DEFAULT, ($account->{adfox_offer} ? @ADFOX_PRESET_PAIDPRODUCTS : ())];
        if ($account->{has_account}) {
            return {
                adfox_link => {
                    login   => $account->{login},
                    presets => $presets,
                    type    => 'bind',
                },
                adfox_offer => ($account->{adfox_offer} ? 1 : 0),
            };
        } else {
            return {
                adfox_link => {
                    presets => $presets,
                    type    => 'new',
                },
                adfox_offer => ($account->{adfox_offer} ? 1 : 0),
            };
        }
    } else {
        return {adfox_link => {type => 'none',},};
    }
}

sub _specific_fields_for_offer_creation {
    my ($project) = @_;

    my %result;

    # Project specific fields
    if ($project eq 'assessor') {
        %result = (allow_existing_test_mode_contract => 1,);
    }

    return %result;
}

sub _specific_fields_for_partner_creation {
    my ($project, $branch, $country_id, $branches) = @_;

    my $is_oferta = $branches->is_oferta($branch);
    my $is_part2  = $branches->is_branch_part2($branch);

    my %result;

    # Project specific fields
    if ($project eq 'default') {
        if ($is_oferta) {
            %result = (
                adfox_link => {
                    presets => [@ADFOX_PRESET_DEFAULT],
                    type    => 'new',
                },
                has_common_offer     => 1,
                has_mobile_mediation => 1,
                has_rsya             => 1,
                is_efir_blogger      => 0,
                is_games             => 0,
                is_mobile_mediation  => 0,
                is_video_blogger     => 0,
            );
        } elsif (!$is_part2) {    # для part2 уже было проставлено в part1
            %result = (
                adfox_link => {
                    presets => [@ADFOX_PRESET_DEFAULT],
                    type    => 'new',
                },
                has_common_offer     => 0,
                has_mobile_mediation => 0,
                has_rsya             => 1,
                is_efir_blogger      => 0,
                is_games             => 0,
                is_mobile_mediation  => 0,
                is_video_blogger     => 0,
            );
        }
    } elsif ($project eq 'efir_blogger') {
        if (!$is_part2) {    # для part2 уже было проставлено в part1
            %result = (
                has_common_offer     => 0,
                has_mobile_mediation => 0,
                has_rsya             => 1,
                is_efir_blogger      => 1,
                is_games             => 0,
                is_mobile_mediation  => 0,
                is_video_blogger     => 0,
            );
        }
    } elsif ($project eq 'games') {
        if ($is_oferta) {
            %result = (
                adfox_link => {
                    presets => [@ADFOX_PRESET_DEFAULT],
                    type    => 'new',
                },
                has_common_offer     => 0,
                has_mobile_mediation => 1,
                has_rsya             => 1,
                is_efir_blogger      => 0,
                is_games             => 1,
                is_mobile_mediation  => 0,
                is_video_blogger     => 0,
            );
        } elsif (!$is_part2) {    # для russia_ph_part2 уже было проставлено в russia_ph_part1
            %result = (
                adfox_link => {
                    presets => [@ADFOX_PRESET_MOBILEMEDIATION],
                    type    => 'new',
                },
                has_common_offer     => 0,
                has_mobile_mediation => 1,
                has_rsya             => 1,
                is_efir_blogger      => 0,
                is_games             => 1,
                is_mobile_mediation  => 0,
                is_video_blogger     => 0,
            );
        }
    } elsif ($project eq 'mobile_mediation') {
        if ($is_oferta) {
            %result = (
                api_mobile_mediation_data => {presets => [@ADFOX_PRESET_DEFAULT],},
                has_common_offer          => 1,
                has_mobile_mediation      => 1,
                has_rsya                  => 1,
                is_efir_blogger           => 0,
                is_games                  => 0,
                is_mobile_mediation       => 1,
                is_video_blogger          => 0,
            );
        } elsif (!$is_part2) {    # для russia_ph_part2 уже было проставлено в russia_ph_part1
            %result = (
                api_mobile_mediation_data => {presets => [@ADFOX_PRESET_MOBILEMEDIATION],},
                has_common_offer          => 0,
                has_mobile_mediation      => 1,
                has_rsya                  => 0,
                is_efir_blogger           => 0,
                is_games                  => 0,
                is_mobile_mediation       => 1,
                is_video_blogger          => 0,
            );
        }
    } elsif ($project eq 'video_blogger') {
        if (!$is_part2) {    # для russia_ph_part2 уже было проставлено в russia_ph_part1
            %result = (
                has_common_offer     => 0,
                has_mobile_mediation => 0,
                has_rsya             => 1,
                is_efir_blogger      => 0,
                is_games             => 0,
                is_mobile_mediation  => 0,
                is_video_blogger     => 1,
            );
        }
    } elsif ($project eq 'assessor') {
        %result = (
            adfox_link => {
                presets => [@ADFOX_PRESET_ASSESSOR],
                type    => 'new',
            },
            has_common_offer     => 1,
            has_mobile_mediation => 1,
            has_rsya             => 1,
            is_efir_blogger      => 0,
            is_assessor          => 1,
            is_games             => 0,
            is_mobile_mediation  => 0,
            is_video_blogger     => 0,
        );
    }

    # Branch specific fields (for all projects)
    $result{need_to_email_processing} = ($branches->is_branch_part1($branch) || $branch eq 'assessor' ? 0 : 1);
    $result{country_id} = $country_id unless $is_part2;
    if ($branches->working_with_balance($branch)) {
        $result{cooperation_form} = $branches->get_balance_person_type($branch);
    }

    $result{is_form_done} = 1 if $is_part2 || $branch !~ /part[12]$/;

    return %result;
}

sub _filter_sensitive_data : SENSITIVE_DATA_ARRAY_POINTERS('/0/adfox_account/password') {
    my ($fields) = @_;

    $fields = clone($fields);
    if (exists($fields->{adfox_account})) {
        delete $fields->{adfox_account}{password};
    }

    return $fields;
}

sub _parse_dict {
    my (%opts) = @_;
    my ($pointer, $dict, $other, $other_value) = @{$opts{args}};

    my $p          = "Field::$dict";
    my $dictionary = $p->get_dictionary();

    my $value = JSON::Pointer->get($opts{fields}, $pointer);

    my @dict_value = grep {$_->{value} eq $value} @$dictionary;
    return {@dict_value ? ($dict => $dict_value[0]{value}, $other => undef) : ($dict => $other_value, $other => $value)
    };
}

sub _nospace {
    my (%opts) = @_;

    my $result  = {};
    my @args    = @{$opts{args}};
    my $pointer = shift @args;
    my $value   = JSON::Pointer->get($opts{fields}, $pointer);
    $value //= '';
    $value =~ s/\s//g;
    return {$opts{output_name} => $value};
}

sub _currency {
    my (%opts) = @_;

    my $result  = {};
    my @args    = @{$opts{args}};
    my $pointer = shift @args;
    my $value   = JSON::Pointer->get($opts{fields}, $pointer);
    return get_currency_code($value);
}

sub _get_bank_name {
    my ($fields) = @_;
    my $bank_data =
        $fields->{'bik'}   ? get_bank_by_bik($fields->{'bik'})
      : $fields->{'swift'} ? get_bank_by_swift($fields->{'swift'})
      :                      undef;

    return $bank_data->{info}{name} if $bank_data;
}

1;
