package Type::Base;

use strict;
use warnings FATAL => 'all';
use utf8;
use open qw(:std :utf8);

use Carp;
use Email::Valid;
use JSV::Validator;

use FormConstants qw($DEFAULT_PROJECT);
use Utils;

sub MODIFY_CODE_ATTRIBUTES {
    my ($package, $sub, @attrs) = @_;

    return Utils::setup_sensitive_data_filters($package, $sub, @attrs);
}

sub new {
    my ($class, %opts) = @_;

    my $language = delete $opts{language};
    croak "Must specify language" unless defined $language;

    my $version = delete $opts{version};
    my $project = delete $opts{project};

    croak if %opts;

    my $self = {
        _language => $language,
        project   => ($project // $DEFAULT_PROJECT),
        version   => ($version // 1),
    };

    bless $self, $class;

    return $self;
}

sub get_api_description {
    my ($self, $user_data, @other) = @_;

    croak if @other;

    my $hint;
    if ($self->can('get_hint')) {
        $hint = $self->get_hint();
        if ($hint && $self->{version} == 2) {
            $hint =~ s/\.\s*$//;
        }
    }
    my $style = ($self->can('get_style') ? $self->get_style() : undef);

    return {
        id          => $self->_get_id(),
        type        => $self->_get_type(),
        json_schema => $self->_get_json_schema(),
        group_name  => $self->get_group_name(),
        ($self->can('_get_name')         ? (name         => $self->_get_name())           : ()),
        ($hint                           ? (hint         => $hint)                        : ()),
        ($self->can('get_value')         ? (value        => $self->get_value($user_data)) : ()),
        ($style                          ? (style        => $style)                       : ()),
        ($self->can('get_dictionary')    ? (dictionary   => $self->get_dictionary())      : ()),
        (keys %{$self->get_conditions()} ? (dependencies => $self->get_conditions())      : ()),
    };
}

sub get_conditions {
    return {};
}

sub is_valid {
    my ($self, $value, $extra) = @_;

    my $conditions = $self->get_conditions();
    my $iterator;
    $iterator = sub {
        my ($node, $or) = @_;

        my $result = ($or ? 0 : 1);
        foreach my $field (keys %$node) {
            my $field_result;
            if ($field eq 'NOT') {
                $field_result = !$iterator->($node->{$field});
            } elsif ($field eq 'AND') {
                $field_result = $iterator->($node->{$field});
            } elsif ($field eq 'OR') {
                $field_result = $iterator->($node->{$field}, 'OR');
            } else {
                $field_result = scalar grep {
                        !defined($_) && !defined($extra->{dependencies}{$field})
                      || defined($_)
                      && defined($extra->{dependencies}{$field})
                      && $_ eq $extra->{dependencies}{$field}
                } (ref($node->{$field}) eq 'ARRAY' ? @{$node->{$field}} : $node->{$field});
            }
            if ($or) {
                $result ||= $field_result;
                last if $result;
            } else {
                $result &&= $field_result;
                last unless $result;
            }
        }

        return $result;
    };

    my $result = $iterator->($conditions);
    if ($result) {
        JSV::Validator->load_environments("draft4");
        my $v = JSV::Validator->new(environment => "draft4");

        $v->{formats} = {
            %{$v->{formats}},
            email => sub {
                Email::Valid->address($_[0]);
            },
        };

        $result = $v->validate($self->_get_json_schema(), $value,);
    } else {
        $result = !defined($value);
    }

    return {is_valid => !!$result,};
}

sub normalize : SENSITIVE_DATA_FILTER_ALL() {
    $_[1];
}

sub get_dependencies {
    my ($self) = @_;

    my @result;
    my $conditions = $self->get_conditions();
    my $iterator;
    $iterator = sub {
        my ($node) = @_;

        foreach my $field (keys %$node) {
            if ($field eq 'OR' || $field eq 'AND' || $field eq 'NOT') {
                $iterator->($node->{$field});
            } else {
                push @result, $field;
            }
        }
    };
    $iterator->($conditions);

    return \@result;
}

sub _get_id {
    croak 'Need to implement _get_id()';
}

sub _get_type {
    croak 'Need to implement _get_type()';
}

sub _get_json_schema {
    croak 'Need to implement _get_json_schema()';
}

sub get_group_name {
    croak 'Need to implement get_group_name()';
}

1;
