package Type::adfox_account;

use strict;
use warnings FATAL => 'all';
use utf8;
use open qw(:std :utf8);

use base qw(Type::Base);

use Carp;
use JSON::XS;
use Partner2 qw(check_pi_adfox_contracts);

sub is_valid : SENSITIVE_DATA_ARRAY_POINTERS('/1/password') {
    my ($self, $value, $extra) = @_;

    my $result = $self->SUPER::is_valid($value, $extra)->{is_valid};

    if ($result) {
        if ($value->{has_account}) {
            $result = check_pi_adfox_contracts(
                adfox_login         => $value->{login},
                adfox_password      => $value->{password},
                allow_paid_services => $value->{adfox_offer} // '',
                branch_id           => $extra->{branch_id},
                lang                => $self->{_language},
                user_id             => $extra->{user_id},
            );

            if ($result->{contracts}) {
                return {is_valid => 1,};
            } else {
                return {
                    is_valid => 0,
                    (
                        $result->{error_token} && $result->{error_message}
                        ? (
                            description => $result->{error_message},
                            error_token => $result->{error_token},
                          )
                        : ()
                    ),
                };
            }
        } else {
            return {is_valid => 1,};
        }
    } else {
        return {is_valid => 0,};
    }
}

sub _get_type {
    return 'adfox_account';
}

sub _get_json_schema {
    my ($self) = @_;

    # maxLength спрашивал у @traden
    return {
        oneOf => [
            {
                type       => 'object',
                properties => {
                    adfox_offer => {type => 'boolean',},
                    has_account => {
                        type => 'boolean',
                        enum => [JSON::XS::true],
                    },
                    login => {
                        type      => 'string',
                        minLength => 1,
                        maxLength => 32,
                    },
                    password => {
                        type      => 'string',
                        minLength => 1,
                        maxLength => 50,
                    },
                },
                required             => ['has_account', 'login', 'password'],
                additionalProperties => 0,
            },
            {
                type       => 'object',
                properties => {
                    has_account => {
                        type => 'boolean',
                        enum => [JSON::XS::false],
                    },
                },
                required             => ['has_account'],
                additionalProperties => 0,
            },
        ],
    };
}

sub get_value {
    {has_account => JSON::XS::false},;
}

sub get_description {
    return <<EOM
Контрол для ввода логина и пароля аккаунта адфокса.

После валидации по схеме данные передаются ручке `intapi/form/check_pi_adfox_contracts.json`,
где проверяется, можно ли для такого логина создать договор оферты.

Ожидается объект:
    {
        "has_account" : true,
        "login" : "adfox-login",
        "password" : "adfox-password"
    }
    или
    {
        "has_account" : true,
        "login" : "adfox-login",
        "password" : "adfox-password",
        "adfox_offer" : true
    }
    или
    {
        "has_account" : false
    }

Для асинхронной валидации поля есть одноименная ручка.

EOM
}

1;
