package Type::iban_or_account;

use strict;
use warnings FATAL => 'all';
use utf8;
use open qw(:std :utf8);

use base qw(Type::Base);

use Carp;
use Business::IBAN;

sub is_valid {
    my ($self, $value, $extra) = @_;

    my $result = $self->SUPER::is_valid($value, $extra)->{is_valid};

    if ($result) {
        if (exists($value->{iban})) {
            # iban проверяем дополнительно
            my $bi = Business::IBAN->new();
            return {is_valid => !!$bi->valid($value->{iban}),};
        } elsif (exists($value->{account})) {
            # account валидируем только по json схеме
            return {is_valid => 1,};
        } else {
            croak "internal error. this can't happen";
        }

        return {is_valid => 0,};

    } else {
        return {is_valid => 0,};
    }
}

sub _get_type {
    return 'iban_or_account';
}

sub _get_json_schema {
    my ($self) = @_;

    return {
        oneOf => [
            {
                type       => 'object',
                properties => {
                    iban => {
                        minLength => 8,
                        maxLength => 34,
                        pattern   => '^[A-Z][A-Z]\d\d[A-Z0-9]+$',
                        type      => 'string',
                    },
                },
                required => ['iban'],
            },
            {
                type       => 'object',
                properties => {
                    account => {
                        type      => 'string',
                        minLength => 4,
                        maxLength => 30,
                        pattern   => '^[-/A-Z0-9]+$',
                    },
                },
                required => ['account'],
            },
        ],
    };
}

sub get_description {
    return <<EOM
Это достаточно сложный контрол.

С помощью этого контрола пользователь вводин либо номер IBAN либо номер счета.

В качестве значения поля с этим типом backend ожидает получить объект

либо:

    {
        "iban" : "NL06INGB0020090358"
    }

либо:

    {
        "account" : "asdf"
    }

Для того чтобы провалидировать правильность ввода, сначала нужно валидировать по json schema
(которая достаточно сложная), в в том случае если в объекте есть поле iban, то еще нужно
сходить в ручку backend `iban` (либо самостоятельно реалзоивать алгоримт проверки iban).

Номер счета валидируется только по json schema.
EOM
}

1;
