package Utils;

use strict;
use warnings FATAL => 'all';
use utf8;
use open qw(:std :utf8);

use Carp;
use Cpanel::JSON::XS;
use Encode;
use Exporter;
use File::Basename;
use File::Slurp qw(read_file);
use JSON::PP qw();
use JSON::Pointer;
use List::Util qw(min max);
use Mojo::Exception;
use Mojo::URL;
use Path::Tiny;
use Scalar::Util qw(blessed);
use Sentry::Raven;
use XML::Twig;
use Cwd qw/ abs_path /;

use FormConstants qw(%PROJECTS);
use Monitoring;

our @ISA       = qw(Exporter);
our @EXPORT_OK = qw(
  get_data_from_json_file
  gettext
  is_valid_language
  get_valid_languages
  to_pretty_json
  is_yndx_login
  is_valid_project
  send_to_sentry
  can_fill_form
  can_fill_branch
  throw_mojo_exception
  normalize_cyrillic_names
  );
our @EXPORT = @EXPORT_OK;

my %LANGUAGES = map {$_ => 1} qw(ru en);
my %translations;
my $exclusion_rx;

my %sentry_filters;

my %body_filters = (
    '/form/api/0/adfox_account' => sub {
        my ($body) = @_;

        eval {
            my $hash = decode_json $body;
            $hash->{password} = 'FILTERED'
              if exists($hash->{password}) && defined($hash->{password});
            $body = encode_json $hash;
        };

        return $body;
    },
);

sub setup_sensitive_data_filters {
    my ($package, $sub, @attrs) = @_;

    my @unknown_attrs = ();

    foreach my $attr (@attrs) {
        my ($type, $args) = ($attr =~ m/^(\w+)\((.*)\)\s*$/);
        my @filter_args;
        @filter_args = map {$_ =~ s/^['"]?([^'"]+)['"]?$/$1/; $_} split(/\s*,\s*/, $args) if $args;

        my $filter_sub = _generate_filter($type, @filter_args);

        if ($filter_sub) {
            die 'Two sensitive data filters for one sub' if exists($sentry_filters{$sub});
            $sentry_filters{$sub} = $filter_sub;
        } else {
            push(@unknown_attrs, $attr);
        }
    }

    return @unknown_attrs;
}

sub _generate_filter {
    my ($filter_type, @filter_args) = @_;

    no strict 'refs';
    my %filters = (
        SENSITIVE_DATA_HASH_KEYS => sub {
            my (%sub_args_hash) = @_;
            for my $key (@filter_args) {
                $sub_args_hash{$key} = 'FILTERED' if exists($sub_args_hash{$key});
            }
            return %sub_args_hash;
        },
        SENSITIVE_DATA_ARRAY_POSITIONS => sub {
            my (@sub_args) = @_;
            for my $pos (@filter_args) {
                $sub_args[$pos] = 'FILTERED' if $pos < @sub_args;
            }
            return @sub_args;
        },
        SENSITIVE_DATA_HASH_POINTERS => sub {
            my (%sub_args_hash) = @_;
            for my $pointer (@filter_args) {
                %sub_args_hash = %{JSON::Pointer->set(\%sub_args_hash, $pointer, 'FILTERED')}
                  if JSON::Pointer->contains(\%sub_args_hash, $pointer);
            }
            return %sub_args_hash;
        },
        SENSITIVE_DATA_ARRAY_POINTERS => sub {
            my (@sub_args) = @_;
            for my $pointer (@filter_args) {
                @sub_args = @{JSON::Pointer->set(\@sub_args, $pointer, 'FILTERED')}
                  if JSON::Pointer->contains(\@sub_args, $pointer);
            }
            return @sub_args;
        },
        SENSITIVE_DATA_FILTER_WITH_SUB => defined($filter_args[0]) ? \&{$filter_args[0]} : undef,
        SENSITIVE_DATA_FILTER_ALL => sub {
            return ('FILTERED_ALL_ARGS');
        },
    );

    return $filters{$filter_type};
}

sub MODIFY_CODE_ATTRIBUTES {
    my ($package, $sub, @attrs) = @_;

    return setup_sensitive_data_filters($package, $sub, @attrs);
}

foreach my $language (keys %LANGUAGES) {
    $translations{$language} =
      get_data_from_json_file(sprintf(dirname(__FILE__) . '/../translations/%s.json', $language));
}

sub get_data_from_json_file {
    my ($file_name) = @_;

    my $data = decode_json encode_utf8 path($file_name)->slurp_utf8;

    return $data;
}

sub gettext {
    my ($message_id, $language) = @_;

    croak "Must specify message_id" unless defined $message_id;

    croak "Invalid language" unless is_valid_language($language);

    if ($ENV{TAP_VERSION} && !defined $translations{ru}->{$message_id}) {
        my $file_name = dirname(__FILE__) . '/../translations/ru.json';
        my $data      = get_data_from_json_file $file_name;
        $data->{$message_id} = $message_id;
        path($file_name)->spew_utf8(to_pretty_json($data));
    }

    return $translations{$language}->{$message_id} || $translations{ru}->{$message_id} || $message_id;
}

sub is_valid_language {
    my ($language) = @_;

    croak "Must specify language" unless defined $language;

    return exists $LANGUAGES{$language};
}

sub get_valid_languages {
    return sort keys %LANGUAGES;
}

sub is_valid_project {
    my ($project) = @_;

    return defined($project) && exists($PROJECTS{$project});
}

sub to_pretty_json {
    my ($data) = @_;

    my $json_coder = JSON::PP->new->pretty->canonical->indent_length(4);

    my $pretty_json = $json_coder->encode($data);

    return $pretty_json;
}

sub is_yndx_login {
    my ($login) = @_;

    return $login =~ /^yndx-/;
}

sub throw_mojo_exception : SENSITIVE_DATA_FILTER_ALL() {
    my $e = shift;

    my $extra = $e->{extra} // {};
    if ($e->{error_xml}) {
        chomp $e->{error_xml};
        eval {
            my $message = XML::Twig->new()->parse($e->{error_xml})->root->first_child('msg')->trimmed_text;
            (my $type = $message) =~ s/\b((?:0x[\da-f]+)|(?:\d+))\b/'X' x length($1)/gie;
            $e = {
                message => $message,
                type    => $type,
            };
        };
        if ($@) {
            (my $type = $e->{error_xml}) =~ s/\b((?:0x[\da-f]+)|(?:\d+))\b/'X' x length($1)/gie;
            $e = {
                message => $e->{error_xml},
                type    => $type,
            };
        }
    }
    if ($e->{api_data}) {
        my $request = $e->{api_data}{request};
        $extra->{api_request} //= $request if $request;
        my $response = $e->{api_data}{response};
        if ($response) {
            $response->{url} = $response->{url}->as_string
              if blessed($response->{url}) && $response->{url}->isa('URI::http');
            $extra->{api_response_raw} = $response;
            eval {$extra->{api_response} = decode_json $response->{content};};
        }
        my $url = $e->{api_data}{url};
        if ($url) {
            $url = Mojo::URL->new($url);
            my $query = $url->query()->to_string();
            my $path  = $url->clone()->query(undef)->to_string();
            $extra->{api_query} //= $query if $query;
            $extra->{api_url} //= $path;
            $e->{message} //= sprintf('Url %s return status: %s', $path, ($response && $response->{status} // '')),;
        }
        $e->{type} //= (caller(2))[3] . ($response->{subtype} ? ' ' . $response->{subtype} : ''),

          send_to_solomon(
            path => join(' ',
                'error_pi_api',
                ((($url // '') =~ m|^/intapi/form/(.+?)\.json|)[0] // 'unknown'),
                ($e->{api_data}{subtype} ? $e->{api_data}{subtype} : ()),
            ),
          );
    }

    my $ex = Mojo::Exception->new($e->{message})->trace(1);
    $ex->{fingerprint} = $e->{fingerprint};
    $ex->{type}        = $e->{type};
    $ex->{extra}       = $extra;
    die $ex;
}

=head2 send_to_sentry

    my $event_id = send_to_sentry(
        exception => $exception # объект исключения
    );

=cut

sub send_to_sentry {
    my (%opts) = @_;

    return undef unless $ENV{SENTRY_DSN};

    my $servername = `hostname -f`;
    chomp($servername);

    my $release;

    # Строка вида 'registry.yandex.net/partners/form_api:2018-06-15_3'
    if (($ENV{QLOUD_DOCKER_IMAGE} // '') =~ /form_api:(.*)/) {
        $release = $1;
    }

    my $exception = delete $opts{exception} // croak 'need exception';
    my $cntrl     = delete $opts{controller};
    my $user      = delete $opts{user};
    my $message   = $exception->to_string;
    my $type      = $exception->{type};
    unless ($type) {
        my $div = ' at ';
        my @parts = split($div, $message, -1);
        if (scalar @parts > 1) {
            $type = pop @parts;
            $message = join($div, @parts) if ($type);
        }

        send_to_solomon(path => 'error_sysdie',);
    }
    $type //= $message;
    my $extra = $exception->{extra} //= {};
    $extra->{PID} = $$;

    my $sentry = Sentry::Raven->new(
        sentry_dsn  => $ENV{SENTRY_DSN},
        servername  => $servername,
        environment => $ENV{FORM_STAGE},
        ($release ? (release => $release) : ()),
    );

    my $event_id;
    foreach (1 .. 3) {
        $event_id = $sentry->capture_exception(
            $message,
            type   => $type,
            logger => 'form',
            ($exception->{fingerprint} ? (fingerprint => ['FormAPI', $exception->{fingerprint}]) : ()),
            Sentry::Raven->stacktrace_context(_convert_stacktrace($exception)),
            Sentry::Raven->request_context(_convert_request($cntrl->req())),
            Sentry::Raven->user_context(_convert_user($cntrl, $user)),
            _convert_extra($cntrl, $extra),
            %opts,
        );
        last if defined($event_id);
        sleep(1);
    }

    return $event_id;
}

=head2 can_fill_form

В этой сабе находится логика с помощью которой определяем может ли
пользователь заполнять анкету.

    my ($can_fill_form, $reason, $message) = can_fill_form(
        canonical_login => $user->{canonical_login},
        can_fill_part2 => $user_data->{can_fill_part2},
        partner2_db => $user_data->{db},
    );

=cut

sub can_fill_form {
    my (%opts) = @_;

    my $canonical_login = delete $opts{canonical_login};
    croak 'no canonical_login' unless defined $canonical_login;

    croak 'no partner2_db' unless exists $opts{partner2_db};
    my $partner2_db = delete $opts{partner2_db};

    my $can_fill_part2 = delete $opts{can_fill_part2};
    croak 'no can_fill_part2' unless defined $can_fill_part2;

    my $language = delete $opts{language};
    croak 'no language' unless is_valid_language($language);

    croak 'no representative' unless exists $opts{representative};
    my $representative = delete $opts{representative};

    my $version = delete $opts{version};
    croak 'no version' unless defined $version;

    croak 'unknown opts: ' . join(',', keys %opts) if %opts;

    # Практически невозможная ситуация, но теоретически может быть
    croak 'impossible: user' if ($can_fill_part2 && !$partner2_db->{user});

    # В случае если пользователь вошел через социальноую авторизацию и еще не создал логин,
    # то в поле содержится пустая строка. Пользователей без логина мы не берем в систему.
    return (
        0,
        'need_postregistration',
        gettext(
'Вы вошли с помощью аккаунта в социальной сети. Для работы с Рекламной сетью необходимо завести логин и пароль <https://passport.yandex.ru/passport?mode=postregistration&create_login=1>',
            $language
        )
    ) if $canonical_login eq '';

    # Логин который начинается с yndx можно создать только из сети яндекса. Это логины
    # не могут быть партнерами.
    return (
        0,
        'technical_account',
        gettext(
'Вы вошли с помощью технического аккаунта. Пожалуйста, выберите другой логин',
            $language
        )
    ) if is_yndx_login($canonical_login);

    # Не даем заполнять анкету, если в ПИ2 есть этот пользователь в таблице users
    # (если в partner2_db->{user} находится undef это значит что пользователя в базе нет,
    # если же пользоавтель в таблице есть, то тут находится HASHREF с данными из таблицы)
    # (Раньше мы не давали заполнять анкету если у пользователя есть роли в системе,
    # но оказалось что пользователей банят в системе с помощью снятия всех ролей,
    # так что пришлось изменить логику)
    return (-1, 'registered', gettext('Вы уже были зарегистрированы', $language))
      if defined($partner2_db->{user}) && !$can_fill_part2;

    # Не даём заполнять анкету, если пользователь является представителем другого пользователя
    # (или у него есть представитель) в каком-нибудь сервисе (например, Директ),
    # который (этот другой пользователь) уже зарегистрировался в ПИ2.
    # Так как у них один client_id на всех и к нему уже привязан договор ПИ.
    return (
        0,
        'representative',
        gettext(
'Вы являетесь представителем другого аккаунта, который уже был зарегистрирован',
            $language
        )
    ) if $representative;

    return 1;
}

=head2 can_fill_branch

В этой сабе находится логика с помощью которой определяем может ли
пользователь заполнять переданную ветку анкеты.

    my ($can_fill_branch, $reason, $message) = can_fill_branch(
        branch_id => $data->{branch_id},
        can_fill_part2 => $user_data->{can_fill_part2},
        partner2_db => $user_data->{db},
    );

=cut

sub can_fill_branch {
    my (%opts) = @_;

    my $branch_id = delete $opts{branch_id};
    croak 'no branch_id' unless $branch_id;

    my $can_fill_part2 = $opts{can_fill_part2};
    my $partner2_db    = $opts{partner2_db};
    my $language       = $opts{language};

    my ($can_fill_form, $reason, $message) = can_fill_form(%opts);

    unless ($can_fill_form) {
        return ($can_fill_form, $reason, $message);
    } elsif ($can_fill_form < 0) {
        # Общий смысл проверок ниже:
        #   если пользователь уже заполнял эту же ветку,
        #   то возвращаем, что ничего делать не нужно
        #   иначе 400
        my $form_data = $partner2_db->{form_data};
        if ($form_data) {
            my $db_branch_id = $form_data->{branch_id} // '';
            if ($db_branch_id eq $branch_id) {
                return -1;
            } else {
                return (
                    0,
                    'conflict',
                    gettext(
'Вы уже были зарегистрированы используя другую форму сотрудничества',
                        $language
                    )
                );
            }
        } else {
            return (0, $reason, $message);
        }
    }

    if ($can_fill_part2) {
        if ($branch_id =~ /_part1$/) {
            return -1;   # Повторная отправка формы после неудачного сабмита
        } elsif ($branch_id !~ /_part2$/) {
            return (0, 'need_part2', gettext('Вы уже были зарегистрированы', $language));
        }
    }

    # Не даём заполнять вторую часть анкеты, если ещё не заполнили первую
    # (если ПИ2 вернул, что не может, или пользователя ещё нет в базе ПИ2)
    return (
        0,
        'need_part1',
        gettext(
'Чтобы продолжить регистрацию, надо быть зарегистрированным',
            $language
        )
    ) if $branch_id =~ /_part2$/ && (!$can_fill_part2 || !defined($partner2_db->{user}));

    return 1;
}

sub _set_source_code_context {
    my ($sentry_frame) = @_;

    my $source_code_context_range = 10;

    if (-f $sentry_frame->{filename}) {
        my @lines = read_file($sentry_frame->{filename}, {binmode => ':utf8'});
        $sentry_frame->{context_line} = $lines[$sentry_frame->{lineno} - 1];
        $sentry_frame->{pre_context} =
          [@lines[max(1, $sentry_frame->{lineno} - $source_code_context_range) - 1 .. $sentry_frame->{lineno} - 2]];
        $sentry_frame->{post_context} = [
            @lines[
              $sentry_frame->{lineno} .. min(scalar(@lines), $sentry_frame->{lineno} + $source_code_context_range) - 1
            ]
        ];
    }
}

sub _prune_at_depth {    # Взято из ПИ2
    my ($ref, $depth, $current_depth) = @_;
    $current_depth //= 0;

    if (!defined($ref)) {
        return 'undef';
    } elsif (
        (!defined($depth) || $current_depth < $depth) && grep {
            $_ eq ref($ref)
        } (qw(ARRAY HASH))
      )
    {
        if (ref($ref) eq 'ARRAY') {
            my @arr;
            for my $i (0 .. $#$ref) {
                $arr[$i] = _prune_at_depth($ref->[$i], $depth, $current_depth + 1);
            }
            return \@arr;
        } elsif (ref($ref) eq 'HASH') {
            my %hash;
            for my $key (keys(%$ref)) {
                $hash{$key} = _prune_at_depth($ref->{$key}, $depth, $current_depth + 1);
            }
            return \%hash;
        }
    } else {
        $ref //= 'undef';
        return qq[$ref];
    }

    return $ref;
}

sub _convert_stacktrace {
    my ($exception) = @_;

    my $cs = $exception->{frames};
    my @st;
    for my $frame (@$cs) {
        my $sentry_frame = {
            filename => $frame->[1],
            lineno   => $frame->[2],
            module   => $frame->[0],
            function => $frame->[3],
            vars     => {'@_' => _prune_at_depth($frame->[-1]),},
        };
        _set_source_code_context($sentry_frame);
        unshift @st, $sentry_frame;
    }

    return \@st;
}

sub _convert_request {
    my ($request) = @_;

    my %data = (
        env     => \%ENV,
        headers => $request->headers()->to_hash(),
        method  => $request->method,
    );

    # Обрезается ключ для сентри
    if ($data{env}{SENTRY_DSN}) {
        $data{env} = {%{$data{env}}};
        $data{env}{SENTRY_DSN} =~ s/:[^@]@/:\[FILTERED\]@/;
    }

    # Обрезаются хвосты сессионных кук согласно информации из курса о защите приватных данных.
    # Сами сессионные куки оставлены (их средняя часть),
    # так как в них хранятся ID всех пользователей мульти-авторизации,
    # информация о которых может быть полезна при анализе событий
    if ($data{headers} && $data{headers}{Cookie}) {
        $data{headers}{Cookie} =~ s/((?:Session_id|sessionid2)=[^; ]+?\|)[^\|; ]+?; /$1\[FILTERED\]; /g;
    }

    return ($request->url->to_string, %data);
}

sub _convert_user {
    my ($c, $user) = @_;

    return (
        ID         => $user->{user_id},
        email      => ($user->{canonical_login} ? $user->{canonical_login} . '@yandex.ru' : ''),
        id         => $user->{canonical_login},
        ip_address => $c->{tx}->{req}->{content}->{headers}->header('X-Real-IP'),
        username   => $user->{canonical_login},
    );
}

sub _convert_extra {
    my ($c, $extra) = @_;

    my $request = $c->req;
    my $url     = $request->url;
    my $query   = $url->query()->to_string();
    my $culprit = $url->clone()->query(undef)->to_string();
    if ($query) {
        $extra->{query_raw} = $query;
        $extra->{query}     = $request->query_params()->to_hash();
    }
    my $body = $request->body;
    if ($body) {
        if (exists($body_filters{$culprit})) {
            $body = $body_filters{$culprit}->($body);
        }
        $extra->{body_raw} = $body;
        utf8::decode $extra->
          {body_raw};    # Чтобы нормально отображались кириллические слова

        eval {$extra->{body} = decode_json $body;};
        $extra->{body} //= $request->body_params()->to_hash();
    }

    return (
        culprit => $culprit,
        extra   => $extra,
    );
}

sub hack_mojo_exception_trace {
    no warnings 'redefine';
    no strict 'refs';

    *{'Mojo::Exception::trace'} = sub {    # Copy of origin + args
        my ($self, $start) = (shift, shift // 1);
        my @frames;

        package DB;
        use Utils;
        while (my @trace = caller($start++)) {
            my $ref = \&{$trace[3]};
            my @args;
            my $module = $trace[3];
            $module =~ s/^(.+)::[^:]+$/$1/;
            my $path = Utils::module_path($module);
            if (!defined($path) || $path =~ m{\./api/lib/}) {
                @args = @DB::args;
                my $filter = $sentry_filters{\&{$trace[3]}};
                if ($filter) {
                    @args = $filter->(@args);
                }
            }
            push @trace,  \@args;
            push @frames, \@trace;
        }
        return $self->frames(\@frames);
    };
    Mojo::Exception::trace(Mojo::Exception->new);
}

sub _get_exclusion_rx {
    my @exclusion = qw(
      аль-
      эд-
      аш-
      зуль-
      эль-
      -паша
      -бей
      -оглы
      -заде
      -сан
      фон
      дер
      ван
      аф
      д'
      д[аеи]?
      дю
      л[ая]
      дел[аья]
      делла
      дос
      );

    foreach (@exclusion) {
        $_ = '(?:^|\s)' . $_
          if (/^[^'-]/)
          ; # Если шаблон исключения не начинается с дефиса или апострофа,
         # надо добавить поглощающий признак начала слова (начало строки или пробел)

        $_ .= '(?=\s|$)'
          if (/[^-']$/)
          ; # Если шаблон исключения не заканчивается на дефис или апостроф,
         # надо добавить непоглощающий признак конца слова (конец строки или пробел)
    }
    my $exclusion = join '|', @exclusion;

    return qr/$exclusion/i;
}

sub normalize_cyrillic_names {
    my ($string) = @_;

    utf8::decode($string) unless utf8::is_utf8($string);
    $string =~ s/[`'"]+/'/g;
    $string =~ s/(\w+)/ucfirst lc $1/ge;
    $exclusion_rx //= _get_exclusion_rx();
    $string =~ s/($exclusion_rx)/lc $1/ge;

    return $string;
}

# from Module::Path
my $SEPARATOR = '/';

sub module_path {
    my $module = shift;
    my $relpath;
    my $fullpath;

    ($relpath = $module) =~ s/::/$SEPARATOR/g;
    $relpath .= '.pm' unless $relpath =~ m!\.pm$!;

  DIRECTORY:
    foreach my $dir (@INC) {
        next DIRECTORY if not defined($dir);

        # see 'perldoc -f require' on why you might find
        # a reference in @INC
        next DIRECTORY if ref($dir);

        next unless -d $dir && -x $dir;

        # The directory path might have a symlink somewhere in it,
        # so we get an absolute path (ie resolve any symlinks).
        # The previous attempt at this only dealt with the case
        # where the final directory in the path was a symlink,
        # now we're trying to deal with symlinks anywhere in the path.
        my $abs_dir = $dir;
        eval {$abs_dir = Cwd::abs_path($abs_dir);};
        next DIRECTORY if $@ || !defined($abs_dir);

        $fullpath = $abs_dir . $SEPARATOR . $relpath;
        return $fullpath if -f $fullpath;
    }

    return undef;
}

1;
