#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION
Скрипт валидирует design блоков RTB через ручку МСФ

=head1 USAGE
  perl ./bin/PI-15150_design_check_validation.pl --accessor=internal_mobile_app_rtb
  perl ./bin/PI-15150_design_check_validation.pl --accessor=internal_mobile_app_rtb --page_ids=1,2,3
  perl ./bin/PI-15150_design_check_validation.pl --accessor=internal_mobile_app_rtb --wo_deleted --wo_godmode

=head1 OPTIONS
  accessor   - модель (необязательный)
  page_ids   - список id площадок через запятую (необязательный)
  wo_deleted - пропускать архивные блоки (необязательный)
  wo_godmode - пропускать годмод блоки (необязательный)
=cut

use lib::abs qw(
  ../lib
  );

use qbit;
use Utils::ScriptWrapper;

my $chunk_size = 5_000;

my $format_not_found_by_name = {};
my %validation_error_grouped;

sub args {
    my ($opts) = @_;

    return (
        'accessor:s'  => \$opts->{'accessor'},
        'page_ids:s'  => \$opts->{'page_ids'},
        'wo_deleted!' => \$opts->{'wo_deleted'},
        'wo_godmode!' => \$opts->{'wo_godmode'},
    );
}

run(
    sub {
        my ($app, $opts) = @_;

        my @models = _get_rtb_block_models($app);
        if ($opts->{accessor}) {
            @models = grep {$opts->{accessor} eq $_} @models;
        }

        for my $model (@models) {
            my ($i, $chunk) = (0, []);

            $validation_error_grouped{$model} = {};

            my $page_id_field_name     = $app->$model->get_page_id_field_name();
            my $multistate_not_deleted = $app->$model->get_multistates_by_filter('not deleted');

            my $and = _get_filter_and($opts, $page_id_field_name, $multistate_not_deleted);

            do {
                $chunk = $app->partner_db->$model->get_all(
                    filter => [AND => $and,],
                    limit  => $chunk_size,
                    offset => $chunk_size * $i,
                    order_by => [$page_id_field_name, 'id']
                );

                foreach my $block (@$chunk) {
                    $block->{page_id} = $block->{$page_id_field_name};

                    my $block_info = {
                        blockinfo => {
                            model               => $model,
                            $page_id_field_name => $block->{$page_id_field_name},
                            id                  => $block->{id},
                            site_version        => $block->{site_version},
                            is_godmode          => $block->{is_custom_bk_data},
                            is_archived         => (0 < grep {$block->{multistate} eq $_} @$multistate_not_deleted),
                        }
                    };

                    my $design;
                    try {
                        $design = _fix_design($app->$model->get_bk_direct_design($block));
                    }
                    catch {
                        print STDERR logstr('get_bk_direct_design error', {%$block_info, error => $_[0]->message});
                    };

                    if ($design) {
                        if (exists $format_not_found_by_name->{$design->{name}}) {
                            _save_invalid(\%validation_error_grouped, $block_info, $design);
                        } else {
                            my $result;
                            my $error;
                            try {
                                $result = $app->api_format_system->validate(
                                    design => $design,
                                    role   => 'partner',
                                    site   => $block->{site_version},
                                );
                            }
                            catch {
                                $error = $_[0];
                            };

                            if ($error) {
                                print STDERR logstr('api error',
                                    {%$block_info, design => $design, api_error => $error->message});
                            } elsif (!$result->{valid}) {
                                _save_invalid(\%validation_error_grouped, $block_info, $design, $result);
                                print logstr('validation error',
                                    {%$block_info, design => $design, validation_error => $result});
                            }
                        }
                    }
                }

                $i++;

            } while (@$chunk);
        }

        print logstr('validation errors grouped', \%validation_error_grouped);
    }
   );

# fix number values for JSON and then some
sub _fix_design {
    my ($design) = @_;

    for (qw/limit fontSize height width/) {
        $design->{$_} += 0 if (exists $design->{$_} && $design->{$_});
    }

    if (exists $design->{borderType} && (($design->{borderType} // '') eq 'none')) {
        delete $design->{borderRadius};
    }

    return $design;
}

sub _get_rtb_block_models {
    my ($app) = @_;
    return grep {
             $app->$_->can('does')
          && $app->$_->does('Application::Model::Role::Has::RTB')
          && 'mobile_mediation_block' ne $_
    } @{$app->product_manager->get_block_model_accessors()};
}

sub _get_filter_and {
    my ($opts, $page_id_field_name, $multistate_not_deleted) = @_;

    my $and = [[id => '<>' => \100500], [id => '<>' => \0],];

    my $page_ids = [];
    if (defined($opts->{'page_ids'})) {
        $page_ids = [split(/,/, $opts->{'page_ids'})];
    }

    if (@$page_ids) {
        push @{$and}, [$page_id_field_name => 'IN' => \$page_ids];
    }
    if ($opts->{wo_deleted}) {
        push @{$and}, [multistate => 'NOT IN' => \$multistate_not_deleted];
    }
    if ($opts->{wo_godmode}) {
        push @{$and}, [is_custom_bk_data => '=' => \0];
    }

    return $and;
}

sub _hash2string {
    my ($hash) = @_;
    my $s = '';
    for (sort keys %$hash) {
        $s .= ($_ // '') . '->' . ($hash->{$_} // '') . '|';
    }
    return $s;
}

sub _save_invalid {
    my ($ve_hh, $block_info, $design, $validation_error) = @_;

    my $model = $block_info->{blockinfo}->{model};

    # if format ($design->{name}) is invalid, nothing else gets validated,
    # so just inc the counter of blocks with this format
    # otherwise save block's design and validation messages
    if ($validation_error && (!exists $validation_error->{items}->{name})) {
        foreach my $k (keys %{$validation_error->{items}}) {
            my $key = _hash2string({$k => $design->{$k}});
            push @{$ve_hh->{$model}->{$key}},
              {
                %$block_info,
                design           => $design,
                validation_error => $validation_error,
                invalid          => {$k => $design->{$k}}
              };
        }
    } else {
        my $key = _hash2string({name => $design->{name}});
        $format_not_found_by_name->{$design->{name}}++;
        $ve_hh->{$model}->{$key} = [] unless exists $ve_hh->{$model}->{$key};
        $ve_hh->{$model}->{$key}->[0] = {invalid => {name => $design->{name}}};
        $ve_hh->{$model}->{$key}->[1]++;
    }
}
