#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION
Скрипт печатает список пейджей и блоков, заведенных и измененнных
сотрудниками после увольнения

С опцией `--do_action_delete` также пытается остановить и заархивировать пейджи,
добавленные таким образом

С опцией `--login=users.domain_login` обрабатывает только указанного пользователя

=head1 USAGE
    ./bin/PI-16752_oneshot_get_changes_doneby_dismissed.pl --do_action_delete --append_logs

=head1 OPTIONS

=cut

use lib::abs qw(
  ../lib
  );

use qbit;
use Utils::ScriptWrapper;

use List::Util qw(min);

my $CHUNK_SIZE = 50;

my @source_model_list = qw(users site mobile_app internal_mobile_app);

my @page_model_list = qw(internal_context_on_site_campaign internal_search_on_site_campaign
  mobile_app_settings context_on_site_campaign search_on_site_campaign video_an_site outdoor indoor);

sub args {
    my ($opts) = @_;

    return (
        'login:s'           => \$opts->{'login'},
        'do_action_delete!' => \$opts->{'do_action_delete'},
    );
}

run(
    sub {
        my ($app, $opts) = @_;

        my %block_model_list =
          map {$_ => $app->$_->get_page_id_field_name()}
          sort grep {$_ !~ /ssp|mediation/ig} @{$app->product_manager->get_block_model_accessors()};

        my $user2check_list = _get_supposedly_dismissed_users($app, $opts->{login});

        my (%user_from_staff_list, @added_page_list, @added_block_list) = ((), (), ());

        for (my $i = 0; $i <= int(scalar(@$user2check_list) / $CHUNK_SIZE); $i++) {
            my $answer = $app->api_staff->persons(
                'fields' => ['login', 'official.quit_at'],
                'filter' => {
                    'login' => [
                        map {$_->{domain_login}}
                          @$user2check_list[$i *
                          $CHUNK_SIZE .. min($i * $CHUNK_SIZE + $CHUNK_SIZE - 1, $#$user2check_list)]
                    ],
                    'official.is_dismissed' => 'true',
                },
            );
            %user_from_staff_list =
              (%user_from_staff_list, map {$_->{login} => $_->{official}->{quit_at}} @{$answer->{result}});
        }

        my @u_list =
          map {
            {
                user_id     => $_->{id},
                pi_login    => $_->{login},
                staff_login => $_->{domain_login},
                quit_at     => $user_from_staff_list{$_->{domain_login}}
            }
          } @$user2check_list;

        for my $user (@u_list) {
            next unless $user->{quit_at};
            printf("USER id=[%s] login=[%s] quit_at=[%s] staff_login=[%s]\n",
                $user->{user_id}, $user->{pi_login}, $user->{quit_at}, $user->{staff_login});
            my $query;
            for my $table (@source_model_list, @page_model_list, keys %block_model_list) {
                my $elem_page_id_fname = $block_model_list{$table};
                my $action_log_table   = $table . '_action_log';
                my $query_union        = $app->partner_db->query->select(
                    table  => $app->partner_db()->$action_log_table,
                    fields => {
                        'action'           => '',
                        'login'            => \$user->{pi_login},
                        'staff_login'      => \$user->{domain_login},
                        'entity'           => \$table,
                        'dt'               => '',
                        'user_id'          => '',
                        'elem_id'          => '',
                        'elem_page_id'     => $elem_page_id_fname ? 'elem_' . $elem_page_id_fname : \undef,
                        page_id_field_name => \$elem_page_id_fname
                    },
                    filter => [
                        'AND',
                        [['user_id' => '=' => \$user->{user_id}], [{'date' => ['dt']} => '>' => \$user->{quit_at}],]
                    ],
                );
                $query = $query ? $query->union($query_union) : $query_union;
            }

            my @change_list = @{$query->get_all()};
            if (@change_list) {
                for my $change (sort {$a->{entity} cmp $b->{entity} || $a->{dt} cmp $b->{dt}} @change_list) {
                    _print_change($change);

                    if ('add' eq ($change->{action} // '')) {
                        if (!$change->{elem_page_id}) {
                            push @added_page_list, $change;
                        } else {
                            push @added_block_list, $change;
                        }
                    }
                }
            } else {
                printf("   OK - no changes\n");
            }
        }

        printf("added PAGES cnt=[%d]\n", scalar @added_page_list);
        _print_change($_) for @added_page_list;
        printf("added BLOCKS cnt=[%d]\n", scalar @added_block_list);
        _print_change($_) for @added_block_list;

        if ($opts->{do_action_delete}) {
            _stop_pages($app, \@added_page_list);
        }
    }
   );

sub _get_supposedly_dismissed_users {
    my ($app, $login) = @_;

    my @internal_role_ids = map {$_->{is_internal} ? $_->{id} : ()} @{$app->rbac->get_roles()};

    my $domain_login_cond =
      $login
      ? [domain_login => '=' => \$login]
      : [{'IFNULL' => [domain_login => \'']} => '<>' => \''];

    # users with filled domain_login AND no internal roles
    return $app->users->get_all(
        fields => [qw(id domain_login login)],
        filter => $app->partner_db->filter(
            [
                AND => [
                    $domain_login_cond,
                    [
                        id => 'NOT IN' => $app->partner_db->query->select(
                            fields => ['user_id'],
                            filter => [role_id => 'IN' => \\@internal_role_ids],
                            table  => $app->partner_db->user_role,
                        )
                    ],
                ]
            ]
        ),
    );
}

sub _stop_pages {
    my ($app, $pages) = @_;

    for my $page (@$pages) {
        my $model = delete $page->{model};
        try {
            $app->$model->do_action($page, $_) for ('stop', 'delete');
            printf(" [%s][%d] stopped and deleted\n", $model, (values %$page)[0]);
        }
        catch {
            printf("!%s\n", $_[0]);
        };
    }
}

sub _print_change {
    my ($change) = @_;
    printf(
        "   model=[%s] action=[%s] page_id=[%s] id=[%d] dt=[%s]\n",
        $change->{entity},
        $change->{action}       // '',
        $change->{elem_page_id} // '',
        $change->{elem_id}      // '',
        $change->{dt}
    );
}
