#!/usr/bin/perl

use Carp;
use HTTP::Tiny;
use URI;
use XML::Simple;
use Getopt::Long;

use lib::abs qw(
  ../lib
  );
use qbit;
use Application;
use Utils::TSV;

my %LOGIN_PAGES;
my %TUTBY_LOGINS;

my @LEVELS =
  qw(advnet_search_on_site_premium advnet_context_on_site_direct advnet_context_on_site_content advnet_context_on_site_natural advnet_search_on_site_direct internal_advnet_context_on_site_direct internal_advnet_context_on_site_rtb internal_advnet_context_on_site_natural internal_advnet_context_on_site_content internal_advnet_search_on_site_direct mobile_app_rtb internal_mobile_app_rtb video_an_site_instream video_an_site_inpage);

sub compare_stat {
    my (%opts) = @_;

    my $filename = "result_$opts{date}.tsv";
    open(my $fh, '>>', $filename) or die "can't open file '$filename' $!";

    my $old_stat = get_raw_stat(
        date    => $opts{date},
        page_id => $opts{page_id},
        level   => $opts{level},
        login   => $opts{login},
        url     => $opts{old_beta_url},
        token   => $opts{token},
    );

    my $new_stat = get_raw_stat(
        date    => $opts{date},
        page_id => $opts{page_id},
        level   => $opts{level},
        login   => $opts{login},
        url     => $opts{new_beta_url},
        token   => $opts{token},
    );

    my $old_stat_data;
    foreach my $el (@{$old_stat}) {
        my $key =
          sprintf("%s\t%s\t%s\t%s", $el->{dt} // '', $el->{page_id} // '', $el->{block_id} // '', $el->{tag_id} // '');
        $old_stat_data->{$key} = {
            page_id  => ($el->{page_id}  // '') . '',
            block_id => ($el->{block_id} // '') . '',
            date     => ($el->{dt}       // '') . '',
            tag_id   => ($el->{tag_id}   // '') . '',
            (map {$_ => ($el->{$_} // '') . ''} grep {/shows$/} keys %$el),
            (map {$_ => ($el->{$_} // '') . ''} grep {/clicks$/} keys %$el),
            (map {$_ => ($el->{$_} // '') . ''} grep {/hits$/} keys %$el),
            partner_wo_nds => sprintf('%.6f', ($el->{partner_wo_nds} // 0)) . '',
            all_wo_nds     => sprintf('%.6f', ($el->{all_wo_nds}     // 0)) . '',
        };
    }

    my $new_stat_data;
    foreach my $el (@{$new_stat}) {
        my $key =
          sprintf("%s\t%s\t%s\t%s", $el->{dt} // '', $el->{page_id} // '', $el->{block_id} // '', $el->{tag_id} // '');
        $new_stat_data->{$key} = {
            page_id  => ($el->{page_id}  // '') . '',
            block_id => ($el->{block_id} // '') . '',
            date     => ($el->{dt}       // '') . '',
            tag_id   => ($el->{tag_id}   // '') . '',
            (map {$_ => ($el->{$_} // '') . ''} grep {/shows$/} keys %$el),
            (map {$_ => ($el->{$_} // '') . ''} grep {/clicks$/} keys %$el),
            (map {$_ => ($el->{$_} // '') . ''} grep {/hits$/} keys %$el),
            partner_wo_nds => sprintf('%.6f', ($el->{partner_wo_nds} // 0)) . '',
            all_wo_nds     => sprintf('%.6f', ($el->{all_wo_nds}     // 0)) . '',
        };
    }

    my $new_total_sum = 0;
    my $old_total_sum = 0;
    foreach my $key (@{array_uniq([keys %$new_stat_data], [keys %$old_stat_data])}) {
        my $new_sum = sprintf("%.6f", ($new_stat_data->{$key}{partner_wo_nds} // 0) / 10000000000);
        my $old_sum = sprintf("%.6f", ($old_stat_data->{$key}{partner_wo_nds} // 0) / 10000000000);
        my $login = $LOGIN_PAGES{$new_stat_data->{$key}{page_id} // $old_stat_data->{$key}{page_id}} // '';
        my $tutby = $TUTBY_LOGINS{$login} // 0;
        print $fh sprintf "$opts{level}\t%s\t%s\t%s\t%s\t%s\t%s\n", $key, $login, $tutby, $new_sum, $old_sum,
          ($new_sum - $old_sum) * 100 / ($old_sum != 0 ? $old_sum : ($new_sum != 0 ? $new_sum : 1));
    }
    close $fh;
    return 1;
}

sub get_raw_stat {
    my (%opts) = @_;

    my $uri = URI->new($opts{url});
    $uri->path('/api/statistics/get_raw.tsv');
    $uri->query_form(
        lang  => 'ru',
        from  => $opts{date},
        to    => $opts{date},
        level => $opts{level},
        (exists $opts{page_id} && defined $opts{page_id} ? (page_id => $opts{page_id}) : ()),
    );

    my $url = $uri->as_string();

    my $response;
    my $cnt = 0;
    while (($response->{status} // '') ne '200' && $cnt < 2000) {
        $response =
          HTTP::Tiny->new(timeout => 600)->request('GET', $url, {headers => {'Authorization' => $opts{token},},});
        $cnt++;
    }

    if ($response->{status} eq '200') {
        $response->{content} =~ s/^#//;
        $response->{content} =~ s/#END\s*\z//;

        my $parsed_tsv = parse_tsv($response->{content});

        return [] if @{$parsed_tsv} == 0;

        my %ids = map {$_ => 1} qw(
          dt
          page_id
          block_id
          tag_id
          );

        my @all_fields = keys(%{$parsed_tsv->[0]});
        my @id_fields = grep {$ids{$_}} @all_fields;

        my @result;

        my $tmp;

        foreach my $el (@{$parsed_tsv}) {
            my $key = join($;, map {$el->{$_}} @id_fields);
            push @{$tmp->{$key}}, $el;
        }

        my $result = [];

        foreach my $tmp_key (keys(%$tmp)) {
            my $row = {};
            foreach my $el (@{$tmp->{$tmp_key}}) {
                foreach my $key (keys %$el) {
                    if ($ids{$key}) {
                        $row->{$key} = $el->{$key};
                    } else {
                        $row->{$key} += $el->{$key};
                    }
                }
            }
            push $result, $row;
        }

        foreach my $el (@$result) {
            foreach my $key (keys(%$el)) {
                if (grep {$key eq $_} qw(partner_w_nds partner_wo_nds)) {
                    $el->{$key} = sprintf('%0.2f', $el->{$key});
                }
            }
        }

        my $sorted = [
            sort {
                     $a->{dt} cmp $b->{dt}
                  || $a->{page_id} <=> $b->{page_id}
                  || $a->{block_id} <=> $b->{block_id}
                  || $a->{tag_id} <=> $b->{tag_id}
              } @$result
        ];

        return $sorted;
    } else {
        croak sprintf('Got %s status code from %s', $response->{status}, $url,);
    }
}

sub get_login {
    my ($app, $page_id) = @_;

    my $login = $app->partner_db->all_pages->get_all(filter => {page_id => $page_id,},)->[0]->{login};

    if (!$login) {
        croak sprintf('Can\'t find login for Page ID %s', $page_id,);
    }

    return $login;
}

sub get_opts {
    my $page_id;
    my $login;
    my $date;
    my $level;
    my $old_beta_url;
    my $new_beta_url;
    my $token;
    my $help;

    GetOptions(
        'page_id=i'      => \$page_id,
        'login=s'        => \$login,
        'date=s'         => \$date,
        'level=s'        => \$level,
        'old_beta_url=s' => \$old_beta_url,
        'new_beta_url=s' => \$new_beta_url,
        'token=s'        => \$token,
        'help|?|h'       => \$help,
    );

    my $errors = [];

    unless (defined($date)) {
        push @$errors, 'Must specify --date';
    }

    unless (defined($level)) {
        push @$errors, 'Must specify --level';
    } else {
        push @$errors, "Incorrect level '$level'" unless $level eq 'all' || grep {$_ eq $level} @LEVELS;
    }

    unless (defined($old_beta_url)) {
        push @$errors, 'Must specify --old_beta_url';
    }

    unless (defined($new_beta_url)) {
        push @$errors, 'Must specify --new_beta_url';
    }

    unless (defined($token)) {
        push @$errors, 'Must specify --token';
    }

    if (@$errors) {
        print join("\n", @$errors), "\n\n";
        show_help();
        exit(1);
    }

    ldump $level;
    return ($page_id, $login, $date, $level, $old_beta_url, $new_beta_url, $token);
}

sub fill_page_logins {
    my ($app) = @_;
    my $pages = $app->all_pages->get_all(fields => [qw(page_id login)]);
    %LOGIN_PAGES = map {$_->{page_id} => $_->{login}} @$pages;
    my $users = $app->users->get_all(fields => [qw(login is_tutby)]);
    %TUTBY_LOGINS = map {$_->{login} => $_->{is_tutby}} @$users;
}

# main
sub main {
    my $app = Application->new();
    $app->pre_run();

    $app->set_cur_user({id => 0});

    no strict 'refs';
    no warnings 'redefine';
    local *{'QBit::Application::check_rights'} = sub {TRUE};

    my ($page_id, $login, $date, $level, $old_beta_url, $new_beta_url, $token) = get_opts();

    fill_page_logins($app);

    @LEVELS = ($level) unless $level eq 'all';

    foreach my $level_id (@LEVELS) {
        compare_stat(
            page_id      => $page_id,
            date         => $date,
            login        => $login,
            level        => $level_id,
            old_beta_url => $old_beta_url,
            new_beta_url => $new_beta_url,
            token        => $token,
        );
    }
    $app->post_run();
}
main();
__END__
