#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

  Скрипт делает запросы в jsonapi perl и java и сравнивает ответы

=head1 USAGE

  perl ./bin/PI-22689_api_diff.pl --accessor=context_on_site_rtb --login=yasaint --token=<TOKEN> --beta_port=8031

=head1 OPTIONS

  accessor      - аксессор модели
  login         - логин пользователя под которым запрашиваются данные
  token         - токен пользователя по которым делаются запросы (нужно что бы у него была возможность делать fake_login)
  beta_port     - порт беты
  filter        - фильтр к модели в json формате
  fields        - поля которые нужно запросить
  request_limit - ограничение page[size] на запрос
  ignore_req_fields - поля, которые не нужено запрашивать

  ignore_available_fields - не участвующие в сравнении available fields
  ignore_editable_fields - не участвующие в сравнении editable fields
  могут принимать в качестве аргумента 'same_as_editable' и 'same_as_available' соответственно
=cut

use lib::abs qw(
  ../lib
  );

use qbit;
use LWP::UserAgent;
use Utils::ScriptWrapper 'util';
use List::MoreUtils qw(uniq);
use File::Spec::Functions qw(catfile);

my $LIMIT     = 1000;
my $DIFF_PATH = './api_diff';
my $BETA_PORT;
my $LWP;

my $total   = 0;
my $correct = 0;
my $error   = 0;

sub args {
    my ($opts) = @_;

    return (
        'accessor:s'                => \$opts->{'accessor'},
        'login:s'                   => \$opts->{'login'},
        'token:s'                   => \$opts->{'token'},
        'beta_port:i'               => \$BETA_PORT,
        'filter:s'                  => \$opts->{'filter'},
        'fields:s'                  => \$opts->{'fields'},
        'request_limit:s'           => \$opts->{'request_limit'},
        'ignore_req_fields:s'       => \$opts->{'ignore_req_fields'},
        'ignore_available_fields:s' => \$opts->{'ignore_available_fields'},
        'ignore_editable_fields:s'  => \$opts->{'ignore_editable_fields'},
        'route:s'                   => \$opts->{'route'},
        'ignore_add_fields:s'       => \$opts->{'ignore_add_fields'}
    );
}

run(
    sub {
        my ($app, $opts) = @_;

        my $accessor = $opts->{'accessor'} // die 'Expected --accessor';
        my $token    = $opts->{'token'}    // die 'Expected --token';
        my $login    = $opts->{'login'}    // die 'Expected --login';
        die 'Expected --beta_port' unless defined($BETA_PORT);
        my $filter = $opts->{'filter'}      ? from_json($opts->{'filter'}) : undef;
        my $limit  = $opts->{request_limit} ? $opts->{request_limit}       : $LIMIT;
        my $route = $opts->{route} // '';

        `rm $DIFF_PATH -R` if -d $DIFF_PATH;

        my $fields = '';
        if ($opts->{'fields'}) {
            $fields = $opts->{'fields'};
        } else {
            $app->set_cur_user(
                {
                    id => $app->users->get_all(
                        fields => [qw/id/],
                        filter => {login => $login},
                      )->[0]->{id}
                }
            );

            my @available_fields = $app->$accessor->api_available_fields();
            my %ignore_fields = map {$_ => 1} split(',', $opts->{ignore_req_fields} // '');
            $fields = join(',', sort grep {!$ignore_fields{$_}} @available_fields);
        }

        $LWP = get_lwp($login, $token);

        my $user = $app->users->get_all(fields => [qw(id)], filter => {login => $login})->[0]
          // die sprintf('User not found "%s"', $login);
        my $user_id = $user->{'id'};

        $app->set_cur_user({id => $user_id, login => $login});

        restore_check_rights();

        my $offset = 0;
        my $blocks;

        if (defined($opts->{ignore_editable_fields}) && $opts->{ignore_editable_fields} eq 'same_as_available') {
            print "Using ignore_editable_fields same as available \n";
            $opts->{ignore_editable_fields} = $opts->{ignore_available_fields};
        }
        if (defined($opts->{ignore_available_fields}) && $opts->{ignore_available_fields} eq 'same_as_editable') {
            print "Using ignore_available_fields same as editable \n";
            $opts->{ignore_available_fields} = $opts->{ignore_editable_fields};
        }
        my @ignore_editable_fields  = split(',', $opts->{ignore_editable_fields}  // '');
        my @ignore_available_fields = split(',', $opts->{ignore_available_fields} // '');

        while (TRUE) {
            my @required_attrs = uniq map {@$_} values %{$app->$accessor->get_fields_depends()->{required}};
            $blocks = $app->$accessor->get_all(
                fields => [qw(public_id), @required_attrs],
                limit  => $limit,
                offset => $offset,
                (defined($filter) ? (filter => $filter) : ()),
            );

            last unless @$blocks;

            my @public_ids = map {$_->{'public_id'}} @$blocks;

            if (!$route) {
                diff_simple_get(\@public_ids, $accessor, $fields, $limit, \@ignore_editable_fields,
                    \@ignore_available_fields);
            } elsif ($route eq 'defaults' || $route eq 'add_fields') {
                foreach my $block (@$blocks) {
                    diff_meta_route($block, $fields, $route, $accessor, \@required_attrs,
                        $opts->{ignore_add_fields} // '');
                }
            }

            last if @$blocks < $limit;

            $offset = $offset + @$blocks;
        }

        print "Total: $total\n";
        print "Error: $error\n";
        print "OK:    $correct\n";
    }
   );

sub diff_meta_route {
    my ($block, $fields, $route, $accessor, $required_attrs, $ignore_add_fields) = @_;

    my $public_id = $block->{public_id};

    $fields = undef if $route eq 'add_fields';
    my $perl_answer = get_meta_route_req(get_perl_endpoint(), $accessor, $route, $fields, $required_attrs, $block);
    my $java_answer = get_meta_route_req(get_java_endpoint(), $accessor, $route, $fields, $required_attrs, $block);
    my $expected    = $perl_answer->{data};
    my $got         = $java_answer->{data};
    if ($route eq 'defaults') {
        $expected = $expected->{attributes};
        $got      = $got->{attributes};

    }
    if ($route eq 'add_fields') {
        my %ignore_fields = map {$_ => 1} split(',', $ignore_add_fields);
        $got      = [sort {$a->{id} cmp $b->{id}} grep {!$ignore_fields{$_->{id}}} @$got];
        $expected = [sort {$a->{id} cmp $b->{id}} grep {!$ignore_fields{$_->{id}}} @$expected];
    }

    my $json_got      = to_json($got,      pretty => TRUE);
    my $json_expected = to_json($expected, pretty => TRUE);
    compare($json_got, $json_expected, $public_id, $route);
}

sub diff_simple_get {
    my ($public_ids, $accessor, $fields, $limit, $ignore_editable_fields, $ignore_available_fields) = @_;

    my $perl_answer = simple_get_req($accessor, $public_ids, $fields, $limit, get_perl_endpoint());
    my $java_answer = simple_get_req($accessor, $public_ids, $fields, $limit, get_java_endpoint());

    my %hash = map {$_->{'id'} => $_->{'attributes'}} @{$perl_answer->{'data'}};

    foreach my $entity (@{$java_answer->{'data'}}) {
        my $public_id = $entity->{'id'};
        my $got       = $entity->{'attributes'};

        my $expected = delete($hash{$public_id});

        foreach my $field (@$ignore_editable_fields) {
            delete $got->{editable_fields}->{$field};
            delete $expected->{editable_fields}->{$field};
        }
        foreach my $field (@$ignore_available_fields) {
            delete $got->{available_fields}->{$field};
            delete $expected->{available_fields}->{$field};
        }
        die sprintf('Not found id "%s" in Perl', $public_id) unless defined($expected);

        my $json_got      = to_json(canonic_view($got),      pretty => TRUE);
        my $json_expected = to_json(canonic_view($expected), pretty => TRUE);

        compare($json_got, $json_expected, $public_id, '');
    }

    die sprintf("Not found ids in Java:\n%s", join("\n", keys(%hash))) if keys(%hash) > 0;
}

sub compare {
    my ($json_got, $json_expected, $public_id, $route) = @_;
    $total++;
    if ($json_got eq $json_expected) {
        $correct++;
        print "$public_id - OK. $correct/$total\n";
    } else {
        $error++;
        print "$public_id - ERROR. $error/$total\n";

        my $dir = catfile($DIFF_PATH, $public_id, $route);
        `mkdir -p $dir`;

        writefile("$dir/got",      $json_got);
        writefile("$dir/expected", $json_expected);

        print `diff $dir/got $dir/expected`;
    }
}

sub canonic_view {
    my ($attributes) = @_;

    if (exists($attributes->{'campaign'})) {
        if (exists($attributes->{'campaign'}{'owner'})) {
            $attributes->{'campaign'}{'owner'}{'features'} = [sort @{$attributes->{'campaign'}{'owner'}{'features'}}]
              if exists($attributes->{'campaign'}{'owner'}{'features'});

            if (exists($attributes->{'campaign'}{'owner'}{'roles'})) {
                $attributes->{'campaign'}{'owner'}{'roles'} =
                  [sort {$a->{'name'} cmp $b->{'name'}} @{$attributes->{'campaign'}{'owner'}{'roles'}}];

                foreach my $role (@{$attributes->{'campaign'}{'owner'}{'roles'}}) {
                    foreach my $field (qw(related_page_accessor conflict_role_for_set required_role_for_set)) {
                        if (exists($role->{$field})) {
                            $role->{$field} = [sort @{$role->{$field}}];
                        }
                    }
                }
            }

        }
    }

    if (exists($attributes->{'dsp_blocks'})) {
        $attributes->{'dsp_blocks'} = [sort @{$attributes->{'dsp_blocks'}}];
    }

    if (exists($attributes->{'dsps'})) {
        $attributes->{'dsps'} = [sort @{$attributes->{'dsps'}}];
    }

    return $attributes;
}

sub get_meta_route_req {
    my ($endpoint, $accessor, $route, $fields, $required_attrs, $block) = @_;

    my %attrs = map {$_ => $block->{$_}} @$required_attrs;
    my $uri = sprintf("%s/v1/%s/%s?attributes=%s", $endpoint, $accessor, $route, to_json(\%attrs));
    $uri .= '&fields=' . $fields if $fields;
    my $request = HTTP::Request->new(GET => URI->new($uri));
    return get_answer($request);
}

sub simple_get_req {
    my ($accessor, $public_ids, $fields, $limit, $endpoint) = @_;

    my $uri = URI->new(
        sprintf(
            "%s/v1/%s?fields[%s]=%s&filter=%s&page[size]=%d",
            $endpoint, $accessor, $accessor, $fields, to_json({id => $public_ids}), $limit
        )
    );

    my $request = HTTP::Request->new(GET => $uri);

    return get_answer($request);
}

sub get_answer {
    my ($request) = @_;

    my $response = $LWP->request($request);

    if ($response->is_success) {
        return from_json(fix_utf($response->decoded_content));
    } else {
        utf8::decode($response->{'_content'});
        ldump($response);
        die $response->status_line;
    }
}

sub get_java_endpoint {
    return 'http://127.0.0.1:' . ($BETA_PORT + 50_000);
}

sub get_perl_endpoint {
    return 'http://127.0.0.1:' . ($BETA_PORT + 30_000);
}

sub get_lwp {
    my ($login, $token) = @_;

    return LWP::UserAgent->new(
        default_headers => HTTP::Headers->new(
            Content_Type  => 'application/vnd.api+json',
            Accept        => 'application/vnd.api+json',
            Authorization => "token $token",
            X_REAL_IP     => "2a02:6b8:c1b:229e:0:4627:6207:0",
            Cookie        => "fakelogin=$login",
        ),
    );
}
