#!/usr/bin/perl -w

=encoding utf8

=head1 DESCRIPTION

Cкрипт сборки базового образа
см https://wiki.yandex-team.ru/partner/w/docs/release/base-and-main-docker-images/

=head1 PARAMS

 base_image_version - build version (optional, by default incremented version from ./base_docker_tag)
 ubuntu_version     - Ubuntu version (optional, "precise" by default)

=head1 USAGE

 perl bin/build_base_docker_image.pl
 perl bin/build_base_docker_image.pl --base_image_version=VERSION  --ubuntu_version=precise

=cut

# common modules
use strict;
use warnings FATAL => 'all';
use feature 'say';
use utf8;

use Getopt::Long;
use Pod::Usage;

my $DEFAULT_UBUNTU_VERSION       = 'precise';
my $BASE_IMAGE_VERSION_FILE_TMPL = 'configs/default/%s/base_docker_tag';
my $BASE_DOCKERFILE_PATH_TMPL    = 'configs/default/%s/Dockerfile.base';
my $BASE_IMAGE_TAG_TMPL          = 'registry.yandex.net/partners/perl-backend-base%s';
my $DOCKERFILE_PATH_TMPL         = 'configs/deploy_backend/%s/Dockerfile';

main();

sub main {

    my $args = _get_args();

    _check_docker_version();

    my $base_image_version_file = sprintf($BASE_IMAGE_VERSION_FILE_TMPL, $args->{ubuntu_version});
    my $base_dockerfile_path    = sprintf($BASE_DOCKERFILE_PATH_TMPL,    $args->{ubuntu_version});
    my $base_image_tag          = sprintf($BASE_IMAGE_TAG_TMPL,
        ($args->{ubuntu_version} eq $DEFAULT_UBUNTU_VERSION ? '' : '-' . $args->{ubuntu_version}));
    my $dockerfile_path = sprintf($DOCKERFILE_PATH_TMPL, $args->{ubuntu_version});

    my $base_image_version = $args->{'base_image_version'} // _get_base_version_from_file($base_image_version_file);

    say "Building base image version $base_image_version";
    _build_docker_image_and_push_to_registry($base_image_version, $base_dockerfile_path, $base_image_tag);
    _update_and_commit_base_image_version($args->{ubuntu_version},
        $base_image_version, $base_image_version_file, $base_image_tag, $dockerfile_path);
}

sub _update_and_commit_base_image_version {
    my ($ubuntu_version, $base_image_version, $base_image_version_file, $base_image_tag, $dockerfile_path) = @_;

    say "Update base docker image version in backend Dockerfile";
    my $escaped = quotemeta($base_image_tag);
    my $cmd = sprintf q[sed -i 's/^FROM \(%s\):[0-9]*\.[0-9]*/FROM \1:%s/' %s], $escaped, $base_image_version,
      $dockerfile_path;
    system($cmd);

    say "Commit updated files to arc";
    system("arc add $base_image_version_file $dockerfile_path");
}

sub _build_docker_image_and_push_to_registry {
    my ($base_image_version, $base_dockerfile_path, $base_image_tag) = @_;

    say "Building docker images";
    my $command = "docker build --no-cache --file $base_dockerfile_path --tag $base_image_tag:$base_image_version .";
    my $status  = system($command);
    die "Error building docker base image" unless $status == 0;
    say "Docker base image successfully build";

    say "Pushing docker image to registry";
    $command = "docker push $base_image_tag:$base_image_version";
    $status  = system($command);
    die "Error pushing image to registry" unless $status == 0;
    say "Docker base image successfully pushed to registry";
}

sub _get_base_version_from_file {
    my ($base_image_version_file) = @_;

    my $current_version;
    if ((-e $base_image_version_file) && open(my $fh, '<', $base_image_version_file)) {
        $current_version = <$fh>;
        chomp($current_version);
        $current_version =~ s/(\d+)$/$1+1/e;
        close($fh);
    } else {
        $current_version = "1.0";
    }
    open(my $fn, '>', $base_image_version_file) or die($!);
    print $fn "$current_version\n";
    close($fn);
    return "$current_version";
}

sub _check_docker_version {
    if (qx/docker -v/ =~ /Docker version (\d+)\.\d+\.\d+,/) {
        die "Docker major version is less than 19" unless $1 >= 19;
    }
}

sub _get_args {
    my $args = {ubuntu_version => $DEFAULT_UBUNTU_VERSION};

    my $result = Getopt::Long::GetOptions(
        'base_image_version:s' => \$args->{'base_image_version'},
        'ubuntu_version:s'     => \$args->{ubuntu_version},
        #---
        'help|?|h' => \$args->{'help'},
    );

    if (!$result || $args->{'help'}) {
        pod2usage(-verbose => 2, -noperldoc => 1);
    }
    return $args;
}
