#!/usr/bin/perl -w

=encoding UTF-8

=head1 DESCRIPTION

    The script build java: jsonapi, intapi(пока отключено)

=head1 USAGE

    ./bin/build_java.pl --dir=/home/partner/beta.8066 --branch=trunk --app=hourglass --app=jsonapi [--repo=/arc/arcadia] [--repo-ver=r123123] [--rebuild]

=head1 OPTIONS

    dir      - dir where the java will be built
    branch   - arcadia branch
    app      - application
    repo     - path to existing mounted/checkouted sources
    repo-ver - version of existing sources
    rebuild  - force rebuilding even if version matches

=cut

use strict;
use warnings FATAL => 'all';
use utf8;
use threads;
use Pod::Usage;
use Getopt::Long qw();

use Carp;
use JSON::PP;
use File::Slurp;
use lib::abs;

my $TRUNK              = 'trunk';
my $PI_SECRETS_PATH    = '/etc/pi-secrets.json';
my $TIMEOUT_IN_SECONDS = 30 * 60;

exit main();

sub main {
    my $args = _get_args();

    my $start        = localtime();
    my $start_in_sec = time();
    print "[$start] start java build\n";
    eval {
        build_java(
            $args->{dir},      $args->{branch},  $args->{'apps'}, $args->{'repo_path'},
            $args->{repo_ver}, $args->{rebuild}, $args->{commit}
        );
    };
    my $end        = localtime();
    my $end_in_sec = time();
    my $total_time = get_time_from_duration($start_in_sec, $end_in_sec);

    if ($@) {
        print("ERROR: $@\n");
        print "[$end] end java build FAIL. Started: $start. Total build time $total_time\n";
        return 1;
    } else {
        print "[$end] end java build OK. Started: $start. Total build time $total_time\n";
        return 0;
    }
}

sub _get_args {
    my $args = {};

    my $result = Getopt::Long::GetOptions(
        'dir=s'      => \$args->{'dir'},
        'branch:s'   => \$args->{'branch'},
        'commit:s'   => \$args->{'commit'},
        'app=s@'     => \$args->{'apps'},
        'repo:s'     => \$args->{'repo_path'},
        'repo-ver:s' => \$args->{'repo_ver'},
        'rebuild!'   => \$args->{'rebuild'},
    );

    if (!$result || $args->{'help'}) {
        pod2usage(-verbose => 2, -noperldoc => 1);
    }

    $args->{'dir'} //= lib::abs::path("../");

    $args->{branch} ||= $TRUNK;
    push(@{$args->{'apps'}}, 'jsonapi') unless @{$args->{'apps'} // []};

    return $args;
}

sub run_cmd {
    my ($cmd) = @_;

    my $exit_status = system($cmd);

    unless ($exit_status == 0) {
        die "Command '$cmd' exited with non zero exit status";
    }
}

sub check_arc_token {
    unless (-e "$ENV{HOME}/.arc/token") {
        my $arc_token = get_secret('arc-token');
        run_cmd("mkdir -p ~/.arc ; chmod 777 ~/.arc ; echo -n '$arc_token' > ~/.arc/token ; chmod 444 ~/.arc/token");
    }
}

sub is_trunk {
    my ($build_dir) = @_;
    my $output = `cd $build_dir/arc/arcadia; arc br | grep '*'`;
    chomp($output);
    return $output eq '* trunk';
}

sub arc_mount {
    my ($build_dir, $branch, $commit) = @_;

    my $datestring = localtime();
    print "[$datestring] start arc mount\n";
    run_cmd("cd $build_dir/; mkdir -p arc/arcadia arc/store");
    run_cmd("cd $build_dir/arc; arc --update");
    run_cmd("cd $build_dir/arc; arc mount -m arcadia/ -S store/");
    run_cmd("cd $build_dir/arc/arcadia; arc checkout trunk") unless is_trunk($build_dir);
    run_cmd("cd $build_dir/arc/arcadia; arc pull trunk");

    if ($commit) {
        run_cmd("cd $build_dir/arc/arcadia; arc checkout $commit");
    } elsif ($branch ne $TRUNK) {
        run_cmd("cd $build_dir/arc/arcadia; arc br -D $branch || true");
        run_cmd("cd $build_dir/arc/arcadia; arc fetch $branch");
        run_cmd("cd $build_dir/arc/arcadia; arc checkout $branch");
        run_cmd("cd $build_dir/arc/arcadia; arc pull");
    }

    my $result_commit = `cd $build_dir/arc/arcadia; arc log --oneline -n 1 partner`;
    return $commit ? $commit : ((split / /, $result_commit)[0]);
}

sub is_arc_unmounted {
    my ($build_dir) = @_;
    my $output = `mount | grep $build_dir/arc/arcadia`;
    return $output eq '';
}

sub arc_unmount {
    my ($build_dir, $branch) = @_;
    if (-d "$build_dir/arc") {
        if (not is_arc_unmounted($build_dir)) {
            if ($branch ne $TRUNK) {
                run_cmd("cd $build_dir/arc/arcadia; arc br -D $branch || true");
            }
            run_cmd("arc unmount $build_dir/arc/arcadia/");

            my $repeat = 0;
            while (not is_arc_unmounted($build_dir) and $repeat < 5) {
                print "waiting arc unmount\n";
                $repeat++;
                sleep(1 * $repeat);
            }

            print "arc unmount OK\n";
        }
    }
}

sub build_java_layer {
    my ($build_dir, $base_dir, $src_dir, $src_ver, $rebuild, $ya, $ya_token, $layer) = @_;
    #build_dir=/tmp/java.home.user.beta.xxxx
    #source_dir=/home/user/beta.xxxx

    my $start             = localtime();
    my $destination_dir   = "$base_dir/java/$layer";
    my $version_file_name = "build_rev";
    my $versionFile       = "$destination_dir/$version_file_name";
    if (-f $versionFile && $src_ver && !$rebuild) {
        my $haveVersion = `cat $versionFile`;
        chomp $haveVersion;
        if ($haveVersion eq $src_ver) {
            print "skipping rebuild $layer ver=$src_ver\n";
            return 0;
        } else {
            print "rebuild old=$haveVersion need=$src_ver\n";
        }
    } else {
        print "building $layer version=$src_ver\n";
    }

    my $build_layer_dir = "$build_dir/$layer";
    run_cmd("mkdir -p $build_layer_dir");
    run_cmd("mkdir -p $destination_dir");
    open my $F, '>', $versionFile;
    my $build_log_name = "build.log";
    my $log_file       = "$destination_dir/$build_log_name";
    print "[$start] start package $layer\n" . "logging build to $log_file\n";
    my $package_json_path = "$src_dir/partner/java/apps/$layer/package-$layer.json";
    run_cmd(
"cd $build_layer_dir; rm -rf * || true; YA_TOKEN=\"$ya_token\" $ya package -DFORCE_VCS_INFO_UPDATE=yes --cache-size=30GiB --tools-cache-size=10GiB $package_json_path > $log_file 2>&1"
    );

    run_cmd(
"cd $destination_dir && find . -type f -not -name '$build_log_name' -not -name '$version_file_name' -delete || true"
    );
    run_cmd(
        "cp $build_layer_dir/yandex-partner-java-$layer*.tar.gz $destination_dir/yandex-partner-java-$layer.tar.gz");
    run_cmd("cd $destination_dir; tar -xzf yandex-partner-java-$layer.tar.gz");
    my $end = localtime();
    print "[$end] end package $layer. Started: $start\n";
    print $F $src_ver;
    return 1;
}

sub build_java {
    my ($base_dir, $branch, $apps, $repo_dir, $repo_ver, $rebuild, $commit) = @_;    #source_dir=/home/user/beta.xxxx

    my $src_ver;
    my $src_dir;

    my $tmp_dir = "java$base_dir";
    $tmp_dir =~ tr/\//./;
    my $build_dir = "/tmp/$tmp_dir";    #/tmp/java.home.user.beta.xxxx - temp dir
    print "Build dir: $build_dir\n";

    if (!$repo_dir) {
        check_arc_token();
        arc_unmount($build_dir, $branch);

        run_cmd("mkdir -p $build_dir");
        $src_ver = arc_mount($build_dir, $branch, $commit);
        $src_dir = "$build_dir/arc/arcadia";

    } else {
        $src_dir = $repo_dir;
        $src_ver = $repo_ver;
    }
    my $ya_token = get_secret('ya-token');
    my $ya       = "$src_dir/ya";

    my @threads;
    # PI-21914 Пока нигде не используется
    foreach my $app (@$apps) {
        push(
            @threads,
            threads->create(
                \&build_java_layer, $build_dir, $base_dir, $src_dir, $src_ver, $rebuild, $ya, $ya_token, $app
            )
        );
    }

    my $build_counter = 0;
    eval {
        local $SIG{ALRM} = sub {die "alarm\n"};
        alarm($TIMEOUT_IN_SECONDS);

        foreach (@threads) {
            $build_counter += $_->join() // 0;
            if (my $err = $_->error()) {
                die "Thread error: $err\n";
            }
        }
        alarm(0);
    };

    if ($@) {
        die unless $@ eq "alarm\n";    # propagate unexpected errors
                                       # timed out
        die "Java build timeout";
    }

    arc_unmount($build_dir, $branch) unless $repo_dir;

    return $build_counter == 0;
}

sub get_secret {
    my ($secret_name) = @_;

    croak "File $PI_SECRETS_PATH not found" unless -e $PI_SECRETS_PATH;

    my $secrets = decode_json scalar read_file $PI_SECRETS_PATH;

    return exists($secrets->{$secret_name}) ? $secrets->{$secret_name} : croak "Secret '$secret_name' not found";
}

sub get_time_from_duration {
    my ($start, $end) = @_;
    my $duration_in_sec = $end - $start;
    my $minutes         = int($duration_in_sec / 60);
    my $seconds         = $duration_in_sec % 60;
    return (sprintf("%s minutes %s seconds", $minutes, $seconds));
}
