#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

  Скрипт для замены дизайнов

=head1 USAGE

  ./bin/oneshots/PI-28037_change_modernAdaptive_to_adaptive0418.pl --designes=adaptive,300x300

=head1 OPTIONS

  designes  - список дизайнов

=cut

use lib::abs qw(
  ../lib
  ../local/lib/perl5
  );

use Clone;
use Pod::Usage;

use qbit;
use Utils::DB;
use Utils::PublicID;
use Utils::ScriptWrapper;

my $app;
my $opts;
my $before;

# patch format
# [
#   {
#     condition => {    # Проверяются через AND
#       field1 => value,
#       field2 => [ value1, value2, ... ],    # Проверяются через OR
#     },
#     change => {
#       field => new_value,
#     },
#   }
# ]

sub args {
    my ($opts) = @_;
    return (
        'patch=s@'           => \$opts->{patch},
        'page_filter=s'      => \$opts->{page_filter},
        'block_filter=s'     => \$opts->{block_filter},
        'designes=s@'        => \$opts->{designes},
        'god_mode_only!'     => \$opts->{god_mode_only},
        'designes_only!'     => \$opts->{designes_only},
        'page_ids=s@'        => \$opts->{page_ids},
        'block_ids=i@'       => \$opts->{block_ids},
        'limit=i'            => \$opts->{limit},
        'limit_per_design=i' => \$opts->{limit_per_design},
        'percent=f'          => \$opts->{percent},
        'resend=s'           => \$opts->{resend},
        'rollback=s'         => \$opts->{rollback},
    );
}

sub prepare_args {
    my ($opts) = @_;

    foreach (qw(block_filter page_filter)) {
        $opts->{$_} = from_json($opts->{$_}) if $opts->{$_};
    }

    if ($opts->{patch}) {
        $opts->{patch} = [$opts->{patch}] if ref($opts->{patch}) ne 'ARRAY';
        my @patches;
        foreach my $patch_string (@{$opts->{patch}}) {
            my $patches = from_json $patch_string;
            $patches = [$patches] if ref($patches) ne 'ARRAY';
            foreach my $patch (@$patches) {
                die 'Each patch should be HASH' if ref($patch) ne 'HASH';
                foreach my $field (qw(condition change)) {
                    die 'Incorrect patch format' if ref($patch->{$field}) ne 'HASH';
                }
            }
            push @patches, @$patches;
        }
        $opts->{patch} = \@patches;
    } else {
        die 'Patch option is missed';
    }

    if ($opts->{designes}) {
        my @raw_designes = split /\s*,\s*/, join ',', @{$opts->{designes}};
        my %designes;
        @designes{@raw_designes} = (TRUE) x scalar @raw_designes;
        $opts->{designes} = \%designes;
    }

    if ($opts->{page_ids}) {
        $opts->{page_ids} = [split /\s*,\s*/, join ',', @{$opts->{page_ids}}];
    }

    if ($opts->{block_ids}) {
        $opts->{block_ids} = [split /\s*,\s*/, join ',', @{$opts->{block_ids}}];
        $opts->{block_ids_design} = [
            map {
                my (undef, $page_id, $block_id) = split_block_public_id($_);
                +{page_id => $page_id, block_id => $block_id,}
              } @{$opts->{block_ids}}
        ];
        $opts->{block_ids_godmode} = [
            map {
                my (undef, $page_id, $block_id) = split_block_public_id($_);
                +{campaign_id => $page_id, id => $block_id,}
              } @{$opts->{block_ids}}
        ];
    }

    foreach (qw(resend rollback)) {
        die sprintf('File "%s" does not exist', $opts->{$_}) if $opts->{$_} && !-f $opts->{$_};
    }
}

run(
    sub {
        ($app, $opts) = @_;

        my $pages_to_resend;

        if ($opts->{resend}) {
            my $f;
            if (open $f, '<', $opts->{resend}) {
                while (defined(my $s = <$f>)) {
                    if ($s =~ /PAGES TO RESEND\s+(.+)$/) {
                        $pages_to_resend = from_json $1;
                        last;
                    }
                }
                close $f;
            } else {
                die sprintf('Error while open file "%s": %s', $opts->{resend}, $!);
            }
        } else {
            if ($opts->{rollback}) {
                if (open my $f, '<', $opts->{rollback}) {
                    while (defined(my $s = <$f>)) {
                        if ($s =~ /(?:DESIGN|BLOCK) BEFORE\s+(?!ROLLBACK)(.+?)\s+(.+)$/) {
                            $before->{$1} = from_json $2;
                        }
                    }
                    close $f;
                } else {
                    die sprintf('Error while open file "%s": %s', $opts->{resend}, $!);
                }
            }

            my %pages;
            unless ($opts->{god_mode_only}) {
                my $chunk_size = 10000;
                my $last_id    = 0;
                while (TRUE) {
                    last if $opts->{limit} && $opts->{stat}{total} && $opts->{stat}{total} >= $opts->{limit};
                    my $query = $app->partner_db->query->select(
                        table  => $app->design_templates->partner_db_table(),
                        fields => {
                            id        => '',
                            page_id   => '',
                            block_id  => '',
                            opts      => '',
                            caption   => '',
                            design_id => 'id',
                        },
                        filter => [
                            AND => [
                                ($opts->{page_ids} ? [page_id => 'IN' => \$opts->{page_ids}] : ()),
                                (
                                    $opts->{block_ids_design}
                                    ? fields_to_filter(
                                        [qw(page_id block_id)],
                                        $opts->{block_ids_design},
                                        for_db => TRUE,
                                      )
                                    : ()
                                ),
                                [type => '=' => \'tga'],
                                [
                                    {json_unquote => [{json_extract => ['opts', \'$.design_settings.name']}]},
                                    'IN', \[sort keys %{$opts->{designes}}]
                                ],
                                # Удалённые не восстанавливаются, игнорируем
                                [multistate => '=' => \0],
                                # Так быстрее, чем offset
                                [id => '>' => \$last_id],
                            ]
                        ],
                    );
                    add_block_filter($query);
                    add_page_filter($query);
                    my $data = $query->limit($chunk_size)->order_by('design_id')->get_all();
                    last unless scalar @$data;
                    $last_id = $data->[-1]{design_id};
                    my $changed_pages = update_designs($data);
                    @pages{@$changed_pages} = ();
                    last if scalar(@$data) < $chunk_size;
                }
            }

            unless ($opts->{designes_only}) {

                my $chunk_size = 1000;
                my $last_date  = '';
                while (TRUE) {
                    last if $opts->{limit} && $opts->{stat}{total} && $opts->{stat}{total} >= $opts->{limit};
                    my $query = $app->partner_db->query->select(
                        table  => $app->context_on_site_rtb->partner_db_table(),
                        fields => {
                            bk_data           => '',
                            block_create_date => 'create_date',
                            campaign_id       => '',
                            id                => '',
                        },
                        filter => [
                            AND => [
                                ($opts->{page_ids} ? [campaign_id => 'IN' => \$opts->{page_ids}] : ()),
                                (
                                    $opts->{block_ids_godmode} ? fields_to_filter(
                                        [qw(campaign_id id)],
                                        $opts->{block_ids_godmode},
                                        for_db => TRUE,
                                      )
                                    : ()
                                ),
                                (
                                      $opts->{block_filter}
                                    ? $app->context_on_site_rtb->_get_db_filter_from_data($opts->{block_filter})
                                      ->expression()
                                    : ()
                                ),
                                [is_custom_bk_data => '=' => \1],
                                # Так быстрее, чем offset
                                # Возможен небольшой нахлёст,
                                # но такие блоки проигнорируются при повторной обработке.
                                # Добавление более сложных проверок для уникальности
                                # увеличит время выполнения и читабельность.
                                [create_date => '>=' => \$last_date],
                            ]
                        ],
                    );
                    add_page_filter($query);
                    my $data = $query->limit($chunk_size)->order_by('block_create_date')->get_all();
                    last unless scalar @$data;
                    $last_date = $data->[-1]{block_create_date};
                    my $changed_pages = update_god_mode('context_on_site_rtb', $data);
                    @pages{@$changed_pages} = ();
                    last if scalar(@$data) < $chunk_size;
                }
            }

            $pages_to_resend = [
                sort {$a <=> $b}
                  keys %pages
            ];
        }

        print logstr 'PAGES TO RESEND', $pages_to_resend;
        $app->all_pages->mark_pages_for_async_update(page_ids => $pages_to_resend)
          if !$opts->{dry_run} && scalar @$pages_to_resend;
        print logstr 'STAT', $opts->{stat} if $opts->{stat};
    }
   );

sub add_block_filter {
    my ($query) = @_;

    if ($opts->{block_filter} || $opts->{page_filter}) {
        $query->join(
            fields  => [],
            table   => $app->context_on_site_rtb->partner_db_table(),
            join_on => [
                AND => [
                    [campaign_id => '=' => {page_id  => $app->design_templates->partner_db_table()}],
                    [id          => '=' => {block_id => $app->design_templates->partner_db_table()}],
                ]
            ],
            (
                $opts->{block_filter}
                ? (filter => $app->context_on_site_rtb->_get_db_filter_from_data($opts->{block_filter}))
                : ()
            ),
        );
    }

    return $query;
}

sub add_page_filter {
    my ($query) = @_;

    if ($opts->{page_filter}) {
        $query->join(
            fields  => [],
            table   => $app->context_on_site_campaign->partner_db_table(),
            join_on => [page_id => '=' => {campaign_id => $app->context_on_site_rtb->partner_db_table()}],
            filter  => $app->context_on_site_campaign->_get_db_filter_from_data($opts->{page_filter}),
        );
    }

    return $query;
}

sub update_designs {
    my ($data) = @_;
    my @result;

    foreach my $design (@$data) {
        delete $design->{design_id};
        next if defined $opts->{percent} && rand() > $opts->{percent};
        try {
            my $original = clone $design;
            $design->{opts} = from_json $design->{opts};
            my $settings = $design->{opts}{design_settings};
            my $name = $settings->{name} // '';
            if ($opts->{rollback}) {
                if ($before->{$original->{id}}) {
                    print logstr 'DESIGN BEFORE ROLLBACK', $original->{id}, $design;
                    $design = $before->{$original->{id}};
                    print logstr 'DESIGN AFTER ROLLBACK', $design->{id}, $design;
                    push @result, $design;
                    $opts->{stat}{$settings->{name}}++;
                    $opts->{stat}{total}++;
                }
            } else {
                return
                  if $opts->{limit_per_design}
                      && $opts->{stat}{$name}
                      && $opts->{stat}{$name} >= $opts->{limit_per_design};
                return if $opts->{limit} && $opts->{stat}{total} && $opts->{stat}{total} >= $opts->{limit};
                return unless modify_settings($settings);
                $opts->{stat}{$name}++;
                $opts->{stat}{total}++;

                $design->{opts} = to_json($design->{opts}, canonical => TRUE,);
                push @result, $design;
                print logstr 'DESIGN BEFORE', $original->{id}, $original;
                print logstr 'DESIGN AFTER',  $design->{id},   $design;
            }
        }
        catch {
            my ($exception) = @_;

            print logstr 'ERROR', $design->{id}, $exception;
        };
    }
    $app->partner_db->design_templates->add_multi(\@result, duplicate_update => TRUE)
      if !$opts->{dry_run} && scalar @result;

    return [map {$_->{page_id}} @result];
}

sub update_god_mode {
    my ($model, $data) = @_;

    my @result;
    foreach my $block (@$data) {
        delete $block->{block_create_date};
        next if defined $opts->{percent} && rand() > $opts->{percent};
        my $block_key = sprintf("[%s]", join(',', @$block{qw(campaign_id id)}));
        try {
            my $original = clone $block;
            return unless $block->{bk_data};
            $block->{bk_data} = from_json $block->{bk_data};
            my $designes = $block->{bk_data}{RtbDesign};
            my $changed  = FALSE;
            my $old      = FALSE;
            unless (ref $designes) {
                # Где-то ещё остался старый формат (json без скобочек)
                # Надо для однотипости обработки привести его к текущему виду
                # а потом запаковать обратно
                $old      = TRUE;
                $designes = {
                    0 => {
                        design => from_json('{' . $designes . '}'),
                        type   => 'tga',
                    }
                };
            }
            if ($opts->{rollback}) {
                if ($before->{$block_key}) {
                    $block->{bk_data} = from_json $before->{$block_key}->{bk_data};
                    for my $design_key (keys %{$block->{bk_data}{RtbDesign}}) {
                        my $name = $block->{bk_data}{RtbDesign}{$design_key}{design}{name} // '';
                        next unless $opts->{designes}{$name};
                        $opts->{stat}{$name}++;
                        $opts->{stat}{total}++;
                    }
                    $changed = TRUE;
                }
            } else {
                foreach my $design (values %$designes) {
                    my $type = $design->{type} // '';
                    next if $type ne 'tga';
                    my $settings = $design->{design};
                    my $name = $settings->{name} // '';
                    if ($opts->{designes}{$name}) {
                        next
                          if $opts->{limit_per_design}
                              && $opts->{stat}{$name}
                              && $opts->{stat}{$name} >= $opts->{limit_per_design};
                        next if $opts->{limit} && $opts->{stat}{total} && $opts->{stat}{total} >= $opts->{limit};
                        if (modify_settings($settings)) {
                            $changed = TRUE;
                            $opts->{stat}{$name}++;
                            $opts->{stat}{total}++;
                        }
                    }
                }
            }
            if ($changed) {
                $block->{bk_data}{RtbDesign} = substr to_json($designes->{0}{design}, canonical => TRUE,), 1, -1
                  if $old;
                $block->{bk_data} = to_json $block->{bk_data};
                push @result, $block;
                print logstr sprintf('BLOCK BEFORE%s', $opts->{rollback} ? ' ROLLBACK' : ''), $block_key, $original;
                print logstr sprintf('BLOCK AFTER%s',  $opts->{rollback} ? ' ROLLBACK' : ''), $block_key, $block;
            }
        }
        catch {
            my ($exception) = @_;

            print logstr 'ERROR', [@$block{qw(campaign_id id)}], $exception;
        };
    }
    $app->$model->partner_db_table->add_multi(\@result, duplicate_update => TRUE)
      if !$opts->{dry_run} && scalar @result;

    return [map {$_->{campaign_id}} @result];
}

sub modify_settings {
    my ($settings) = @_;

    my $changed = FALSE;
    foreach my $patch (@{$opts->{patch}}) {
        my $ok                     = TRUE;
        my $settings_for_condition = clone $settings;
        foreach my $field (sort keys %{$patch->{condition}}) {
            if ((defined($settings_for_condition->{$field}) && !defined($patch->{condition}{$field}))
                || !(defined($settings_for_condition->{$field}) && defined($patch->{condition}{$field})))
            {
                $ok = FALSE;
                last;
            }
            if (defined($patch->{condition}{$field})) {
                $patch->{condition}{$field} = [$patch->{condition}{$field}]
                  if ref($patch->{condition}{$field}) ne 'ARRAY';
                unless (in_array($settings_for_condition->{$field}, $patch->{condition}{$field})) {
                    $ok = FALSE;
                    last;
                }
            }
        }
        next unless $ok;
        foreach my $field (sort keys %{$patch->{change}}) {
            if (defined $patch->{change}{$field}) {
                $settings->{$field} = $patch->{change}{$field};
            } else {
                delete $settings->{$field};
            }
        }
        $changed = TRUE;
    }

    return $changed;
}
