#!/usr/bin/perl

=encoding UTF-8
=cut

=head1 DESCRIPTION

Скрипт для сравнения данных котоыре возвращает API ПИ1 и ПИ2

 * https://wiki.yandex-team.ru/partner/w/partner1-api-sp-get-stat/
 * https://wiki.yandex-team.ru/partner/w/api-statistics-get-raw/

=cut

# common modules
use feature 'say';
use Carp;
use HTTP::Tiny;
use URI;
use XML::Simple;
use Getopt::Long;
use Term::ANSIColor qw(colored);

# project modules
use lib::abs qw(
  ../lib
  );
use qbit;
use Application;
use Utils::TSV;
use Test::More;
use Test::Differences qw(eq_or_diff);

# global vars

# subs
sub compare_stat {
    my (%opts) = @_;

    my $partner1_stat = get_partner1_stat(
        date                 => $opts{date},
        partner1_campaign_id => $opts{partner1_campaign_id},
        login                => $opts{login},
    );

    my @partner1_stat_uniform;
    foreach my $el (@{$partner1_stat}) {
        push @partner1_stat_uniform,
          {
            page_id       => $el->{CID} . '',
            date          => $el->{Date} . '',
            tag_id        => $el->{StatID} . '',
            shows         => $el->{Shows} . '',
            partner_w_nds => sprintf('%.2f', $el->{Cost}) . '',
          };
    }

    my $partner2_stat = get_partner2_stat(
        date                 => $opts{date},
        page_id              => $opts{page_id},
        level                => $opts{level},
        partner2_url         => $opts{partner2_url},
        partner2_oauth_token => $opts{partner2_oauth_token},
    );

    my $partner2_shows_field = '';
    if ($opts{level} eq 'advnet_context_on_site_rtb') {
        $partner2_shows_field = 'rtb_block_shows';
    } elsif ($opts{level} eq 'advnet_context_on_site_direct') {
        $partner2_shows_field = 'direct_context_shows';
    } else {
        say colored(
            spritnf(
                'Don\'t know about partner2 shows field name on level "%s". Need to fix this script.', $opts{level}
            )
        );
    }

    my @partner2_stat_uniform;
    foreach my $el (@{$partner2_stat}) {
        push @partner2_stat_uniform,
          {
            page_id  => ($el->{page_id}               // '') . '',
            block_id => ($el->{block_id}              // '') . '',
            date     => ($el->{dt}                    // '') . '',
            tag_id   => ($el->{tag_id}                // '') . '',
            shows    => ($el->{$partner2_shows_field} // '') . '',
            partner_w_nds => sprintf('%.2f', ($el->{partner_w_nds} // 0)) . '',
          };
    }

    eq_or_diff(
        to_json(\@partner2_stat_uniform, pretty => 1),
        to_json(\@partner1_stat_uniform, pretty => 1),
        sprintf('Date: %s, Page ID: %s', $opts{date}, $opts{page_id}),
    );

    return 1;
}

=head2 get_partner1_stat

    my $partner1_stat = get_partner1_stat(
        date => '2017-05-10'
        partner1_campaign_id => 477326,
        login => 'rambler-p',
    );

Метод идет в продакшн ручку ПИ1.

https://wiki.yandex-team.ru/partner/w/partner1-api-sp-get-stat

Возвращает ARRAYREF, пример:

    [
        {
            CampID => 477326,
            CID => 159243,
            Cost => 8487.2845,
            Date => 2017-05-10,
            Place => 'rtb',
            Shows => 1015166,
            StatID => 2,
        },
        {
            CampID => 477326,
            CID => 159243,
            Cost => 10.6575,
            Date => 2017-05-10,
            Place => 'rtb',
            Shows => 785,
            StatID => 3,
        },
    ]

=cut

sub get_partner1_stat {
    my (%opts) = @_;

    my $uri = URI->new('https://partner.yandex.ru');
    $uri->path('/audit/partn');
    $uri->query_form(
        cmd    => 'sp_get_stat',
        ID     => $opts{partner1_campaign_id},
        From   => $opts{date},
        To     => $opts{date},
        statid => 1,
    );

    my $url = $uri->as_string();

    my $response = HTTP::Tiny->new()->request('GET', $url, {headers => {'Cookie' => 'fake_login=' . $opts{login},},});

    if ($response->{status} eq '200') {
        my $data = XMLin($response->{content}, ForceArray => ['stat'],);
        return $data->{stat} // [];
    } else {
        croak sprintf('Got %s status code from %s', $response->{status}, $url,);
    }
}

=head2 get_partner2_stat

    my $partner2_stat = get_partner2_stat(
        date => '2017-05-10',
        page_id => 159243,
        level => 'advnet_context_on_site_rtb',
        partner2_url => 'https://dev-partner2.yandex.ru:8468',
        partner2_oauth_token => 'asdf...',
    );

Метод идет в указанную систему ПИ2.

https://wiki.yandex-team.ru/partner/w/api-statistics-get-raw/

Возвращает ARRAYREF, пример:

    [
        {
            block_id => 8,
            dt => 2017-05-10,
            page_id => 159243,
            partner_w_nds => 0.4801,
            partner_wo_nds => 0.4069,
            rtb_all_w_nds => 0.7621,
            rtb_all_wo_nds => 0.6459,
            rtb_bad_win_partner_price_w_nds => 0.0000,
            rtb_bad_win_partner_price_wo_nds => 0.0000,
            rtb_bad_win_price_w_nds => 0.0000,
            rtb_bad_win_price_wo_nds => 0.0000,
            rtb_block_bad_shows => 0,
            rtb_block_bad_shows_own_adv => 0,
            rtb_block_bad_win_hits => 0,
            rtb_block_bad_win_hits_own_adv => 0,
            rtb_block_bad_win_hits_unsold => 0,
            rtb_block_direct_clicks => 1,
            rtb_block_direct_shows => 105,
            rtb_block_hits => 43,
            rtb_block_hits_own_adv => 0,
            rtb_block_hits_unsold => 0,
            rtb_block_shows => 40,
            rtb_block_shows_own_adv => 0,
            tag_id => 0,
        },
        ...
    ]

=cut

sub get_partner2_stat {
    my (%opts) = @_;

    my $uri = URI->new($opts{partner2_url});
    $uri->path('/api/statistics/get_raw.tsv');
    $uri->query_form(
        lang    => 'ru',
        from    => $opts{date},
        to      => $opts{date},
        level   => $opts{level},
        page_id => $opts{page_id},
    );

    my $url = $uri->as_string();

    my $response =
      HTTP::Tiny->new()->request('GET', $url, {headers => {'Authorization' => $opts{partner2_oauth_token},},});

    if ($response->{status} eq '200') {
        $response->{content} =~ s/^#//;
        $response->{content} =~ s/#END\s*\z//;

        my $parsed_tsv = parse_tsv($response->{content});

        return [] if @{$parsed_tsv} == 0;

        # На один и тот же dt/page_id/tag_id/... ручка может возвращать
        # несколько строк. Суммирую все эти строки

        my %ids = map {$_ => 1} qw(
          dt
          page_id
          block_id
          tag_id
          );

        my @all_fields = keys(%{$parsed_tsv->[0]});
        my @id_fields = grep {$ids{$_}} @all_fields;

        my @result;

        my $tmp;

        foreach my $el (@{$parsed_tsv}) {
            my $key = join($;, map {$el->{$_}} @id_fields);
            push @{$tmp->{$key}}, $el;
        }

        my $result = [];

        foreach my $tmp_key (keys(%$tmp)) {
            my $row = {};
            foreach my $el (@{$tmp->{$tmp_key}}) {
                foreach my $key (keys %$el) {
                    if ($ids{$key}) {
                        $row->{$key} = $el->{$key};
                    } else {
                        $row->{$key} += $el->{$key};
                    }
                }
            }
            push $result, $row;
        }

        foreach my $el (@$result) {
            foreach my $key (keys(%$el)) {
                if (grep {$key eq $_} qw(partner_w_nds partner_wo_nds)) {
                    $el->{$key} = sprintf('%0.2f', $el->{$key});
                }
            }
        }

        my $sorted = [
            sort {
                     $a->{dt} cmp $b->{dt}
                  || $a->{page_id} <=> $b->{page_id}
                  || $a->{block_id} <=> $b->{block_id}
                  || $a->{tag_id} <=> $b->{tag_id}
              } @$result
        ];

        return $sorted;
    } else {
        croak sprintf('Got %s status code from %s', $response->{status}, $url,);
    }
}

sub get_partner1_campaign_id {
    my ($app, $page_id) = @_;

    my $model = $app->partner_db->all_pages->get_all(filter => {page_id => $page_id,},)->[0]->{model};

    if (!$model) {
        croak sprintf('Can\'t find Page ID %s in partner2 mysql table "all_pages"', $page_id,);
    }

    my $page_id_field_name = $app->$model->get_page_id_field_name();

    my $data = $app->$model->get_all(
        fields => [$page_id_field_name, 'pi_id'],
        filter => {$page_id_field_name => $page_id,},
    );

    my $campaign_id = $data->[0]->{'pi_id'};

    if (!$campaign_id) {
        croak sprintf('Can\'t find partner1 Campaign ID for Page ID %s ("model %s")', $page_id, ($model // 'undef'),);
    }

    return $campaign_id;
}

sub get_login {
    my ($app, $page_id) = @_;

    my $login = $app->partner_db->all_pages->get_all(filter => {page_id => $page_id,},)->[0]->{login};

    if (!$login) {
        croak sprintf('Can\'t find login for Page ID %s', $page_id,);
    }

    return $login;
}

sub show_help {
    say <<EOM

Скрипт для сравнения статистики API ПИ1 и ПИ2.

Скрипт работает с продакшн ПИ1 и указанной системой ПИ2 (это может быть как бета ПИ2,
так и ТС или прод).

Пример вызова:

    ./bin/compare_api.pl \
        --page_id=159243 \
        --date=2017-05-10 \
        --level=advnet_context_on_site_rtb \
        --partner2_url=https://dev-partner2.yandex.ru:8468
        --partner2_oauth_token="asdf..."

Все параметры являются обязательными.

 * partner2_url - адрес беты ПИ2 на которой отключена провека токенов и которая смотрит в прод
 * partner2_oauth_token - OAuth токен с которым есть возможность ходить в интерфейс
 * page_id - можно указать только один
 * date - дата в виде 2017-05-10, можно указать только одну дату
 * level - по какому уровню статистики ПИ2 будет проходить проверка

Скрипт забирает данные из ручки ПИ1 и из ручки ПИ2, приводит из к одному виду и сравнивает что
получилось.

 * https://wiki.yandex-team.ru/partner/w/partner1-api-sp-get-stat/
 * https://wiki.yandex-team.ru/partner/w/api-statistics-get-raw/

Скрипт использует Test::Differences::eq_or_diff() для сравнения данных:

 * Got (первая колонка) - это даныне ПИ2
 * Expected (вторая колонка) - это данные ПИ1

EOM
}

sub get_opts {
    my $page_id;
    my $date;
    my $level;
    my $partner2_url;
    my $partner2_oauth_token;
    my $help;

    GetOptions(
        'page_id=i'              => \$page_id,
        'date=s'                 => \$date,
        'level=s'                => \$level,
        'partner2_url=s'         => \$partner2_url,
        'partner2_oauth_token=s' => \$partner2_oauth_token,
        'help|?|h'               => \$help,
    ) or show_help();

    if ($help) {
        show_help();
        exit 0;
    }

    my $errors = [];

    unless (defined($page_id)) {
        push @$errors, 'Must specify --page_id';
    }

    unless (defined($date)) {
        push @$errors, 'Must specify --date';
    }

    unless (defined($level)) {
        push @$errors, 'Must specify --level';
    }

    unless (defined($partner2_url)) {
        push @$errors, 'Must specify --partner2_url';
    }

    unless (defined($partner2_oauth_token)) {
        push @$errors, 'Must specify --partner2_oauth_token';
    }

    if (@$errors) {
        print join("\n", map {colored('ERROR! ' . $_, 'red')} @$errors), "\n\n";
        show_help();
        exit(1);
    }

    return ($page_id, $date, $level, $partner2_url, $partner2_oauth_token);
}

# main
sub main {
    my $app = Application->new();
    $app->pre_run();

    $app->set_cur_user({id => 0});

    no strict 'refs';
    no warnings 'redefine';
    local *{'QBit::Application::check_rights'} = sub {TRUE};

    my ($page_id, $date, $level, $partner2_url, $partner2_oauth_token) = get_opts();

    my $partner1_campaign_id = get_partner1_campaign_id($app, $page_id);
    my $login = get_login($app, $page_id);

    compare_stat(
        page_id              => $page_id,
        date                 => $date,
        partner1_campaign_id => $partner1_campaign_id,
        login                => $login,
        level                => $level,
        partner2_url         => $partner2_url,
        partner2_oauth_token => $partner2_oauth_token,
    );

    $app->post_run();

    done_testing();
}
main();
__END__
