#!/usr/bin/perl

use Carp;
use HTTP::Tiny;
use URI;
use XML::Simple;
use Getopt::Long;

use lib::abs qw(
  ../lib
  );
use qbit;
use Application;
use Utils::TSV;

my %LOGIN_PAGES;

my $FIELDS_MAP = {
    context_on_site_rtb => {
        mol => {
            level     => 'context_on_site_rtb',
            shows     => 'shows',
            hits      => 'hits_render',
            clicks    => 'clicks_direct',
            partner   => 'partner_wo_nds',
            all       => 'all_wo_nds',
            page_id   => 'page_id',
            public_id => 'complex_block_id'
        },
        billing => {
            level     => 'advnet_context_on_site_rtb',
            shows     => 'rtb_block_shows',
            hits      => 'rtb_block_hits',
            clicks    => 'rtb_block_direct_clicks',
            partner   => 'rtb_partner_wo_nds',
            all       => 'rtb_all_wo_nds',
            page_id   => 'campaign_id',
            public_id => 'public_id',
        },
    },
    context_on_site_direct => {
        mol => {
            level     => 'context_on_site_direct',
            shows     => 'shows_direct',
            hits      => 'hits_render',
            clicks    => 'clicks_direct',
            partner   => 'partner_wo_nds',
            all       => 'all_wo_nds',
            page_id   => 'page_id',
            public_id => 'complex_block_id'
        },
        billing => {
            level     => 'advnet_context_on_site_direct',
            shows     => 'direct_context_shows',
            hits      => 'rtb_block_hits',
            clicks    => 'direct_context_clicks',
            partner   => 'direct_context_partner_wo_nds',
            all       => 'direct_context_all_wo_nds',
            page_id   => 'campaign_id',
            public_id => 'public_id',
        },
    },
    internal_context_on_site_direct => {
        mol => {
            level     => 'internal_context_on_site_direct',
            shows     => 'shows_direct',
            hits      => 'hits_render',
            clicks    => 'clicks_direct',
            partner   => 'partner_wo_nds',
            all       => 'all_wo_nds',
            page_id   => 'page_id',
            public_id => 'complex_block_id'
        },
        billing => {
            level     => 'internal_advnet_context_on_site_direct',
            shows     => 'direct_context_shows',
            hits      => 'rtb_block_hits',
            clicks    => 'direct_context_clicks',
            partner   => 'direct_context_partner_wo_nds',
            all       => 'direct_context_all_wo_nds',
            page_id   => 'campaign_id',
            public_id => 'public_id',
        },
    }
};

sub compare_stat {
    my (%opts) = @_;

    my $mol_stat = get_statistics(
        %opts,
        path => '/api/statistics2/get.json',
        type => 'mol'
    );

    my $billing_stat = get_statistics(
        %opts,
        path => '/api/statistics/get.json',
        type => 'billing'
    );

    my $mol_stat_data;
    foreach my $el (@{$mol_stat}) {
        my $block_id = $el->{public_id};
        $block_id =~ s/[A-Z\-]*\d+-(\d+)/$1/;
        my $key = sprintf("%s\t%s\t%s", $el->{dt} // '', $el->{page_id} // '', $block_id // '');
        $mol_stat_data->{$key} = {
            page_id   => ($el->{page_id}   // '') . '',
            public_id => ($el->{public_id} // '') . '',
            date      => ($el->{dt}        // '') . '',
            login     => ($el->{login}     // '') . '',
            shows     => ($el->{shows}     // 0) . '',
            hits      => ($el->{hits}      // 0) . '',
            clicks    => ($el->{clicks}    // 0) . '',
            partner => sprintf('%.2f', ($el->{partner} // 0)) . '',
            all     => sprintf('%.2f', ($el->{all}     // 0)) . '',
        };
    }

    my $billing_stat_data;
    foreach my $el (@{$billing_stat}) {
        my $block_id = $el->{public_id};
        $block_id =~ s/[A-Z\-]+\d+-(\d+)/$1/;
        my $key = sprintf("%s\t%s\t%s", $el->{dt} // '', $el->{page_id} // '', $block_id // '');
        $billing_stat_data->{$key} = {
            page_id   => ($el->{page_id}   // '') . '',
            public_id => ($el->{public_id} // '') . '',
            date      => ($el->{dt}        // '') . '',
            login     => ($el->{login}     // '') . '',
            shows     => ($el->{shows}     // 0) . '',
            hits      => ($el->{hits}      // 0) . '',
            clicks    => ($el->{clicks}    // 0) . '',
            partner => sprintf('%.2f', ($el->{partner} // 0)) . '',
            all     => sprintf('%.2f', ($el->{all}     // 0)) . '',
        };
    }

    my $billing_total_sum    = 0;
    my $mol_total_sum        = 0;
    my $billing_total_shows  = 0;
    my $mol_total_shows      = 0;
    my $billing_total_hits   = 0;
    my $mol_total_hits       = 0;
    my $billing_total_clicks = 0;
    my $mol_total_clicks     = 0;
    foreach my $key (@{array_uniq([keys %$billing_stat_data], [keys %$mol_stat_data])}) {
        my $billing_login   = ($billing_stat_data->{$key}{login}   // 'no_billing_lgin');
        my $mol_login       = ($mol_stat_data->{$key}{login}       // 'no_mol_login');
        my $billing_sum     = ($billing_stat_data->{$key}{partner} // 0);
        my $mol_sum         = ($mol_stat_data->{$key}{partner}     // 0);
        my $billing_sum_all = ($billing_stat_data->{$key}{all}     // 0);
        my $mol_sum_all     = ($mol_stat_data->{$key}{all}         // 0);
        my $billing_shows   = ($billing_stat_data->{$key}{shows}   // 0);
        my $mol_shows       = ($mol_stat_data->{$key}{shows}       // 0);
        my $billing_hits    = ($billing_stat_data->{$key}{hits}    // 0);
        my $mol_hits        = ($mol_stat_data->{$key}{hits}        // 0);
        my $billing_clicks  = ($billing_stat_data->{$key}{clicks}  // 0);
        my $mol_clicks      = ($mol_stat_data->{$key}{clicks}      // 0);
        $billing_total_sum    += $billing_sum;
        $mol_total_sum        += $mol_sum;
        $billing_total_shows  += $billing_shows;
        $mol_total_shows      += $mol_shows;
        $billing_total_hits   += $billing_hits;
        $mol_total_hits       += $mol_hits;
        $billing_total_clicks += $billing_clicks;
        $mol_total_clicks     += $mol_clicks;
        print join "\t", $key, $billing_login, $mol_login, $billing_sum_all, $mol_sum_all, $billing_sum, $mol_sum,
          $billing_shows,  $mol_shows,
          $billing_hits,   $mol_hits,
          $billing_clicks, $mol_clicks . "\n";
    }
    return 1;
}

sub get_statistics {
    my (%opts) = @_;

    my $uri = URI->new($opts{beta_url});
    $uri->path($opts{path});
    $uri->query_form(
        "dimension_field" => "date|day",
        "entity_field"    => "login",
        "entity_field"    => $FIELDS_MAP->{$opts{level}}->{$opts{type}}->{page_id},
        "entity_field"    => $FIELDS_MAP->{$opts{level}}->{$opts{type}}->{public_id},
        "field"           => $FIELDS_MAP->{$opts{level}}->{$opts{type}}->{shows},
        "field"           => $FIELDS_MAP->{$opts{level}}->{$opts{type}}->{clicks},
        "field"           => $FIELDS_MAP->{$opts{level}}->{$opts{type}}->{hits},
        "field"           => $FIELDS_MAP->{$opts{level}}->{$opts{type}}->{partner},
        "field"           => $FIELDS_MAP->{$opts{level}}->{$opts{type}}->{all},
        "period"          => $opts{date_from},
        "period"          => $opts{date_to},
        (
            $opts{type} eq 'mol'
            ? ("filter" => to_json(["block_level", "IN", [$FIELDS_MAP->{$opts{level}}->{$opts{type}}->{level}]]))
            : ("level" => $FIELDS_MAP->{$opts{level}}->{$opts{type}}->{level})
        ),
        "order_by" => "[]",
        "total"    => 0,
        "vat"      => -1
    );

    my $url = $uri->as_string();

    my $response =
      HTTP::Tiny->new(timeout => 900)->request('GET', $url, {headers => {'Authorization' => $opts{token},},});

    if ($response->{status} eq '200') {
        $response->{content} =~ s/^#//;
        $response->{content} =~ s/#END\s*\z//;

        my $data   = from_json($response->{content})->{data};
        my $points = $data->{data};
        if ($opts{type} eq 'mol') {

            $points = $data->{points};
            $points = [
                map {
                    {%{$_->{dimensions}}, %{$_->{measures}->[0]}}
                  } @$points
            ];
        }

        $points = [
            map {
                {
                    login     => $_->{login},
                    public_id => $_->{$FIELDS_MAP->{$opts{level}}->{$opts{type}}->{public_id}},
                    page_id   => $_->{$FIELDS_MAP->{$opts{level}}->{$opts{type}}->{page_id}},
                    shows     => $_->{$FIELDS_MAP->{$opts{level}}->{$opts{type}}->{shows}},
                    clicks    => $_->{$FIELDS_MAP->{$opts{level}}->{$opts{type}}->{clicks}},
                    hits      => $_->{$FIELDS_MAP->{$opts{level}}->{$opts{type}}->{hits}},
                    partner   => $_->{$FIELDS_MAP->{$opts{level}}->{$opts{type}}->{partner}},
                    all       => $_->{$FIELDS_MAP->{$opts{level}}->{$opts{type}}->{all}},
                    dt        => ref($_->{date}) eq '' ? $_->{date} : $_->{date}->[0],
                }
              } @$points
        ];

        return [] if @{$points} == 0;

        my %ids = map {$_ => 1} qw(
          dt
          page_id
          public_id
          );

        my @all_fields = keys(%{$points->[0]});
        my @id_fields = grep {$ids{$_}} @all_fields;

        my @result;

        my $tmp;

        foreach my $el (@{$points}) {
            my $key = join($;, map {$el->{$_}} @id_fields);
            push @{$tmp->{$key}}, $el;
        }

        my $result = [];

        foreach my $tmp_key (keys(%$tmp)) {
            my $row = {};
            foreach my $el (@{$tmp->{$tmp_key}}) {
                foreach my $key (keys %$el) {
                    if ($ids{$key} || $key eq 'login') {
                        $row->{$key} = $el->{$key};
                    } else {
                        $row->{$key} += $el->{$key} // 0;
                    }
                }
            }
            push $result, $row;
        }

        foreach my $el (@$result) {
            foreach my $key (keys(%$el)) {
                if (grep {$key eq $_} qw(partner all)) {
                    $el->{$key} = sprintf('%0.2f', $el->{$key});
                }
            }
        }

        my $sorted =
          [sort {$a->{dt} cmp $b->{dt} || $a->{page_id} <=> $b->{page_id} || $a->{public_id} cmp $b->{public_id}}
              @$result];

        return $sorted;
    } else {
        croak sprintf('Got %s status code from %s', $response->{status}, $url,);
    }
}

sub get_opts {
    my $page_id;
    my $login;
    my $date_from;
    my $date_to;
    my $level;
    my $beta_url;
    my $token;
    my $help;

    GetOptions(
        'page_id=i'   => \$page_id,
        'login=s'     => \$login,
        'date_from=s' => \$date_from,
        'date_to=s'   => \$date_to,
        'level=s'     => \$level,
        'beta_url=s'  => \$beta_url,
        'token=s'     => \$token,
        'help|?|h'    => \$help,
    );

    my $errors = [];

    unless (defined($date_from)) {
        push @$errors, 'Must specify --date';
    }

    unless (defined($level)) {
        push @$errors, 'Must specify --level';
    }

    unless (exists $FIELDS_MAP->{$level}) {
        push @$errors, "Unknown level '$level'";
    }

    unless (defined($beta_url)) {
        push @$errors, 'Must specify --beta_url';
    }

    unless (defined($token)) {
        push @$errors, 'Must specify --token';
    }

    if (@$errors) {
        print join("\n", @$errors), "\n\n";
        exit(1);
    }

    return ($page_id, $login, $date_from, $date_to, $level, $beta_url, $token);
}

# main
sub main {
    my $app = Application->new();
    $app->pre_run();

    $app->set_cur_user({id => 0});

    no strict 'refs';
    no warnings 'redefine';
    local *{'QBit::Application::check_rights'} = sub {TRUE};

    my ($page_id, $login, $date_from, $date_to, $level, $beta_url, $token) = get_opts();

    compare_stat(
        page_id   => $page_id,
        date_from => $date_from,
        date_to   => $date_to,
        login     => $login,
        level     => $level,
        beta_url  => $beta_url,
        token     => $token,
    );

    $app->post_run();
}
main();
__END__
