#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

    Скрипт "копирует" пейджи и блоки с одного или нескольких логинов на другой.

=head1 USAGE

  ./bin/copy_pages.p --page_ids=12345,54321,...

=head1 OPTIONS

  dst_login            - логин пользователю, которому копируем площадки
  src_login            - список логинов источников площадок (если не указан, используется page_ids)
  page_id              - cписок площадок, которые копируем
  exclude_page_id      - список площадок, которые игнорируем (используется совместно с src_login)
  page_models          - список пейджовых моделей, площадки которых копируем
  exclude_page_models  - список пейджовых моделей, которые игнорируем
  block_models         - список блочных моделей, блоки которых копируем
  exclude_block_models - список блочных моделей, которые игнорируем
  block_patch          - json с изменениями для блоков

=cut

# TODO
# Скрипт делался для копирования видео-пейджей
# Для других моделей возможно нужны дополнительные допиливания (дизайны)
# После релиза видео-сценариев, для видео тоже нужны будут доработки

use Scalar::Util qw(looks_like_number);

use lib::abs qw(
  ../lib
  );

use qbit;
use Utils::ScriptWrapper;

my $APP;
my $OPTS;

sub args {
    my ($opts) = @_;

    return (
        'dst_login=s'             => \$opts->{dst_login},
        'src_login=s@'            => \$opts->{src_login},
        'page_id=s@'              => \$opts->{page_id},
        'exclude_page_id=s@'      => \$opts->{exclude_page_id},
        'page_models=s@'          => \$opts->{page_models},
        'exclude_page_models=s@'  => \$opts->{exclude_page_models},
        'block_models=s@'         => \$opts->{block_models},
        'exclude_block_models=s@' => \$opts->{exclude_block_models},
        'block_patch=s'           => \$opts->{block_patch},
    );
}

sub prepare {
    foreach (qw(src_login page_id exclude_page_id page_models exclude_page_models block_models exclude_block_models)) {
        if ($OPTS->{$_}) {
            my @list = split /\s*,\s*/, join ',', @{$OPTS->{$_}};
            my %list;
            @list{@list} = ();
            $OPTS->{$_} = [/page_id/ ? sort {$a <=> $b} keys %list : sort keys %list];
        }
    }
    unless ($OPTS->{rollback}) {
        die "Should specify src_login or page_id\n" unless ($OPTS->{src_login} || $OPTS->{page_id});
        die "Should specify dst_login\n" unless ($OPTS->{dst_login});
        my $user = $APP->users->get_all(
            fields => [qw(id login)],
            filter => {(looks_like_number($OPTS->{dst_login}) ? 'id' : 'login') => $OPTS->{dst_login}},
        )->[0];
        if ($user) {
            $OPTS->{login}   = $user->{login};
            $OPTS->{user_id} = $user->{id};
        } else {
            die sprintf(qq[User "%s" not found\n], $OPTS->{dst_login});
        }
        $OPTS->{domains} = {
            map {$_->{domain_id} => TRUE} @{
                $APP->partner_db->owner_site->get_all(
                    fields => [qw(domain_id)],
                    filter => {user_id => $OPTS->{user_id}},
                )
              }
        };
        $OPTS->{block_patch} = from_json $OPTS->{block_patch} if $OPTS->{block_patch};
    }
}

sub get_pages {
    my ($opts) = @_;

    my $filter = $APP->partner_db->filter();
    if ($OPTS->{src_login}) {
        my (@logins, @ids);
        foreach (@{$OPTS->{src_login}}) {
            if (looks_like_number($_)) {
                push @ids, $_;
            } else {
                push @logins, $_;
            }
        }
        $filter->or({login    => \@logins}) if (@logins);
        $filter->or({owner_id => \@ids})    if (@ids);
    }
    $filter->and({page_id => $OPTS->{page_id}}) if $OPTS->{page_id};
    $filter->and_not({page_id => $OPTS->{exclude_page_id}}) if $OPTS->{exclude_page_id};
    $filter->and({model => $OPTS->{page_models}}) if $OPTS->{page_models};
    $filter->and_not({model => $OPTS->{exclude_page_models}}) if $OPTS->{exclude_page_models};
    $filter->and_not({$_ => 1}) foreach (qw(is_deleted is_readonly is_blocked is_rejected));

    my $data = $APP->all_pages->get_all(
        fields   => [qw(model page_id)],
        filter   => $filter,
        order_by => [qw(page_id)],
    );
    print logstr pages => scalar @$data;

    my %pages;
    foreach (@$data) {
        $pages{$_->{model}} //= [];
        push @{$pages{$_->{model}}}, $_->{page_id};
    }

    return \%pages;
}

sub create_page {
    my ($model, %opts) = @_;

    if (exists($opts{login})) {
        $opts{login} = $OPTS->{login};
    } elsif (exists($opts{owner_id})) {
        $opts{owner_id} = $OPTS->{user_id};
    }

    if (exists($opts{block_title}) && !defined($opts{block_title})) {
        delete($opts{block_title});
    }

    if (exists($opts{domain_id}) && !$OPTS->{domains}{$opts{domain_id}}) {
        $APP->partner_db->owner_site->add(
            {
                domain_id   => $opts{domain_id},
                user_id     => $OPTS->{user_id},
                link_stat   => 'https://metrika.yandex.ru/',
                create_date => curdate(oformat => 'db_time'),
            }
        );
        $OPTS->{domains}{$opts{domain_id}} = TRUE;
    }

    return $model->add(%opts);
}

sub update_page {
    my ($model, $page_id, %opts) = @_;

    if (scalar keys %opts) {
        if (exists($opts{mirrors})) {
            $opts{mirrors} = [
                map {ref($_) ? $_->{domain} : $_}
                grep {(ref($_) && $_->{moderation_status} eq 'approved') || !ref($_)} @{$opts{mirrors}}
            ];
        }

        if (exists($opts{assistants})) {
            $opts{assistants} = [map {delete($_->{login}); $_} @{$opts{assistants}}];
        }

        $model->do_action($page_id, 'edit', %opts);
    }
}

sub patch_block {
    my ($block, $patch) = @_;

    # check conditions
    if ($patch->{where}) {
        foreach (keys %{$patch->{where}}) {
            my $val = $patch->{where}{$_};
            return
              if (defined($val) && (!defined($block->{$_}) || $val ne $block->{$_})
                || !defined($val) && defined($block->{$_}));
        }
    }

    # apply patch
    @{$block}{keys %{$patch->{set}}} = values %{$patch->{set}} if ($patch->{set});
    foreach my $field (keys %{$patch->{replace} // []}) {
        my ($search, $val) = @{$patch->{replace}{$field}};
        $block->{$field} =~ s/$search/$val/g if defined($block->{$field});
    }
}

sub create_block {
    my ($block_model, $new_page_id, $block) = @_;

    my $block_model_name = $block_model->accessor();
    my $new_block        = {
        %$block,
        $block_model->get_page_id_field_name() => $new_page_id,
        page_id                                => $new_page_id,
        create_date => curdate(oformat => 'db_time'),
        multistate  => 0,
    };
    $new_block->{public_id} = $block_model->public_id($new_block);
    $new_block->{bk_data} =~ s/$block->{public_id}/$new_block->{public_id}/g;
    delete $new_block->{articles} if defined($new_block->{picategories});

    # TODO: copy video scenaries instead of set to default, feature instream_vmap
    $new_block->{vmap_id} =
      $block_model->video_scenaries->get_or_create_default_scenario($new_page_id, $new_block->{adfox_block});

    # TODO: дизайны

    patch_block($new_block, $OPTS->{block_patch}) if $OPTS->{block_patch};

    my $stash = $block_model->hook_stash->init(
        mode => 'add',
        $block_model->hook_stash_add_fields,
        data => {opts => $new_block, current => $new_block}
    );
    $block_model->hook_fields_processing_before_validation($new_block);
    $block_model->hook_preparing_fields_to_save($new_block);
    $block_model->hook_writing_to_database($new_block);

    $APP->partner_db->$block_model_name->edit($new_block, {multistate => $block->{multistate}});

    return $new_block->{public_id};
}

sub copy_blocks {
    my ($model, $page_id, $new_page_id) = @_;

    my %exclude_block_models = map {$_ => TRUE} @{$OPTS->{exclude_block_models} // []};
    my @models = grep {!$exclude_block_models{$_}} @{$model->get_block_model_names()};
    if ($OPTS->{block_models}) {
        my %block_models = map {$_ => TRUE} @{$OPTS->{block_models}};
        @models = grep {$block_models{$_}} @models;
    }
    my $count        = 0;
    my $max_block_id = 0;
    foreach my $block_model_name (@models) {
        my $block_model = $APP->$block_model_name;
        my $all_block   = $block_model->get_all(
            fields => ['*'],
            filter => {page_id => $page_id, multistate => 'not deleted'},
        );
        foreach my $block (@$all_block) {
            my $public_id = create_block($block_model, $new_page_id, $block);
            print logstr 'COPY BLOCK', [$block->{public_id}, $public_id, $block->{caption}];
            $max_block_id = $block->{id} if ($block->{id} > $max_block_id);
            $count++;
        }
    }
    my $block_seq_db_table = $model->block_seq_db_table();
    $block_seq_db_table->replace(
        {$block_seq_db_table->primary_key->[0] => $new_page_id, next_block_id => $max_block_id + 1});

    return $count;
}

run(
    sub {
        ($APP, $OPTS) = @_;

        prepare();
        if ($OPTS->{rollback}) {
            my %pages;
            my $f;
            if (open($f, '<', $OPTS->{rollback})) {
                my $line;
                while (defined($line = <$f>)) {
                    if ($line =~ /CREATE PAGE\s+(\d+)/ || $line =~ /COPY PAGE\s+\["\d+","(\d+)"/) {
                        $pages{$1} = TRUE;
                    }
                }
                close $f;
            } else {
                die sprintf(q[Can't open "%s": %s], $OPTS->{rollback}, $!);
            }
            my $data = $APP->all_pages->get_all(
                fields   => [qw(model page_id)],
                filter   => {page_id => [keys %pages]},
                order_by => [qw(page_id)]
            );
            foreach my $row (@$data) {
                my $model_name = $row->{model};
                my $page_id    = $row->{page_id};
                unless (
                    eval {
                        $APP->$model_name->do_action($page_id, 'delete');
                        print logstr 'DELETE PAGE' => $page_id;
                    }
                  )
                {
                    print logstr FAILED => $page_id => ($@ ? "$@" : ());
                }
            }
        } else {
            my $pages = get_pages();
            foreach my $model_name (keys %$pages) {
                print logstr $model_name, scalar @{$pages->{$model_name}};
                my $model      = $APP->$model_name;
                my $add_fields = $model->get_add_fields();
                foreach my $page_id (@{$pages->{$model_name}}) {
                    unless (
                        eval {
                            my $page = $model->get($page_id, fields => ['*']);

                            my $new_page_id = create_page($model, hash_transform($page, [keys %$add_fields]));
                            print logstr 'CREATE PAGE' => $new_page_id;

                            my $editable_fields =
                              $model->get($new_page_id, fields => [qw(editable_fields)])->{editable_fields};
                            update_page($model, $new_page_id,
                                hash_transform($page, [grep {!$add_fields->{$_}} keys %$editable_fields]));
                            $model->do_action($new_page_id, 'register_in_balance')
                              if ($model_name eq 'context_on_site_campaign');
                            $APP->partner_db->$model_name->edit($new_page_id, {multistate => $page->{multistate}});
                            print logstr 'COPY PAGE' => [$page_id, $new_page_id, $page->{caption}];

                            my $blocks_couont = copy_blocks($model, $page_id, $new_page_id);
                            $APP->partner_db->$model_name->edit($new_page_id, {blocks_count => $blocks_couont});

                            $model->do_action($new_page_id, 'set_need_update');

                            return TRUE;
                        }
                      )
                    {
                        print logstr FAILED => $page_id => ($@ ? "$@" : ());
                    }
                }
            }
        }
    }
   )
