#!/usr/bin/perl

=encoding UTF-8
=cut

=head1 DESCRIPTION

Скрипт для копирования всех данных пользователей из базы ПИ1 в базу ПИ2.

Скрипт ходит в базу ПИ1, базу ПИ2 и ручку ПИ1.

Пример запуска:

    ./bin/copy_users_and_all_their_data_from_partner1_to_partner2.pl \
        --login_file=logins.txt \
        --statistics_start_date=2016-08-03 \
        --validator_report=validator_report.log \
        --mark_users_as_migrated

=cut

# project modules
use lib::abs qw(
  ../lib
  );
use qbit;
use Application;

# common modules
use feature 'say';
use Carp;
use Term::ANSIColor qw(colored);
use Date::Format;
use Utils::Logger qw/INFO WARN/;

# global vars
my $app;

# subs

# main
sub main {
    _check_branch();

    $app = Application->new();
    $app->pre_run();

    my @all_rights = keys %{$app->get_registred_rights()};
    my $tmp_rights = $app->add_tmp_rights(@all_rights);

    my $user_id = 155209804;    # yndx.bessarabov

    my $user = $app->users->get(
        $user_id,
        fields => [
            qw(
              client_id
              id
              multistate
              name
              roles

              )
        ],
    );
    $app->set_cur_user($user);

    $app->migration->read_command_line_opts();

    $app->migration->connect_to_partner1_database($app->migration->get_partner1_database_settings(),);

    my $stage = $app->migration->get_stage();

    if ($stage eq 'main') {
        stage_main();
    } elsif ($stage eq 'mark_logins_as_migrated') {
        stage_mark_logins_as_migrated();
    }

    $app->migration->output_summary();

    say "\n"
      . colored('Success.', 'green')
      . ' Please copy log file '
      . colored($app->migration->get_log_file_name(), 'green')
      . ' to Startrek ticket.';

    $app->post_run();

}

sub stage_main {
    {
        no strict 'refs';
        no warnings 'redefine';

        *{'Application::Model::PartnerDB::all_pages'} = sub {$app->partner_db->all_pages_view};
    }

    $app->migration->check_campaigns_multistates();  # check that context and search campaigns have the same bits in PI2

    # This call can modify the effective login list
    $app->migration->check_that_all_campaign_multistates_are_known();    # states that can be mapped from PI1 to PI2

    # Get the effective list after filtering by user and campaign states
    my @logins = $app->migration->get_logins();

    # Find logins linked by the same ADFOX id
    # This call may ask to you add more (linked) logins to migration and throw
    $app->migration->check_that_all_logins_here(@logins);

    my $i = 0;
    foreach my $login (@logins) {
        $i++;

        say colored(sprintf('## Login: %s (%s/%s)', $login, $i, scalar(@logins)), 'cyan');

        say '  ' . time2str('%Y/%m/%d %X', time), ' Common part START';

        try {

            $app->migration->copy_user($login);
            $app->migration->copy_role($login);
            $app->migration->copy_user_adfox($login);

            $app->migration->copy_site($login);
            $app->migration->copy_owner_site($login);

            $app->migration->copy_context_on_site_campaigns($login);
            $app->migration->copy_context_on_site_rtb($login);
            $app->migration->copy_context_on_site_adfox($login);
            $app->migration->fix_context_on_site_block_seq($login);
            $app->migration->copy_context_on_site_direct($login);
            #$app->migration->add_stat_rtb_blocks_for_direct_pages($login);
            $app->migration->copy_context_on_site_stripe($login);
            $app->migration->copy_block_tags($login);

            $app->migration->copy_search_on_site_campaigns($login);
            $app->migration->copy_search_on_site_direct($login);

            $app->migration->copy_user_global_excluded_domains($login);
            $app->migration->copy_user_global_excluded_phones($login);

            $app->migration->copy_events($login);

            say '  ' . time2str('%Y/%m/%d %X', time), ' Common part END';

            say "\n\n";

            say '  ' . time2str('%Y/%m/%d %X', time), ' Statistics part START';

            $app->migration->copy_context_rtb_stat($login);
            $app->migration->copy_context_adfox_stat($login);
            $app->migration->copy_context_direct_stat($login);
            $app->migration->copy_context_stripe_stat($login);
            $app->migration->copy_context_market_stat($login);
            $app->migration->copy_context_mcb_stat($login);
            $app->migration->copy_context_market_api_stat($login);
            $app->migration->copy_context_page_stat($login);

            $app->migration->copy_search_direct_stat($login);
            $app->migration->copy_search_premium_stat($login);
            $app->migration->copy_search_market_stat($login);
            $app->migration->copy_search_mcb_stat($login);
            $app->migration->copy_search_market_api_stat($login);
            $app->migration->copy_search_page_stat($login);

            $app->migration->reset_check_statistics_context_on_site_rtb($login);
            $app->migration->reset_check_statistics_context_on_site_adfox($login);
            # context_on_site_direct флаг check_statistics снимать не нужно
            $app->migration->reset_check_statistics_context_on_site_stripe($login);

            say '  ' . time2str('%Y/%m/%d %X', time), ' Statistics part END';

            if ($app->migration->need_mark_users_as_migrated()) {
                say "\n\n";

                say '  ' . time2str('%Y/%m/%d %X', time), ' Mark user as migrated';

                $app->migration->mark_user_as_migrated_to_pi2($login);
            }

            say "\n\n\n";
        }
        catch {
            my ($exception) = @_;
            print "EXCEPTION dump:\n";
            print $exception->as_string('full');

            # Immediately stop if DB errors
            #
            if ($exception->isa('Exception::DB')) {
                throw $exception;
            }
        };
    }
}

sub stage_mark_logins_as_migrated {

    my @logins = $app->migration->get_logins();

    $app->migration->check_that_all_logins_here(@logins);

    my $i = 0;
    foreach my $login (@logins) {
        say colored(sprintf('## Login: %s (%s/%s)', $login, $i++, scalar(@logins)), 'cyan');
        $app->migration->mark_user_as_migrated_to_pi2($login);
        say '';
    }
}

sub _check_branch {
    my $master_logs = `git log master -1`;

    my ($master_commit) = ($master_logs =~ /commit\s(.*)/);

    my $branch_logs = `git log`;

    my $is_fresh = FALSE;
    while ($branch_logs =~ /commit\s([0-9a-z]{40})/msig) {
        if ($master_commit eq $1) {
            $is_fresh = TRUE;

            last;
        }
    }

    throw gettext('Your branch outdated') unless $is_fresh;
}

main();
__END__
