#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

Скрипт для создания тикета для переводчиков с новыми переводами в текущем релизе

=head1 USAGE

perl bin/create_startrek_ticket_for_translators.pl --tanker-project=pi2 \
                               --tickets 'PI-123 PI-1234 INFRASTRUCTURE-4321' \
                               --dry-run

perl bin/create_startrek_ticket_for_translators.pl --tanker-project=pi-frontend \
                               --tickets 'PI-1037 INFRASTRUCTURE-3434' \
                               --json

=head1 OPTIONS

  tanker-project - Проект (pi2 или pi-frontend)
  tickets        - Список тикетов, переводы которых должны быть включены в тикет
  dry-run        - Вместо создания тикета, вывести его текст в STDOUT
  json           - Создание тикета и вывод ответа в STDOUT в формате JSON

=cut

use lib::abs qw(../lib);

use qbit;

use Getopt::Long qw();
use Pod::Usage;
use Config::IniFiles;
use Yandex::StarTrek;
use Yandex::Tanker;
use PiSecrets;

die 'File ".tanker" does not exists.' unless -f '.tanker';

my $config = Config::IniFiles->new(-file => '.tanker', -fallback => 'GENERAL', -nocase => TRUE)
  // die("File \".tanker\" is invalid:\n" . join("\n", @Config::IniFiles::errors));

my $TANKER_SEPARATOR  = '+++MSGCTXT+++:';
my $startrek_base_url = 'https://st.yandex-team.ru/';
my $tanker_base_url   = 'https://tanker-beta.yandex-team.ru/';
# will be added as a followers of created ticket, so translators know who to ask
my @TICKET_FOLLOWERS = qw(zurom blizzard forx);

main();

sub get_yandex_tanker {
    my ($opts) = @_;

    my $tanker_url;
    if (defined($config->val('GENERAL', 'server')) && $config->val('GENERAL', 'server') eq 'production') {
        $tanker_url = $Yandex::Tanker::PROD_URL;
    } elsif (defined($config->val('GENERAL', 'server')) && $config->val('GENERAL', 'server') eq 'test') {
        $tanker_url = $Yandex::Tanker::TEST_URL;
    } else {
        die 'No "server" option in config file ".tanker". Expected it to be "production" or "test".';
    }

    my $tanker = Yandex::Tanker->new(
        {
            project => $opts->{project},
            url     => $tanker_url,
            token   => scalar($config->val('GENERAL', 'token')),
        }
    );

    return $tanker;
}

sub yandex_startrek_create_issue {
    my ($params) = @_;

    my $ys = Yandex::StarTrek->new(oauth_token => get_secret('startrek-token'));

    $params->{type} //= 'task';

    my ($json, $status, $response_str) = $ys->_get_api_response(
        method       => 'POST',
        url          => '/v2/issues',
        content_data => $params,
    );

    die "Got unexpected status $status. Full content: $response_str" unless $json;

    return $json->{key};
}

# my $opts = {
#     project => pi2,
#     tickets => [
#         'PI-12345',
#         'INFRASTRUCTUREPI-1234',
#     ],
# };
sub get_translations_from_tanker {
    my ($opts) = @_;

    print "Downloading translations from tanker\n" unless $opts->{json};
    my $tanker  = get_yandex_tanker($opts);
    my $keysets = from_json(Encode::decode_utf8($tanker->get_project_tjson(status => 'unapproved')))->{'keysets'};
    my %tickets = map {$_ => 1} @{$opts->{tickets}};

    print "Finding translations\n" unless $opts->{json};
    my $result = {};
    foreach my $keyset (keys(%$keysets)) {
        foreach my $key (keys(%{$keysets->{$keyset}{'keys'}})) {
            my $ticket_context = $keysets->{$keyset}->{'keys'}->{$key}->{'info'}->{'context'};
            my ($ticket, $context_with_space, $context) =
              ($ticket_context // '') =~ /^$startrek_base_url([A-Za-z]{2,}-\d+)( (.*))?$/;

            if ($ticket and $tickets{$ticket}) {
                $result->{$keyset}->{$key} = 1;
            }
        }
    }

    return $result;
}

sub _get_opts {
    my $project   = '';
    my $tickets   = '';
    my $dry_run   = undef;
    my $json      = undef;
    my $initiator = '';
    my $help      = undef;

    Getopt::Long::GetOptions(
        #--- Obligatory
        'tanker-project=s' => \$project,
        'tickets=s'        => \$tickets,
        'dry-run'          => \$dry_run,
        'json'             => \$json,
        'initiator=s'      => \$initiator,
        #---
        'help|?|h' => \$help,
    ) or pod2usage(1);

    pod2usage(-verbose => 2, -noperldoc => 1) if $help;

    my @tickets = grep {$_} split /\s+/, $tickets;

    die 'No tickets specified' if @tickets == 0;
    die 'Option --tanker-project is required' unless $project;
    die '--tanker-project must be pi2 or pi-frontend' unless ($project eq 'pi2' or $project eq 'pi-frontend');
    die 'You can\'t use --json and --dry-run together' if $dry_run and $json;

    return {
        project   => $project,
        tickets   => \@tickets,
        dry_run   => $dry_run,
        json      => $json,
        initiator => $initiator
    };
}

sub main {
    my $opts = _get_opts();

    my $translations = get_translations_from_tanker($opts);

    if (keys %$translations) {
        my @strs;
        foreach my $keyset (sort keys %$translations) {
            foreach my $key (sort keys %{$translations->{$keyset}}) {
                push @strs,
                  sprintf(
                    '%s?project=%s&branch=master&keyset=%s&key=%s',
                    $tanker_base_url, uri_escape($opts->{project}),
                    uri_escape($keyset), uri_escape($key)
                  );
            }
        }

        my $keys_tree = join("\n", @strs);
        my $deadline = date_add(curdate(), day => 14, oformat => 'db');

        my $text = "Сервис (группа сервисов)::
PI

Тип задачи:
Перевод

Вид текста:
Интерфейс и формы

Суть задачи:
Перевод новой фичи

<{**Ссылки на ключи в Танкере:**
$keys_tree
}>

Язык исходного текста:
Русский/Russian (ru-RU)

Язык перевода:
Английский/English US (en-US)

Дедлайн:
$deadline

Обоснование дедлайна:
Плановое обновление

**Дополнительная информация**
если языков в кейсете нет, то добавьте их, пожалуйста
";

        my $ys_create_ticket_params = {
            queue               => 'TRANSLATEADMIN',
            type                => 'translation',
            branch              => 'new',
            summary             => '[PI] — Перевод — ru-RU > en-US — Перевод интерфейса',
            description         => $text,
            components          => ['interface'],
            deadline            => $deadline,
            serviceServiceGroup => 'PI',
            sourceLanguage      => 'ru-RU',
            targetLanguage      => ['en-US'],
            abcService          => [{id => 867}],
            tags                => ['upgrade'],
            followers           => [@TICKET_FOLLOWERS, $opts->{initiator} ? ($opts->{initiator}) : ()],
        };

        if ($opts->{dry_run}) {
            ldump([[$ys_create_ticket_params], ['$ys_create_ticket_params']]);
            $ys_create_ticket_params->{summary} = 'TESTING - SKIP IT ' . $ys_create_ticket_params->{summary};
        }

        my $id = yandex_startrek_create_issue($ys_create_ticket_params);

        if ($opts->{json}) {
            print to_json({ticket_created => JSON::XS::true, url => $startrek_base_url . $id});
        } else {
            printf "New %s translations ticket - %s%s\n", $opts->{project}, $startrek_base_url, $id;
        }

    } else {
        if ($opts->{json}) {
            print to_json({ticket_created => JSON::XS::false});
        } else {
            printf "No translations found for project '%s'\n", $opts->{project};
        }
    }
}
