#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

    The script prepares the Deploy spec template by filling placeholders with real values
    (You can view the current build settings in the Deploy interface.
    Or you van use "ya tool dctl get stage partner-<Stage>-stage")

=head1 USAGE

    # 1. Обновляем шаблон
    ./bin/deploy/create_deploy_specs.pl --stage=test  --yav_token=`cat ~/.yav_token`  --with_cur_tags --sort_specs  --cached --update_tmpl

    # 2. Проверем  результат
    ./bin/deploy/create_deploy_specs.pl --stage=test  --yav_token=`cat ~/.yav_token`  --with_cur_tags --sort_specs  --cached --vimdiff_json

    # 3. Если нужно обновить локальные спеки и проставить плейсхолдеры в часто изменяемых местах (во избежание больших диффов при ПРах)
    ./bin/deploy/create_deploy_specs.pl --stage=test  --yav_token=`cat ~/.yav_token`  --with_cur_tags --sort_specs  --cached --vimdiff_json --add_placeholders

    # 4. Получаем команду для заливки в Deploy
    ./bin/deploy/create_deploy_specs.pl --stage=test  --with_cur_tags  --yav_token=`cat ~/.yav_token`

    # 5. Если нужно только обновить версии Workload, то указываем нужные таги
    ./bin/deploy/create_deploy_specs.pl \
        --stage=release-integration-test \
        --perl_tag=experimental-blizzard-0.0.23 \
        --node_tag=0.0.933_e7089d8f37625777bb227c03e55a5cb256d5f913 \
        --java_tag=1.31e7132e7caa3e5cf59558de948d3b65194f1f96-1
        --yav_token=`cat ~/.yav_token`

=head1 OPTIONS

    stage                        - base deploy spec (--stage=test)

    Optional:
        cached                   - cache secrets to speed up
        sort_specs               - sort specs

        with_cur_tags            - use current stage tags

        update_tmpl              - run vimdiff  new_spec_template.json  template_spec.json
        vimdiff_tmpl             - run vimdiff  new_spec.json           template_spec.json
        vimdiff_json             - run vimdiff  cur_spec.json           new_spec.json
        vimdiff_yaml             - run vimdiff  cur_spec.yaml           new_spec.yaml

        stage_name               - A stage name (--stage_name=blizzard-test-pi-backend-stage)
        config                   - A path to a deploy spec (--config=configs/test/deploy-stage.json)
        project                  - A project name (--project=blizzard-test-pi-backend)

        perl_tag                 - perl docker image tag (--perl_tag=experimental-blizzard-0.0.23)
        perl_image               - A perl docker image name (--perl_image=partners/perl-backend)

        java_tag                 - java docker images tag (--java_tag=1.31e7132e7caa3e5cf59558de948d3b65194f1f96-1)

        node_tag                 - nodejs docker image tag (--node_tag=0.0.933_e7089d8f37625777bb227c03e55a5cb256d5f913)
        node_image               - A nodejs docker image name (--node_image=)

        database_image           - Database docker image. Registry is used registry.yandex.net (--database_image='partners/partner2-db-general')
        database_tag             - Image tag for database_image or latest. database_tag is used only if database_image is specified (--database_tag=2.18.2128-2020-04-29)

        mock_api_tag             - MockApi image tag ("0.0.1")
        mock_api_image           - MockApi image name ("mock-service")

        yav_token                - A token for yav
        yp_token                 - A token for yp (https://oauth.yandex-team.ru/authorize?response_type=token&client_id=f8446f826a6f4fd581bf0636849fdcd7)
        owner                    - A stage owner (--owner=blizzard)
        adfox_host               - Adfox HOST (--adfox_host=login.adfox.ru).
        ya_dir                   - directory where located ya
        execute                  - Run script with executing "ya tool dctl put stage"
        save_temp_files          - Remove temp files

        add_placeholders         - Replace unstable values with placeholders

=cut

use lib::abs qw(../../lib);

use qbit;

use Pod::Usage;
use Getopt::Long qw();

use Carp 'croak';
use JSON::PP qw();
use YAML::PP;
use Term::ANSIColor qw(colored);

use Scalar::Util qw(refaddr);

use File::Temp qw(tempfile);

use Utils::Betas;
use Utils::Deploy;
use Utils::Deploy::Secrets;

use PiSecrets;
use Utils::DockerRegistry qw(get_last_docker_db_image_tag);

my $UPDATES = {
    adfox        => \&set_adfox_host,
    cron         => \&set_cron_app_images_for_boxes,
    autotest_dc  => \&set_autotest_dc,
    autotest_dt  => \&set_autotest_disk_type,
    fixtures     => \&set_testapi_image_for_boxes,
    frontend     => \&set_frontend_node_images_for_boxes,
    hourglass    => \&set_hourglass_image_for_boxes,
    java_jsonapi => \&set_java_jsonapi_tag_for_boxes,
    java_intapi  => \&set_java_intapi_tag_for_boxes,
    mysql        => \&set_mysql_images_for_boxes,
    perl         => \&set_perl_app_images_for_boxes,
    mock_api     => \&set_mock_api_images_for_boxes,
    redeploy_db  => \&set_redeploy_enforcer,
    secrets      => \&set_secrets,
    haproxy      => \&set_haproxy_image_for_boxes
};

my $YPP;
my $autotest_dc_by_stage_name;
my $autotest_disk_type_by_stage_name;

main();

sub main {

    $YPP = YAML::PP->new(boolean => 'JSON::PP', header => 0, indent => 2,);

    patch_yaml_pp();

    my $args = _get_args();

    set_replica_set_path();

    my @result;
    my @files;

    my $stages = $args->{stage_name};
    $stages = [$stages] if ref $stages ne 'ARRAY';

    my $tmpl_file_path = $args->{'config'};

    foreach my $stage_name (@$stages) {

        $args->{'stage_name'} = $stage_name;

        _set_current_stage_tags($args) if $args->{with_cur_tags};

        my $spec =
          $args->{update_tmpl}
          ? get_cur_spec($stage_name, $YPP)
          : get_cur_template($args);

        $spec->{annotations}{project} = $args->{project};
        $spec->{meta}{id}             = $stage_name;

        foreach my $acl (@{$spec->{meta}{acl}}) {
            map {s/^(deploy:partner)\.[^.]+\.(MAINTAINER)$/$1\.$stage_name\.$2/; $_} @{$acl->{subjects}};
        }

        unshift @{$spec->{meta}{acl}[0]{subjects}}, $args->{owner} if $args->{owner};

        my $stage_units = $STAGE_DEFAULTS->{$args->{stage}}->{'units'};
        foreach my $unit_name (keys %$stage_units) {
            my $unit = $stage_units->{$unit_name};
            $unit->{'deploy_unit_name'} = $unit_name;
            foreach my $unit_update (@{$unit->{'unit_updates'} // []}) {
                $UPDATES->{$unit_update}->($spec, $unit, $args, for_tmpl => $args->{update_tmpl});
            }
        }

        if ($args->{update_tmpl}) {
            if ($args->{sort_specs}) {
                my $tmpl_spec = get_cur_template($args);
                _dump_tmpl_to_json($args, $tmpl_spec);
            }
            enrich_new_tmpl($args, $spec);
            my $new_tmpl_path = _dump_new_to_tmpl($args, $spec);
            system(sprintf 'vimdiff  %s   %s', $new_tmpl_path, $tmpl_file_path);
        } else {
            $spec->{spec}{revision_info}{description} = "Manual update by create_deploy_specs.pl";

            if ($args->{'vimdiff_tmpl'}) {
                my $new_json_spec_filename = _dump_new_to_json($args, $spec);
                system(sprintf 'vimdiff  %s   %s', $new_json_spec_filename, $tmpl_file_path);
            } elsif ($args->{'vimdiff_json'}) {
                my $cur_json_spec_filename = _dump_cur_to_json($args, $YPP);
                my $new_json_spec_filename = _dump_new_to_json($args, $spec);
                system(sprintf 'vimdiff  %s   %s', $cur_json_spec_filename, $new_json_spec_filename);
            } elsif ($args->{'vimdiff_yaml'}) {
                my $cur_yaml_spec_filename = get_cur_yaml($args->{stage_name});
                my $new_yaml_spec_filename = _dump_new_to_yaml($args, $spec, $YPP);
                system(sprintf 'vimdiff  %s   %s', $cur_yaml_spec_filename, $new_yaml_spec_filename);
            } else {
                my $new_yaml_spec_filename = _dump_new_to_yaml($args, $spec, $YPP);
                push @result, "ya tool dctl put stage -c xdc $new_yaml_spec_filename";
                push @files,  $new_yaml_spec_filename;
            }
        }

    }
    print "To upload new spec to Deploy execute:\n" if @result;
    for my $cmd (sort @result) {
        print "$cmd\n";
        run_shell($cmd) if $args->{execute};
    }
    clean_temp_files() unless $args->{save_temp_files};
}

sub _sort_spec {
    my ($args, $spec) = @_;

    my $deploy_units = $spec->{spec}{deploy_units};
    foreach my $unit_name (keys %$deploy_units) {
        my $unit = $STAGE_DEFAULTS->{$args->{stage}}{units}{$unit_name};

        # yav_secrets
        my $pod_template_spec = get_node_by_path(
            $args, $unit, $spec,
            [
                'spec',                      'deploy_units',
                $unit->{'deploy_unit_name'}, @{$unit->{'replica_set_path'}},
                'pod_template_spec',         'spec'
            ]
        );
        if (exists $pod_template_spec->{yav_secrets}) {
            $pod_template_spec->{yav_secrets} = [sort @{$pod_template_spec->{yav_secrets}}];
        }

        my $tvm_config = {};
        unless (grep {$_ eq $args->{stage}} $FUNC_TESTS_STAGE) {
            $tvm_config = get_node_by_path($args, $unit, $spec, ['spec', 'deploy_units', $unit_name, 'tvm_config']);

            # tvm clients
            $tvm_config->{'clients'} = [sort {$a->{source}{alias} cmp $b->{source}{alias}} @{$tvm_config->{'clients'}}]
              if $tvm_config->{'clients'};
        }

        # tvm destinations
        foreach my $tvm_client (@{$tvm_config->{'clients'} // []}) {
            $tvm_client->{'destinations'} = [sort {$a->{alias} cmp $b->{alias}} @{$tvm_client->{'destinations'}}];
        }

        # clusters
        if ($unit->{'replica_set_path'}[0] eq 'multi_cluster_replica_set') {
            my $replica_set =
              get_node_by_path($args, $unit, $spec,
                ['spec', 'deploy_units', $unit_name, @{$unit->{'replica_set_path'}}]);
            $replica_set->{'clusters'} = [sort {$a->{cluster} cmp $b->{cluster}} @{$replica_set->{'clusters'}}];
        }

        my $pod_spec = get_node_by_path(
            $args, $unit, $spec,
            [
                'spec', 'deploy_units', $unit_name, @{$unit->{'replica_set_path'}},
                'pod_template_spec', 'spec', 'pod_agent_payload', 'spec'
            ]
        );

        # boxes
        $pod_spec->{'boxes'} = [sort {$a->{id} cmp $b->{id}} @{$pod_spec->{'boxes'}}];

        # env
        foreach my $box (@{$pod_spec->{'boxes'}}) {
            if ($box->{'env'}) {
                $box->{'env'} = [sort {$a->{name} cmp $b->{name}} @{$box->{'env'}}];
            }
        }

        # mutable_workloads
        $pod_spec->{'mutable_workloads'} =
          [sort {$a->{workload_ref} cmp $b->{workload_ref}} @{$pod_spec->{'mutable_workloads'}}];

        # workloads
        $pod_spec->{'workloads'} = [sort {$a->{id} cmp $b->{id}} @{$pod_spec->{'workloads'}}];
    }

    return 1;
}

sub set_autotest_dc {
    my ($spec, $unit, $args) = @_;

    $autotest_dc_by_stage_name ||= get_per_cluster_stages();

    foreach my $dc (keys %$autotest_dc_by_stage_name) {
        if (in_array($args->{stage_name}, $autotest_dc_by_stage_name->{$dc})) {
            $spec->{spec}{deploy_units}{$unit->{'deploy_unit_name'}}{replica_set}{per_cluster_settings} =
              {$dc => {deployment_strategy => {max_unavailable => 1}, pod_count => 1}};
        }
    }
}

sub set_autotest_disk_type {
    my ($spec, $unit, $args) = @_;

    $autotest_disk_type_by_stage_name ||= get_stages_storage_class(($args->{project} || 'partner'), $args->{yp_token});
    foreach my $disk_type (keys %$autotest_disk_type_by_stage_name) {
        if (in_array($args->{stage_name}, $autotest_disk_type_by_stage_name->{$disk_type})) {

            my $disk_volume_requests = get_node_by_path(
                $args, $unit, $spec,
                [
                    'spec',                      'deploy_units',
                    $unit->{'deploy_unit_name'}, @{$unit->{'replica_set_path'}},
                    'pod_template_spec',         'spec',
                    'disk_volume_requests'
                ]
            );

            $disk_volume_requests->[0]->{storage_class} = $disk_type;
        }
    }
}

sub _get_args {
    my $args = {};

    my $result = Getopt::Long::GetOptions(

        #--- Obligatory
        'stage=s' => \$args->{'stage'},

        #--- Optional
        'with_cur_tags!' => \$args->{'with_cur_tags'},

        'cached!'    => \$args->{'cached'},
        'sort_specs' => \$args->{'sort_specs'},

        'update_tmpl!'  => \$args->{'update_tmpl'},
        'vimdiff_tmpl!' => \$args->{'vimdiff_tmpl'},
        'vimdiff_json!' => \$args->{'vimdiff_json'},
        'vimdiff_yaml!' => \$args->{'vimdiff_yaml'},

        'stage_name:s' => \$args->{'stage_name'},
        'config:s'     => \$args->{'config'},
        'project:s'    => \$args->{'project'},

        'perl_tag:s'   => \$args->{'perl_tag'},
        'perl_image:s' => \$args->{'perl_image'},

        'node_tag:s'   => \$args->{'node_tag'},
        'node_image:s' => \$args->{'node_image'},

        'java_tag:s' => \$args->{'java_tag'},

        'database_tag:s'   => \$args->{'database_tag'},
        'database_image:s' => \$args->{'database_image'},

        'mock_api_tag:s'   => \$args->{'mock_api_tag'},
        'mock_api_image:s' => \$args->{'mock_api_image'},

        'yav_token:s' => \$args->{'yav_token'},
        'yp_token:s'  => \$args->{'yp_token'},

        'owner:s'          => \$args->{'owner'},
        'adfox_host:s'     => \$args->{'adfox_host'},
        'ya_dir:s'         => \$args->{'ya_dir'},
        'help|?|h'         => \$args->{'help'},
        'execute!'         => \$args->{'execute'},
        'save_temp_files!' => \$args->{'save_temp_files'},

        'add_placeholders' => \$args->{'add_placeholders'},
    );

    if (!$result || $args->{'help'}) {
        pod2usage(-verbose => 2, -noperldoc => 0);
    }

    for (qw/stage yav_token yp_token/) {
        next if $_ eq 'yp_token';
        ERROR("'$_' expected") unless defined($args->{$_});
    }

    my @stage_valid_values = keys %$STAGE_DEFAULTS;
    ERROR('Wrong "stage"', 'valid values: ' . join(', ', sort @stage_valid_values))
      unless (in_array($args->{stage}, \@stage_valid_values));

    if ($args->{stage_name}) {
        my @stage_names_valid_values = map {ref($_) ? @$_ : $_} map {$_->{stage_name};} values %$STAGE_DEFAULTS;
        ERROR('Wrong "stage_name"', "Valid values: " . join(', ', sort @stage_names_valid_values))
          unless (in_array($args->{stage_name}, \@stage_names_valid_values));
    }

    unless ($args->{with_cur_tags} || $args->{'update_tmpl'}) {
        foreach my $arg_name (qw( perl_tag  node_tag  java_tag  mock_api_tag )) {
            ERROR(qq["$arg_name" expected], '(or use --with_cur_tags)') unless defined($args->{$arg_name});
        }

        if (exists $STAGE_DEFAULTS->{$args->{stage}}{units}{$DATABASE_UNIT_NAME}) {
            foreach my $arg_name (qw( database_tag database_image )) {
                ERROR(qq["$arg_name" expected], '(or use --with_cur_tags)') unless defined($args->{$arg_name});
            }
        }
    }

    _enrich_args($args);

    return $args;
}

sub _enrich_args {
    my ($args) = @_;

    for my $opt (
        qw(adfox_host adfox_host1 project perl_image stage_name stage_tag config node_image cron_unit mock_image))
    {
        $args->{$opt} //= $STAGE_DEFAULTS->{$args->{stage}}{$opt};
    }

    $args->{stage_tag} //= $args->{'stage'};

    $args->{force_db_redeploy} = $STAGE_DEFAULTS->{$args->{stage}}{force_db_redeploy};

    if ($args->{database_image}) {
        $args->{database_tag} //= get_last_docker_db_image_tag($args->{database_image});
    }

    $args->{save_temp_files} = TRUE if not defined $args->{'save_temp_files'};

    return 1;
}

sub _set_current_stage_tags {
    my ($args) = @_;

    my $cur_spec_data = get_cur_spec($args->{stage_name}, $YPP);

    # Backend
    if (exists $STAGE_DEFAULTS->{$args->{stage}}{units}{Backend}) {
        my $unit = {deploy_unit_name => 'Backend'};

        # PerlApp
        @$args{qw(perl_image perl_tag)} = set_perl_app_images_for_boxes($cur_spec_data, $unit, undef)
          if get_box_spec($cur_spec_data, 'Backend', 'PerlApp');

        # JavaAppJSONAPI
        $args->{'java_tag'} = set_java_jsonapi_tag_for_boxes($cur_spec_data, $unit, undef)
          if get_box_spec($cur_spec_data, 'Backend', 'JavaAppJSONAPI');

        # FrontendNode
        @$args{qw(node_image node_tag)} = set_frontend_node_images_for_boxes($cur_spec_data, $unit, undef)
          if get_box_spec($cur_spec_data, 'Backend', 'FrontendNode');

        # MockAPI
        @$args{qw(mock_api_image mock_api_tag)} = set_mock_api_images_for_boxes($cur_spec_data, $unit, undef)
          if get_box_spec($cur_spec_data, 'Backend', 'MockAPI');

    }

    # Database
    if (exists $STAGE_DEFAULTS->{$args->{stage}}{units}{$DATABASE_UNIT_NAME}) {
        my $unit = {deploy_unit_name => $DATABASE_UNIT_NAME};

        # MySQL
        @$args{qw(database_image database_tag)} = set_mysql_images_for_boxes($cur_spec_data, $unit, undef)
          if get_box_spec($cur_spec_data, $DATABASE_UNIT_NAME, 'MySQL');
    }

    return 1;
}

sub set_cron_app_images_for_boxes {
    return set_box_image_and_tag(
        @_,
        'box'   => 'CronApp',
        'image' => 'perl_image',
        'tag'   => 'perl_tag',
    );
}

sub set_frontend_node_images_for_boxes {
    return set_box_image_and_tag(
        @_,
        'box'   => 'FrontendNode',
        'image' => 'node_image',
        'tag'   => 'node_tag',
    );
}

sub set_haproxy_image_for_boxes {
    return set_box_param(
        @_,
        'box'   => 'Haproxy',
        'param' => 'tag',
        'arg'   => 'perl_tag',
    );
}

sub set_hourglass_image_for_boxes {
    return set_box_param(
        @_,
        'box'   => 'JavaAppHourglass',
        'param' => 'tag',
        'arg'   => 'java_tag',
    );
}

sub set_java_intapi_tag_for_boxes {
    return set_box_param(
        @_,
        'box'   => 'JavaAppINTAPI',
        'param' => 'tag',
        'arg'   => 'java_tag',
    );
}

sub set_java_jsonapi_tag_for_boxes {
    return set_box_param(
        @_,
        'box'   => 'JavaAppJSONAPI',
        'param' => 'tag',
        'arg'   => 'java_tag',
    );
}

sub set_mock_api_images_for_boxes {
    return set_box_image_and_tag(
        @_,
        'box'   => 'MockAPI',
        'image' => 'mock_api_image',
        'tag'   => 'mock_api_tag',
    );
}

sub set_mysql_images_for_boxes {
    my ($spec, $unit, $args, %opts) = @_;

    if ($args) {
        my $disk_volume_requests = get_node_by_path(
            $args, $unit, $spec,
            [
                'spec',                      'deploy_units',
                $unit->{'deploy_unit_name'}, @{$unit->{'replica_set_path'}},
                'pod_template_spec',         'spec',
                'disk_volume_requests'
            ]
        );

        $disk_volume_requests->[0]{id} = "$args->{stage_name}-disk-0";
    }

    return set_box_image_and_tag(
        $spec, $unit, $args, %opts,
        'box'   => 'MySQL',
        'image' => 'database_image',
        'tag'   => 'database_tag',
    );

}

sub set_perl_app_images_for_boxes {
    return set_box_image_and_tag(
        @_,
        'box'   => 'PerlApp',
        'image' => 'perl_image',
        'tag'   => 'perl_tag',
    );
}

sub _dump_new_to_yaml {
    my ($args, $spec, $YPP) = @_;
    _sort_spec($args, $spec) if $args->{sort_specs};
    my $new_yaml_spec_filename_yaml = sprintf 'deploy-%s-spec-new.yaml', $args->{stage_name};
    $YPP->dump_file($new_yaml_spec_filename_yaml, $spec);
    return $new_yaml_spec_filename_yaml;
}

sub _dump_cur_to_json {
    my ($args, $YPP) = @_;
    my $cur_spec_data = get_cur_spec($args->{stage_name}, $YPP);
    _sort_spec($args, $cur_spec_data) if $args->{sort_specs};
    my $cur_json_spec_filename = sprintf 'deploy-%s-spec-cur.json', $args->{stage_name};
    writefile($cur_json_spec_filename, to_json($cur_spec_data, pretty => TRUE, use_pp => TRUE));
    system(sprintf 'python bin/deploy/add_placeholders.py %s %s', $cur_json_spec_filename, $args->{stage})
      if ($args->{add_placeholders});
    return $cur_json_spec_filename;
}

sub _dump_new_to_tmpl {
    my ($args, $spec) = @_;
    _sort_spec($args, $spec) if $args->{sort_specs};
    (my $new_tmpl_path = $args->{'config'}) =~ s/.json/.new.json/;
    writefile($new_tmpl_path, to_json($spec, pretty => TRUE, use_pp => TRUE));
    return $new_tmpl_path;
}

sub _dump_tmpl_to_json {
    my ($args, $tmpl_spec) = @_;
    _sort_spec($args, $tmpl_spec) if $args->{sort_specs};
    writefile($args->{'config'}, to_json($tmpl_spec, pretty => TRUE, use_pp => TRUE));
    return $args->{'config'};
}

sub _dump_new_to_json {
    my ($args, $spec) = @_;
    _sort_spec($args, $spec) if $args->{sort_specs};
    my $new_json_spec_filename = sprintf 'deploy-%s-spec-new.json', $args->{stage_name};
    writefile($new_json_spec_filename, to_json($spec, pretty => TRUE, use_pp => TRUE));
    return $new_json_spec_filename;
}

sub enrich_new_tmpl {
    my ($args, $spec) = @_;

    # acl
    map {
        $_->{subjects} =
          [(grep {$_ ne 'abc:service:478'} @{$_->{subjects}}), 'abc:service:478']
    } @{$spec->{meta}->{acl}};

    # leave only "deploy_engine"
    map {delete $spec->{labels}->{$_}} grep {$_ ne 'deploy_engine'} keys %{$spec->{labels}};

    # tag
    $spec->{labels}->{tags} = [$args->{stage_tag}];

    # revision_info
    $spec->{spec}{revision_info} = {};

    # revision
    $spec->{spec}{revision} = undef;
    my $deploy_units = $spec->{spec}{deploy_units};
    map {$deploy_units->{$_}{revision} = undef} keys %$deploy_units;

    return 1;
}

sub get_cur_template {
    my ($args) = @_;
    my $tmpl_spec_data = from_json(readfile($args->{'config'}), use_pp => TRUE);
    return $tmpl_spec_data;
}

sub set_box_image_and_tag {
    my ($spec, $unit, $args, %opts) = @_;
    my $image = set_box_param(
        $spec, $unit, $args, %opts,
        'param' => 'name',
        'arg'   => $opts{image},
    );
    my $tag = set_box_param(
        $spec, $unit, $args, %opts,
        'param' => 'tag',
        'arg'   => $opts{tag},
    );
    return ($image, $tag);
}

sub set_box_param {
    my ($spec,     $unit,      $args,       %opts)            = @_;
    my ($box_name, $box_param, $args_param, $is_for_template) = @opts{qw( box  param  arg  for_tmpl )};

    my $box_spec = get_box_spec($spec, $unit->{'deploy_unit_name'}, $box_name);
    if ($args) {
        $box_spec->{$box_param} = $is_for_template ? '%s' : $args->{$args_param};
    }
    return $box_spec->{$box_param};
}

sub set_adfox_host {
    my ($spec, $unit, $args, %opts) = @_;

    my $pod_agent_payload_spec = get_node_by_path(
        $args, $unit, $spec,
        [
            'spec', 'deploy_units',
            $unit->{'deploy_unit_name'},
            @{$unit->{'replica_set_path'}},
            'pod_template_spec', 'spec', 'pod_agent_payload', 'spec'
        ]
    );

    # ... /boxes/.../ADFOX_HOST
    my $box_spec = get_node_by_id($pod_agent_payload_spec->{'boxes'}, 'PerlApp');
    my $node = eval {get_node_by_tag($box_spec->{env}, 'name', 'ADFOX_HOST')};
    my $perl_app_env = $node->{value}{literal_env} if $node;

    # ... /workloads/.../ADFOX_HOST
    $node = get_node_by_id($pod_agent_payload_spec->{'workloads'}, 'frontend_node');
    $node = get_node_by_tag($node->{env}, 'name', 'ADFOX_HOST');
    my $front_node_env = $node->{value}{literal_env};

    if ($args) {
        $perl_app_env->{value} = $opts{for_tmpl} ? '%s' : $args->{adfox_host1} if $perl_app_env;
        $front_node_env->{value} =
            $opts{for_tmpl}         ? '%s'
          : ref $args->{adfox_host} ? $args->{adfox_host}{$args->{stage_name}}
          :                           $args->{adfox_host};
    } else {
        return ($perl_app_env->{value}, $front_node_env->{value});
    }
}

sub get_node_by_path {
    my ($args, $unit, $config, $path) = @_;

    croak('Expected array') if ref($path) ne 'ARRAY';

    my $node     = $config;
    my @cur_path = ();
    foreach my $key (@$path) {
        if (ref($node) ne 'HASH' || !exists($node->{$key})) {
            croak(
                sprintf(
                    'Key "%s" not found by path "%s" (stage "%s"; unit "%s")',
                    $key, join('.', @cur_path),
                    $args->{stage}, $unit->{'deploy_unit_name'}
                )
            );
        }

        $node = $node->{$key};
        push @cur_path, $key;
    }

    return $node;
}

sub set_testapi_image_for_boxes {
    return set_box_param(
        @_,
        'box'   => 'JavaAppTESTAPI',
        'param' => 'tag',
        'arg'   => 'java_tag',
    );
}

sub set_redeploy_enforcer {
    my ($spec, $unit, $args, %opts) = @_;

    my $is_for_template = $opts{for_tmpl};

    if ($args->{force_db_redeploy}) {
        my $static_resources = get_node_by_path(
            $args, $unit, $spec,
            [
                'spec', 'deploy_units',
                $unit->{'deploy_unit_name'},
                @{$unit->{'replica_set_path'}},
                'pod_template_spec', 'spec', 'pod_agent_payload', 'spec', 'resources', 'static_resources'
            ]
        );

        my $node = get_node_by_id($static_resources, 'RedeployEnforcer');
        $node->{files}{files}[0]{raw_data} = $is_for_template ? '%s' : "" . time();
    }
}

sub set_secrets {
    my ($spec, $unit, $args) = @_;

    my $pod_template_spec = get_node_by_path(
        $args, $unit, $spec,
        [
            'spec',                      'deploy_units',
            $unit->{'deploy_unit_name'}, @{$unit->{'replica_set_path'}},
            'pod_template_spec',         'spec'
        ]
    );

    if ($args->{update_tmpl}) {
        my $sec_ids = $pod_template_spec->{yav_secrets}
          // [map {$_->{secret_id}} values %{$pod_template_spec->{secrets}}];
        $pod_template_spec->{yav_secrets} = [sort @$sec_ids];
        delete $pod_template_spec->{secrets};
    } else {
        my $secrets_cache_filename =
          sprintf('.yav_secrets_%s_%s_cache.json', $args->{stage}, $unit->{'deploy_unit_name'});

        my $secrets;
        if ($args->{cached}) {

            $secrets = eval {from_json(readfile($secrets_cache_filename), use_pp => TRUE)};

            my $sec_keys = {map {$_ => 1} map {s/^.*:sec-/sec-/; $_} keys %$secrets};

            my $is_equal = 1;
            foreach my $key (@{$pod_template_spec->{'yav_secrets'}}) {
                unless (exists $sec_keys->{$key}) {
                    $is_equal = 0;
                    last;
                } else {
                    delete $sec_keys->{$key};
                }
            }
            $is_equal = 0 if scalar(keys %$sec_keys) > 0;
            $secrets = undef unless $is_equal;
        }

        unless ($secrets) {
            $secrets = load_secrets_tokens(
                $pod_template_spec->{'yav_secrets'}, $args->{stage_name},
                $unit->{'deploy_unit_name'},         $args->{yav_token}
            );
            writefile($secrets_cache_filename, to_json($secrets, pretty => TRUE, use_pp => TRUE));
        }

        $pod_template_spec->{'secrets'} = $secrets;

        delete $pod_template_spec->{'yav_secrets'};
    }

    return 1;
}

sub clean_temp_files {
    map {unlink $_} (
        glob(".yav_secrets_*_cache.json"), glob("deploy-*-spec-cur.yaml"),
        glob("deploy-*-spec-cur.json"),    glob("deploy-*-spec-new.yaml")
    );
}

sub ERROR {
    my ($msg, $descr) = @_;
    $descr = defined($descr) ? "\n\t" . $descr : '';
    print "\n";
    die colored('ERROR: ' . $msg, 'red') . $descr . "\n";
}
