#!/usr/bin/perl

use lib::abs qw(../../lib);

use qbit;
use File::Copy;
use Net::INET6Glue::INET_is_INET6;
use HTTP::Request;
use LWP::UserAgent;

use Utils::Deploy qw(resolve_db_endpoint);

my $ROOT_DIRECTORY = '/usr/share/partners';
my $STAGE_NAME     = lc($ENV{'ENVIRONMENT'});
my $CONFIGS_PATH   = $ROOT_DIRECTORY . '/configs/' . $STAGE_NAME . '/';

main();

sub main {
    if ($ENV{USE_DEPLOY_DATABASE}) {
        generate_haproxy_config();
    }

    add_system_configs();

    copy_app_configs();

    create_links();

    generate_db_config();
}

sub generate_haproxy_config {
    my $host;
    my $ua         = LWP::UserAgent->new();
    my $pod        = $ua->request(HTTP::Request->new('GET', 'http://localhost:1/pod_attributes'),);
    my $stage_name = from_json($pod->decoded_content)->{metadata}{labels}{deploy}{stage_id};
    for my $cluster (qw(iva vla sas myt)) {
        $host //= resolve_db_endpoint($cluster, $stage_name);
    }

    my $tt = Template->new(
        ABSOLUTE  => 1,
        VARIABLES => {host => $host,},
    );

    $tt->process($CONFIGS_PATH . '/templates/haproxy-mysql-db.conf.tt2', {}, $CONFIGS_PATH . 'haproxy-mysql-db.conf',)
      or die $tt->error();
}

sub copy_app_configs {

    # Копирование файлов по схеме:
    # config name => path to copy
    my %configs = (
        'Application.json'                => $ROOT_DIRECTORY . '/lib/',
        'WebInterface.json'               => $ROOT_DIRECTORY . '/lib/',
        'haproxy-ch-db.conf'              => '/etc/partner/',
        'haproxy-mysql-db.conf'           => '/etc/partner/',
        'log4perl.conf'                   => '/etc/partner/',
        'nginx-locations.conf'            => '/etc/nginx/conf.d/partners/env-locations.conf',
        'main-logbroker-push-client.yaml' => '/etc/yandex/statbox-push-client/',
        'crossdc-lbkx-push-client.yaml'   => '/etc/yandex/statbox-push-client/',
    );

    # Копирование конфигов
    while (my ($config_name, $path) = each %configs) {
        copy($CONFIGS_PATH . $config_name, $path) or die "Copy '$config_name' failed: $!";
    }
}

sub create_links {
    # link => source
    my %links = ('/etc/mtab' => '/proc/mounts',);

    # Создание линков
    while (my ($link, $source) = each %links) {
        unless (-e $link) {
            symlink($source, $link) or die "Symlink '$link => $source' failed: $!";
        }
    }
}

sub generate_db_config {
    my $db_config = {
        "partnerdb2" => {
            "host"     => $ENV{MYSQL_HOST},
            "port"     => $ENV{MYSQL_PORT},
            "user"     => $ENV{MYSQL_USER},
            "password" => $ENV{MYSQL_PASSWORD},
        },
    };

    my $db_config_json = to_json($db_config, pretty => 1);

    writefile($ROOT_DIRECTORY . '/lib/DatabaseConfig.json', $db_config_json);
}

sub add_system_configs {
    _replace_line_in_file(
        dest   => '/etc/security/pam_env.conf',
        regexp => 'PERL5LIB',
        line   => 'PERL5LIB        DEFAULT="/usr/share/partners/local/lib/perl5"'
    );
}

sub _replace_line_in_file {
    my %opts = @_;
    my ($dst_path, $regex, $new_line) = @opts{qw( dest  regexp  line)};

    # my $cmd = sprintf q[cat %s | grep -v '%s' > $1%s.rep.tmp'', $file_path, $grep_regex;
    # `$cmd`;

    my $tmp_path = $dst_path . ".tmp.$$";
    # system("touch $tmp_path");

    open my $fh_dst, '< :encoding(UTF-8)', $dst_path or die qq[Failed to open dst file "$dst_path"];
    open my $fh_tmp, '> :encoding(UTF-8)', $tmp_path or die qq[Failed to open tmp file "$tmp_path"];

    while (my $line = <$fh_dst>) {
        if ($line =~ /$regex/) {
            last;
        }
        print $fh_tmp $line;
    }
    my $rest = do {local $/; <$fh_dst>};

    print $fh_tmp $new_line, "\n";
    print $fh_tmp $rest if $rest;

    close $fh_dst;
    close $fh_tmp;

    system("cat $tmp_path > $dst_path");
    system("rm $tmp_path");

    return 1;
}
