#!/usr/bin/perl -w

=encoding UTF-8

=head1 DESCRIPTION

    The script kills a process

=head1 USAGE

    ./bin/deploy/kill_process.pl --name=interface_fcgi

=head1 OPTIONS

    command             - Command for kill a process. Default: 'kill %d'; Example: --command='kill -QUITE %d'
    name                - Name of a process. Example: --name=interface_fcgi
    pid_file            - File with a process id. Example: --pid_file=/var/run/process.pid
    timeout_before_kill - Seconds before kill. Example: --timeout_before_kill=15
    wait                - Wait while a process shutdown. Example: --wait

=cut

use lib::abs qw(../../lib);

use qbit;

use Pod::Usage;
use Getopt::Long qw();

my $FOLDER = './nginx';

main();

sub main {
    my $args = _get_args();

    my $pid = _get_pid($args);

    unless (defined($pid)) {
        die sprintf('No such process - %s', $pid // 'undef');
    }

    sleep($args->{'timeout_before_kill'}) if defined($args->{'timeout_before_kill'});

    my $command = $args->{'command'} // 'kill %d';

    run_shell(sprintf($command, $pid));

    if ($args->{'wait'}) {
        die 'ID not found' unless defined($pid);

        while (-e "/proc/$pid" && -e "/proc/$pid/exe") {
            sleep(2);
        }
    }

    print "#END\n";
}

sub _get_pid {
    my ($args) = @_;

    my $pid;
    if (defined($args->{'pid_file'}) || (defined($args->{'name'}) && -e $FOLDER && -d $FOLDER)) {
        my $pid_file = $args->{'pid_file'} // "$FOLDER/$args->{'name'}.pid";

        if (-e $pid_file) {
            $pid = run_shell("cat $pid_file");
            chomp($pid);
        }
    } elsif (defined($args->{'name'})) {
        my $ps = run_shell("ps aux | grep workload_$args->{'name'}_start");
        ($pid) = $ps =~ /--wait (\d+)/;
    }

    return $pid;
}

sub _get_args {
    my $args = {};

    my $result = Getopt::Long::GetOptions(
        'command=s'             => \$args->{'command'},
        'name=s'                => \$args->{'name'},
        'pid_file=s'            => \$args->{'pid_file'},
        'timeout_before_kill=i' => \$args->{'timeout_before_kill'},
        'wait!'                 => \$args->{'wait'},
        'help|?|h'              => \$args->{'help'},
    );

    if (!$result || $args->{'help'} || !_got_correct_args($args)) {
        pod2usage(-verbose => 2, -noperldoc => 1);
    }

    return $args;
}

sub _got_correct_args {
    my ($args) = @_;

    unless (defined($args->{'pid_file'}) || defined($args->{'name'}) || defined($args->{'command'})) {
        print("Expected at least one of args: pid_file, name, command\n\n");

        return FALSE;
    }

    if (defined($args->{'pid_file'}) && defined($args->{'name'})) {
        print("Expected one of args: pid_file, name\n\n");

        return FALSE;
    }

    return TRUE;
}
