#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

Скрипт для просмотра DIFF логов отправки пейджа за период

=head1 USAGE

./bin/diff_bk_data.pl \
    --ticket=PI-10529  \
    --from_date='2017-09-01 12:40:47' \
    --to_date='01.09.2017 17:10:41' \
    --page_id=136342

=head1 OPTIONS

  ticket    - Номер тикета, он же имя папки куда будут складываться загруженные JSON
  from_date - Дата с которой смотреть логи отпраки
  block_id  - Дата по которую смотреть логи отпраки
  page_id   - ID площадки

=cut

# common modules
use feature 'say';
use Carp;

# project modules
use Getopt::Long qw();
use File::Slurp;
use File::Path qw(make_path);
use FindBin qw($Bin);
use Moment;
use Pod::Usage;

use lib::abs '../lib';
use qbit;
use Application;

main();

sub main {
    my ($ticket_id, $from_date, $to_date, $page_id, $force) = _get_args();

    my $app = setup();

    my $path = sprintf '%s/diff_dk_data/%s', $Bin, $ticket_id;

    my @file_paths = ();
    if (!$force && -d $path) {
        @file_paths = grep {m|/$page_id|} glob($path . '/*');
    }

    unless (@file_paths) {

        File::Path::make_path($path);

        system("rm $path/${page_id}*.json || true");

        my $months = get_months_by_date_range($from_date, $to_date);

        my $prev_file_path;
        foreach my $yyyy_mm (@$months) {

            my $query = sprintf q[
                SELECT
                    id,
                    dt,
                    login,
                    CONVERT( uncompress(params)  USING 'utf8' ) as request
                FROM
                    partner_logs.bk_edit_page_%s
                WHERE
                    page_id = %d
                    and dt between '%s' and '%s'
                ORDER BY
                    dt
            ], $yyyy_mm, $page_id, $from_date, $to_date;

            say $query;

            my $data = $app->partner_logs_db->_get_all($query);

            next unless $data && @$data;

            my $count = 0;
            foreach my $el (@$data) {

                say sprintf(
                    '%d/%d. %s MSK. Page ID=%d. Login=%s, id=%d ',
                    ++$count, $#$data + 1,
                    $el->{dt}, $page_id, $el->{login}, $el->{id}
                );

                my $el_json = to_json(from_json($el->{request})->[0], pretty => 1);

                my $dt = $el->{dt};
                $dt =~ s/[ :]/_/g;
                my $file_path = sprintf '%s/%d_%s.%d.json', $path, $page_id, $dt, $el->{id};

                write_file($file_path, {binmode => ':utf8',}, $el_json);

                push @file_paths, $file_path;
            }
        }
    }

    my $cmd   = '';
    my $count = 0;
    my $prev_file_path;
    foreach my $file_path (@file_paths) {
        if (defined $prev_file_path) {
            my $curr_cmd = sprintf 'colordiff -u  %s  %s', $prev_file_path, $file_path;
            $cmd .= sprintf q[echo -e '%d/%d. %s | less -R\n'; %s; echo -e '%s\n'; ], ++$count, $#file_paths, $curr_cmd,
              $curr_cmd, '=' x 100;
        }
        $prev_file_path = $file_path;
    }

    system sprintf '{ echo "%s"; ls -1 %s; %s} | less -R', $path, $path, $cmd;

    $app->post_run();
}

sub get_months_by_date_range {
    my ($from_date, $to_date) = @_;

    my @months = ();

    my $curr_date = Moment->new(dt => $from_date)->get_month_start();
    while ($curr_date->get_dt() le $to_date) {
        my $yyyy_mm = join '_', ($curr_date->get_dt() =~ /^(\d{4})-(\d{2})/);
        push @months, $yyyy_mm;
        $curr_date = $curr_date->plus(day => 31)->get_month_start();
    }

    return \@months;
}

sub setup {

    my $app = Application->new();
    $app->pre_run();

    $app->set_cur_user({id => 0, login => 'system-cron'});

    no strict 'refs';
    no warnings 'redefine';
    *{'QBit::Application::check_rights'} = sub {TRUE};

    return $app;
}

sub _get_args {
    my $ticket_id = 'PI-undef';
    my $from_date = undef;
    my $to_date   = undef;
    my $page_id   = undef;
    my $force     = 0;
    my $help      = 0;
    Getopt::Long::GetOptions(
        #--- Obligatory
        'ticket=s'    => \$ticket_id,
        'from_date:s' => \$from_date,
        'to_date:s'   => \$to_date,
        'page_id=i'   => \$page_id,
        'force!'      => \$force,

        #---
        'help|?|h' => \$help,
    ) or pod2usage(1);

    pod2usage(-verbose => 2, -noperldoc => 1) if $help;

    $from_date //= 'undef';
    $to_date   //= $from_date;

    die 'Wrong from_date ' . ($from_date // 'undef')
      unless $from_date && $from_date =~ /^\d{2,4}[-.]\d{2}[-.]\d{2,4}( \d{2}:\d{2}:\d{2})?$/;
    die 'Wrong from_date ' . ($to_date // 'undef')
      unless $to_date && $to_date =~ /^\d{2,4}[-.]\d{2}[-.]\d{2,4}( \d{2}:\d{2}:\d{2})?$/;
    die 'Wrong page_id ' . ($page_id // 'undef') unless $page_id && $page_id =~ /^\d+$/;

    # 12.09.2017 -> 2017-09-12
    $from_date =~ s|^(\d{2})\.(\d{2})\.(\d{4})|$3-$2-$1|;
    $to_date   =~ s|^(\d{2})\.(\d{2})\.(\d{4})|$3-$2-$1|;

    # 2017-09-12 -> 2017-09-12 00:00:00
    $from_date .= ' 00:00:00' unless $from_date =~ / \d{2}:\d{2}:\d{2}$/;
    $to_date   .= ' 00:00:00' unless $to_date   =~ / \d{2}:\d{2}:\d{2}$/;

    return ($ticket_id, $from_date, $to_date, $page_id, $force);
}
