#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

Скрипт сравнивает статистику на уровне DB

=head1 USAGE

perl bin/diff_statistics.pl

perl bin/diff_statistics.pl \
     --accessors=statistics_advnet_context_on_site_rtb,statistics_advnet_context_on_site_direct --date=2018-02-25

perl bin/diff_statistics.pl \
     --first_connection='{"password":"s3cret","user":"root","port":3478,"host":"127.0.0.1"}' \
     --second_connection='{"password":"s3cret","user":"root","port":3478,"host":"partnerdb-mysql-test2.yandex.ru"}'

=head1 OPTIONS

  accessors         - Аксессоры статистических уровней (дефолт: все уровни)
  date              - дата сравнения (дефолт: текущая дата)
  first_connection  - JSON параметры подключения к базе (дефолт: конфиг беты)
  second_connection - JSON параметры подключения к базе (дефолт: конфиг ТС)

=cut

use strict;
use warnings;

# project modules
use lib::abs '../lib';
use qbit;
use Application;

# common modules
use Pod::Usage;
use Getopt::Long qw();

use Test::More;
use Test::Differences qw(eq_or_diff);

main();

# main
sub main {

    my $app;
    {
        $app = Application->new();
        $app->pre_run();

        no strict 'refs';
        no warnings 'redefine';
        *{'QBit::Application::check_rights'} = sub {TRUE};
    }

    my ($accessors, $date, $first_db, $second_db) = _get_args();

    my %stat_accessors = map {$_ => TRUE} @$accessors;

    my $db_settings = {
        first_db  => $first_db,
        second_db => $second_db,
    };

    foreach my $accessor (sort keys(%$app)) {
        next
          unless blessed($app->{$accessor})
              && $app->$accessor->isa('Application::Model::Statistics::Product')
              && $app->$accessor->can('query') ne \&{"Application::Model::Statistics::Product::query"};

        next if %stat_accessors && !$stat_accessors{$accessor};

        my $table_name = $app->$accessor->db_table_name();

        my %pk = map {$_ => TRUE} @{$app->partner_db->$table_name->primary_key};

        my $q = $app->partner_db->query->select(
            table => $app->partner_db->$table_name,
            fields =>
              {map {$_->name => {SUM => [$_->name]}} grep {!$pk{$_->name}} @{$app->partner_db->$table_name->fields}},
            filter => {dt => $date}
        );

        my $result = {};
        foreach my $db (keys(%$db_settings)) {
            $app->set_option(partner_db => {%{$db_settings->{$db}}, database => 'partner'});
            $app->partner_db->close_dbh();

            $result->{$db} = $q->get_all();
        }

        eq_or_diff(
            to_json($result->{'first_db'},  pretty => TRUE),
            to_json($result->{'second_db'}, pretty => TRUE),
            $accessor, {context => 0}
        );
    }

    done_testing();
}

sub _get_args {

    my $accessors = '';
    my $help      = 0;

    my ($date, $first_db, $second_db);

    Getopt::Long::GetOptions(
        #--- Obligatory
        'accessors=s'         => \$accessors,
        'date=s'              => \$date,
        'first_connection=s'  => \$first_db,
        'second_connection=s' => \$second_db,

        #---
        'help|?|h' => \$help,
    ) or pod2usage(1);

    pod2usage(-verbose => 2, -noperldoc => 1) if $help;

    $date //= curdate(oformat => 'db');

    my @accessors = split(/,/, $accessors);

    if (defined($first_db)) {
        $first_db = from_json($first_db);
    } else {
        my $config = from_json(readfile(lib::abs::path('../lib/DatabaseConfig.json')));

        $first_db = $config->{'partnerdb2'};
    }

    if (defined($second_db)) {
        $second_db = from_json($second_db);
    } else {
        my $config = from_json(readfile(lib::abs::path('../configs/test/DatabaseConfig.json')));

        $second_db = $config->{'partnerdb2'};
    }

    return (\@accessors, $date, $first_db, $second_db);
}
