#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

  Скрипт для модификации DSP на блоках

=head1 USAGE

  ./bin/dsp.pl --dry-run

=head1 OPTIONS

  models            - список моделей (иначе все)
  no-models         - список исключённых моделей
  page-ids          - список пейджей
  block-ids         - список блоков
  no-god-mode
  god-mode          - обрабатка блоков в год-моде
                      не задано   - все подряд
                      god-mode    - только год-мод
                      no-god-mode - исключать год-мод
  chunk-size        - размер обрабатываемых пачек
  limit             - ограничение на суммарное количество обрабатываемых блоков
  limit-per-model   - ограничение на суммарное количество обрабатываемых блоков
                      в рамках одно модели
  percent           - процент обрабатываемых блоков
  no-resend
  resend            - флаг необходимости перепослать пейджи
  rollback          - откат по заданному файлу логов предыдущего выполнения

=cut

use lib::abs qw(
  ../lib
  ../local/lib/perl5
  );

use Clone;
use Pod::Usage;

use qbit;
use Utils::DB;
use Utils::PublicID;
use Utils::ScriptWrapper;

my $CHUNK_SIZE = 10000;
my @MODELS     = qw(
  context_on_site_content
  context_on_site_natural
  context_on_site_rtb
  internal_context_on_site_content
  internal_context_on_site_natural
  internal_context_on_site_rtb
  internal_mobile_app_rtb
  mobile_app_rtb
  ssp_context_on_site_rtb
  ssp_mobile_app_rtb
  ssp_video_an_site_rtb
  video_an_site_fullscreen
  video_an_site_inpage
  video_an_site_instream
  );

my $app;
my $opts;

sub args {
    my ($opts) = @_;
    return (
        'models=s@'         => \$opts->{models},
        'no-models=s@'      => \$opts->{no_models},
        'page-ids=s@'       => \$opts->{page_ids},
        'block-ids=s@'      => \$opts->{block_ids},
        'god-mode!'         => \$opts->{god_mode},
        'chunk-size=i'      => \$opts->{chunk_size},
        'limit=i'           => \$opts->{limit},
        'limit-per-model=i' => \$opts->{limit},
        'percent=f'         => \$opts->{percent},
        'resend!'           => \$opts->{resend},
        'rollback=s'        => \$opts->{rollback},
        'split=i'           => \$opts->{split},
    );
}

sub prepare_args {
    my ($opts) = @_;

    $opts->{chunk_size} //= $CHUNK_SIZE;
    $opts->{models}     //= \@MODELS;
    $opts->{resend}     //= TRUE;

    foreach (qw(block_ids models no_models page_ids)) {
        if ($opts->{$_}) {
            $opts->{$_} = array_uniq(split /\s*,\s*/, join ',', @{$opts->{$_}});
        }
    }

    foreach (qw(rollback)) {
        die sprintf('File "%s" does not exist', $opts->{$_}) if $opts->{$_} && !-f $opts->{$_};
    }
}

run(
    sub {
        ($app, $opts) = @_;

        if ($opts->{rollback}) {
            if (open my $f, '<', $opts->{rollback}) {
                my @data;
                while (defined(my $s = <$f>)) {
                    if ($s =~ /BEFORE\s+(.+?)$/) {
                        my $block = from_json $1;
                        push @data, $block;
                    }
                }
                close $f;
            } else {
                die sprintf('Error while open file "%s": %s', $opts->{resend}, $!);
            }
            return;
        }

        my @models;
        if ($opts->{models}) {
            @models = @{$opts->{models}};
        } else {
            @models = @{$app->product_manager->get_block_model_accessors()};
        }
        @models = @{arrays_difference(\@models, $opts->{no_models})}
          if ($opts->{no_models});
        my %pages;
        foreach my $block_model_name (sort @models) {
            my $block_model        = $app->$block_model_name;
            my $rules              = $block_model->get_dsp_rule_set();
            my $deps               = $rules->build_dependencies('default_dsps');
            my $page_id_field_name = $block_model->get_page_id_field_name();
            my $last_date          = '';
            my $iteration          = 0;
            while (TRUE) {
                $iteration++;
                print logstr(($opts->{split} ? [$opts->{cur_part_idx}] : ()),
                    'HEARTBEAT', [$iteration], $block_model_name, $last_date);
                last if $opts->{limit} && $opts->{stat}{total} && $opts->{stat}{total} >= $opts->{limit};

                # Фильтры разбиты на составляющие, чтобы скомбинироватьфильтры модели и бд
                my $filter = $app->partner_db->filter();

                $filter->and([{'MOD' => [$page_id_field_name, \$opts->{split}]}, '=', \($opts->{cur_part_idx} - 1)])
                  if ($opts->{split} && $opts->{cur_part_idx});

                # Нет такого фильтра, перенёс фильтрацию в саму обработку
                # $filter->and($block_model->get_db_filter([dsp_mode => '<>' => 'auto' ]));

                $filter->and($block_model->get_db_filter([$page_id_field_name => 'IN' => $opts->{page_ids}]))
                  if $opts->{page_ids};

                $filter->and(
                    $block_model->get_db_filter(
                        fields_to_filter(
                            [$page_id_field_name, 'id'],
                            [
                                map {
                                    my (undef, $page_id, $block_id) = split_block_public_id($_);
                                    +{$page_id_field_name => $page_id, id => $block_id,}
                                  } @{$opts->{block_ids}}
                            ],
                        )
                    )
                ) if $opts->{block_ids};

                $filter->and($block_model->get_db_filter([is_custom_bk_data => '=' => ($opts->{god_mode} ? 1 : 0)]))
                  if defined($opts->{god_mode});

                # Так быстрее, чем offset
                # Возможен небольшой нахлёст,
                # но такие блоки проигнорируются при повторной обработке.
                # Добавление более сложных проверок для уникальности
                # увеличит время выполнения и читабельность.
                $filter->and($block_model->get_db_filter([create_date => '>=' => $last_date]))
                  if $last_date;

                my $data = $block_model->get_all(
                    fields => [
                        qw(
                          create_date
                          dsp_mode
                          dsps
                          id
                          opts
                          page_id
                          public_id
                          ),
                        $page_id_field_name,
                        ($deps->{block} ? @{$deps->{block}} : ()),
                    ],
                    # Возможна ситуация, когда фильтр пустой (первая итерация и нет никаких опций)
                    ($filter->expression() ? (filter => $filter) : ()),
                    order_by => [qw(create_date)],
                    limit    => $opts->{chunk_size},
                );
                last unless scalar @$data;
                $last_date = $data->[-1]{create_date};
                my $changed_pages = set_dsp_mode_auto_for_blocks_with_default_dsps($block_model, $data);
                @pages{@$changed_pages} = ();
                last if scalar(@$data) < $opts->{chunk_size};
                last
                  if $opts->{limit_per_model}
                      && $opts->{stat}{$block_model_name}
                      && $opts->{stat}{$block_model_name} >= $opts->{limit_per_model};
                last if $opts->{limit} && $opts->{stat}{total} && $opts->{stat}{total} >= $opts->{limit};
            }
        }

        my $pages_to_resend = [
            sort {$a <=> $b}
              keys %pages
        ];
        print logstr(($opts->{split} ? [$opts->{cur_part_idx}] : ()), 'PAGES TO RESEND', $pages_to_resend);
        $app->all_pages->mark_pages_for_async_update(
            page_ids           => $pages_to_resend,
            use_separate_queue => TRUE,
        ) if !$opts->{dry_run} && $opts->{resend} && scalar @$pages_to_resend;
        print logstr(($opts->{split} ? [$opts->{cur_part_idx}] : ()), 'STAT', $opts->{stat}) if $opts->{stat};
    }
   );

sub set_dsp_mode_auto_for_blocks_with_default_dsps {
    my ($model, $data) = @_;

    my $model_name = $model->accessor();
    my @result;
    my @pages;
    foreach my $block (@$data) {
        next if $block->{dsp_mode} eq 'auto';
        next if defined $opts->{percent} && rand() > $opts->{percent};

        last
          if $opts->{limit_per_model}
              && $opts->{stat}{$model_name}
              && $opts->{stat}{$model_name} >= $opts->{limit_per_model};
        last if $opts->{limit} && $opts->{stat}{total} && $opts->{stat}{total} >= $opts->{limit};

        if (is_block_has_default_dsps($model, $block)) {
            print logstr(($opts->{split} ? [$opts->{cur_part_idx}] : ()), 'BEFORE',
                [$block->{public_id}, $block->{opts}]);
            $block->{opts}{dsp_mode} = $block->{dsp_mode} = 'auto';
            print logstr(($opts->{split} ? [$opts->{cur_part_idx}] : ()), 'AFTER',
                [$block->{public_id}, $block->{opts}]);
            $block->{opts} = to_json $block->{opts}, canonical => TRUE,;
            push @result, {map {$_ => $block->{$_}} $model->get_page_id_field_name(), qw(id opts)};
            push @pages, $block->{page_id};
            $opts->{stat}{$model_name}++;
            $opts->{stat}{total}++;
        }
    }

    try {
        $model->partner_db_table->add_multi(\@result, duplicate_update => TRUE)
          if !$opts->{dry_run} && scalar @result;
    }
    catch {
        my ($exception) = @_;

        print logstr(($opts->{split} ? [$opts->{cur_part_idx}] : ()), 'ERROR_DATA',      \@result);
        print logstr(($opts->{split} ? [$opts->{cur_part_idx}] : ()), 'ERROR_EXCEPTION', $exception);
    };

    return \@pages;
}

sub is_block_has_default_dsps {
    my ($model, $block) = @_;

    my $default_dsps = [map {$_->{id}} @{$model->get_default_dsps($block)}];
    my $block_dsps = [map {ref($_) ? $_->{dsp_id} : $_} @{$block->{dsps}}];
    my $intersection = arrays_intersection($default_dsps, $block_dsps);

    return arrays_eq($default_dsps, $block_dsps);
}

sub key_data_for_split {
    my ($opts) = @_;

    my $name  = 'fake_split_key';
    my $split = $opts->{split};
    $opts->{$name} = [1 .. $split];

    return $name;
}
