#!/usr/bin/perl

=encoding UTF-8
=cut

=head1 USAGE

Скрипт предназначен для упрощения дебага проблем ACL intapi ПИ2.

Скрипту нужно обязательно передать параметр --acl - значение которого это
строка с описание ACL. В ответ скрипт развернет этот ACL.

Пример:

    ./bin/explain_acl.pl --acl='_C_MARKET_STATGATE_ or _C_MARKET_STATGATE_TESTING_'

Скрипту так же можно указать необязательный параметр --ip - тогда в ответе
будет дополнительная информация, входит ли этот ip в ACL.

Пример:

    ./bin/explain_acl.pl --acl='_C_MARKET_STATGATE_ or _C_MARKET_STATGATE_TESTING_' --ip='2a02:6b8:0:1a1e:badb:82e5:148c:ca9d'

=cut

use Getopt::Long;
use Term::ANSIColor qw(colored);
use Pod::Usage;
use feature qw(say);

use lib::abs qw(
  ../lib
  );
use qbit;
use Application;

# main
sub main {

    $ENV{'LAZY_LOAD'} = TRUE;

    my $acl;
    my $ip;
    my $help;

    Getopt::Long::GetOptions(
        #--- Obligatory
        'acl:s' => \$acl,
        #---
        'ip:s'     => \$ip,
        'help|?|h' => \$help,
    ) or pod2usage(1);

    pod2usage(-verbose => 2, -noperldoc => 1) if $help;

    #-- Проверяем зн-ия входных параметров
    my $errors = [];
    push @$errors, 'You must specify "--acl"' unless $acl;

    if (@$errors) {
        print join("\n", map {colored('ERROR! ' . $_, 'red')} @$errors), "\n\n";
        pod2usage(-verbose => 2, -noperldoc => 1);
        exit(1);
    }

    no strict 'refs';
    no warnings 'redefine';
    # Чтобы скрипт не выводил логи вазимодействия с HBF
    *{'Utils::Logger::INFO'} = sub { };

    my $app = Application->new();
    $app->pre_run();
    $app->set_cur_user({id => 0});

    my $found;

    my @elements = $app->intapi_acl->get_elements_from_acl($acl);

    $app->set_option('check_intapi_acl', '1');

    say '';
    say sprintf 'ACL "%s" состоит из следующих элементов: ', colored($acl, 'blue');
    say '';

    foreach my $element (@elements) {
        say sprintf '## %s', $element;

        if ($app->intapi_acl->is_macros($element)) {
            say '';
            say
              'HBF раскрывает этот макрос в следующие сети/адреса/хосты:';

            my @nets_ips_and_hosts = $app->api_hbf->get_hosts_from_macros($element);

            foreach my $net_or_ip_or_host (@nets_ips_and_hosts) {

                my @subnets = $app->intapi_acl->acl2subnets($net_or_ip_or_host);

                my $addition = '';
                if ($ip && $app->intapi_acl->check_acl(\@subnets, $ip)) {
                    $addition = ' - IP адрес ' . colored($ip, 'green') . ' входит в этот ACL';
                    $found = 1;
                }

                say " * $net_or_ip_or_host $addition";
            }
        } else {
            if ($ip && $app->intapi_acl->check_acl([$app->intapi_acl->acl2subnets($element)], $ip)) {
                say 'IP адрес ' . colored($ip, 'green') . ' входит в этот ACL';
                $found = 1;
            }
        }

        say '';

    }

    if ($ip && !$found) {
        say 'IP адрес ' . colored($ip, 'red') . ' не входит в этот ACL';
        say '';
    }

    $app->post_run();
}
main();
__END__
