#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

Скрипт для расшифровки значаний поля "multistate"

=head1 USAGE

perl bin/explain_multistate.pl --accessor=context_on_site_rtb --multistate=6,38

perl bin/explain_multistate.pl --accessor=context_on_site_rtb --page_id=1037 --block_id=2

perl bin/explain_multistate.pl --accessor=context_on_site_rtb --pk_json={"id":12}

=head1 OPTIONS

  accessor   - Акксессор продуктовой модели
  multistate - статусы через запятую
  page_id    - ID площадки
  block_id   - ID блока
  noactions  - не выводит actions
  pk_json    - JSON задающий primary key

=cut

use strict;
use warnings;

# project modules
use lib::abs '../lib';
use qbit;
use Application;

# common modules
use Pod::Usage;
use Getopt::Long qw();
use JSON::XS qw();
use Time::HiRes qw(gettimeofday tv_interval);
use Term::ANSIColor qw(colored);

# global vars
my $RIGHTS;

$ENV{'LAZY_LOAD'} = TRUE;

main();

# main
sub main {

    my $t0 = [gettimeofday()];

    my $app = _get_app();

    my ($accessor, $page_id, $block_id, $multistate, $pk_obj, $with_actions) = _get_args();

    my $model_obj = $app->$accessor;

    my @multistates = split /\s*,\s*/, $multistate;

    @multistates = _get_object_multistate($model_obj, $page_id, $block_id, $pk_obj) unless @multistates;

    # Флаги
    {
        print "Bits:\n";
        printf "  %d - (%s)\n", $_, join(', ', $model_obj->get_multistate_name_as_list($_)) foreach @multistates;

        my $mask = '%-4s ' x scalar(@multistates);

        print "\n";
        printf "  %-3s  $mask %-5s  %s\n", 'bit', @multistates, 'value', 'name';
        print "  ------------------------------\n";

        my $multistate_bits = $model_obj->get_multistate_bits();

        my $count = 0;
        foreach my $bit (sort {$a <=> $b} keys %$multistate_bits) {
            my $bit_name = $multistate_bits->{$bit};

            my @has_bits = ();
            foreach my $multistate (@multistates) {
                push @has_bits, ($multistate & $bit ? '+' : '');
            }

            printf "  %-3d  $mask %-5d  %s\n", $count++, @has_bits, $bit, $bit_name;
        }
    }

    # actions
    if ($with_actions) {
        printf "\nActions:\n";

        my $model_multistates = $model_obj->get_multistates();

        my $actions_data = {};
        foreach my $multistate (@multistates) {
            my $model_multistate_actions = $model_multistates->{$multistate};

            $actions_data->{$multistate} = {
                map {$_ => $model_obj->get_action_name($_)}
                sort keys %$model_multistate_actions
            };
        }

        $Data::Dumper::Sortkeys = 1;

        $Data::Dumper::Terse = 1;
        ldump $actions_data;
    }

    $app->post_run();

    my $elapsed = sprintf '%.2f', tv_interval($t0, [gettimeofday()]);
    print "\nEND ($elapsed sec)\n";
}

sub _get_object_multistate {
    my ($model_obj, $page_id, $block_id, $pk_obj) = @_;

    my $object = {};

    unless (defined $pk_obj) {
        my $page_field_name = $model_obj->get_page_id_field_name();
        my $filter =
             $model_obj->isa('Application::Model::Page')
          || $model_obj->isa('Application::Model::Block')
          ? {$page_field_name => $page_id}
          : {$page_field_name => $page_id, id => $block_id};

        $object = $model_obj->get_all(fields => ['multistate'], filter => $filter)->[0];
    } else {
        $object = $model_obj->get($pk_obj, fields => ['multistate']);
    }

    die 'No obects found for specified parameters' unless defined $object;

    return $object->{multistate} // 0;
}

sub _get_app {

    my $app = Application->new();
    $app->pre_run();

    no strict 'refs';
    no warnings 'redefine';
    *{'QBit::Application::check_rights'} = sub {TRUE};

    return $app;
}

sub _get_args {

    my $accessor     = 0;
    my $page_id      = undef;
    my $block_id     = undef;
    my $multistate   = undef;
    my $pk_json      = undef;
    my $help         = 0;
    my $with_actions = 1;
    Getopt::Long::GetOptions(
        #--- Obligatory
        'accessor=s'   => \$accessor,
        'multistate:s' => \$multistate,
        'page_id=i'    => \$page_id,
        'block_id:i'   => \$block_id,
        'pk_json:s'    => \$pk_json,
        'actions!'     => \$with_actions,
        #---
        'help|?|h' => \$help,
    ) or pod2usage(1);

    pod2usage(-verbose => 2, -noperldoc => 1) if $help;

    my $errors   = [];
    my @page_ids = ();

    my $pk_obj = undef;
    if (defined $pk_json) {
        local $@;
        $pk_obj = eval {JSON::XS->new->utf8->allow_nonref->decode($pk_json)};
        push @$errors, $@ if $@;
    }

    #-- TODO: Проверяем зн-ия входных параметров
    if (!defined($multistate) && !defined($page_id) && !defined($pk_obj)) {
        push @$errors, 'Нужно указать либо "multistate" либо "page_id" либо "pk_json"';
    }

    if (@$errors) {
        print join("\n", map {colored('ERROR! ' . $_, 'red')} @$errors), "\n\n";
        pod2usage(-verbose => 2, -noperldoc => 1);
        exit(0);
    }

    return ($accessor, $page_id, $block_id, $multistate, $pk_obj, $with_actions);
}
