#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

    Скрипт создает матрицу рефакторинга для пейджей и для блоков
    NOTE!
    1) Для тестов, которые на некоторых моделях не получилось проверить скрпитом,
    проставить результаты в файле PI-14350_manual_checks.json
    2) Перед проверками "validation" и "api" выполнить ./bin/explain_refactoring/get_editable_fields_for_refactoring.pl

=head1 USAGE

    ./bin/explain_refactoring.pl

    ./bin/explain_refactoring.pl --accessor=internal_mobile_app_rtb

=head1 OPTIONS

  accessor - Аксессор продуктовой модели. Выводит подробную информацию по рефакторингу модели.

=cut

use Carp;
use warnings;
use strict;
use DDP;
use File::Slurp qw(read_file write_file);
use Pod::Usage;
use Getopt::Long qw();

use lib::abs qw(
  ./../lib
  );
use qbit;
use Application;

my $PAGE        = 'Page';
my $BLOCK       = 'Block';
my $TABLE_START = "#|\n";
my $TABLE_END   = "|#\n";

my $MODEL_FIELDS_PATH       = lib::abs::path("explain_refactoring/model_fields");
my $MANUAL_CHECKS_FILE_PATH = lib::abs::path("explain_refactoring/exceptions.json");
# поля, которые не нужно отдавать через API
my $FIELDS_NO_API = {
    owner_client_id            => TRUE,
    internal_campaign_id       => TRUE,
    type_bk                    => TRUE,
    bk_state_name              => TRUE,
    opts                       => TRUE,
    is_my                      => TRUE,
    cur_user_is_read_assistant => TRUE,
};

my $MODEL_ACCESSOR = '';

my $CHECKERS = {
    'get_structure_*' => {
        'check' => sub {
            my ($app, $model) = @_;
            no strict 'refs';
            if ($app->{$model->{'accessor'}}->can('get_structure_model_fields') eq
                \&QBit::Application::Model::DBManager::get_structure_model_fields)
            {
                return "NO";
            }
            return "YES";
          }
    },
    'validation' => {
        'check' => sub {
            # У каждого поля из editable_fields должно быть свойство need_check
            my ($app, $model, $manual_check) = @_;
            my $not_need_validation = $manual_check->{$model->{'accessor'}}->{'validation'} // [];
            my $editable_fields = get_data_from_file($MODEL_FIELDS_PATH . "/$model->{'accessor'}/editable_fields.json");
            unless (%$editable_fields) {
                return '-';
            }
            my $fields = $app->{$model->{'accessor'}}->get_model_fields();

            my $field_without_need_check = [];
            for my $field (keys %$editable_fields) {
                unless (exists($fields->{$field}->{'need_check'}) || in_array($field, $not_need_validation)) {
                    push(@$field_without_need_check, $field);
                }
            }
            return '100%' unless (@$field_without_need_check);
            return $field_without_need_check if ($MODEL_ACCESSOR);
            return int(100 - (scalar @$field_without_need_check) / (scalar keys %$editable_fields) * 100) . "%";
          }
    },
    'validatable_mixin' => {
        'check' => sub {
            my ($app, $model) = @_;
            if ($app->{$model->{'accessor'}}->isa('Application::Model::ValidatableMixin')) {
                return 'YES';
            }
            return "NO";
          }
    },
    'api' => {
        'check' => sub {
            # У каждого поля из available_fields должно быть свойство api = 1
            my ($app, $model) = @_;
            my $fields = $app->{$model->{'accessor'}}->get_model_fields();
            my $available_fields =
              get_data_from_file($MODEL_FIELDS_PATH . "/$model->{'accessor'}/available_fields.json");
            unless (%$available_fields) {
                return '-';
            }
            my $field_without_api = [];
            for my $field (keys %$available_fields) {
                unless (exists($fields->{$field}->{'api'}) && $fields->{$field}->{'api'} == 1
                    || exists($FIELDS_NO_API->{$field}))
                {
                    push(@$field_without_api, $field);
                }
            }
            return '100%' unless (@$field_without_api);
            return $field_without_api if ($MODEL_ACCESSOR);
            return int(100 - (scalar @$field_without_api) / (scalar keys %$available_fields) * 100) . "%";
          }
    },
    'hooks' => {
        'check' => sub {
            my ($app, $model) = @_;
            if ($app->{$model->{'accessor'}}->can('add') ne \&Application::Model::Common::add) {
                return '?';
            }
            return "YES";
          }
    },
};

my @EXTRA_ACCESSORS = qw(users);
my $ROLE_FIELDS     = {};
my $ROLES_TO_CHECK  = [];

main();

sub main {
    my $app = Application->new();
    $app->pre_run();
    $MODEL_ACCESSOR = _get_arg();

    $ROLES_TO_CHECK = [keys %Role::Tiny::INFO];
    my $manual_check     = get_data_from_file($MANUAL_CHECKS_FILE_PATH);
    my $blocks_accessors = $app->product_manager->get_block_model_names();
    my $pages_accessors  = $app->product_manager->get_page_model_accessors();
    push(@$pages_accessors, @EXTRA_ACCESSORS);

    get_role_fields($app, $ROLES_TO_CHECK, [@$blocks_accessors, @$pages_accessors]);
    clear_consumed_fields($ROLES_TO_CHECK);

    if ($MODEL_ACCESSOR) {
        my $model_mode = '';
        if (in_array($MODEL_ACCESSOR, $blocks_accessors)) {
            $model_mode = $BLOCK;
        } elsif (in_array($MODEL_ACCESSOR, $pages_accessors)) {
            $model_mode = $PAGE;
        } else {
            die "WRONG model accessor: $MODEL_ACCESSOR!\n";
        }
        my $model->{$model_mode} = [
            {
                'package'  => ref($app->$MODEL_ACCESSOR),
                'accessor' => $MODEL_ACCESSOR
            }
        ];
        check_model($app, $model_mode, $model->{$model_mode}->[0], $manual_check);
        my $model_checks = $model->{$model_mode}->[0];
        for my $key (keys %$model_checks) {
            delete $model_checks->{$key} if (!ref($model_checks->{$key}) && $model_checks->{$key} eq '-');
        }
        remove_consumed_roles($model);
        p $model;
        exit(0);
    }

    my $refactor_models = {
        $PAGE  => [map {$_ = {'package' => ref($app->$_), 'accessor' => $_}} @$pages_accessors],
        $BLOCK => [map {$_ = {'package' => ref($app->$_), 'accessor' => $_}} @$blocks_accessors],
    };

    for my $mod (keys %$refactor_models) {
        for my $model (@{$refactor_models->{$mod}}) {
            check_model($app, $mod, $model, $manual_check);
        }
    }

    remove_consumed_roles($refactor_models);
    my $non_applicable_checks = get_non_applicable_checks($refactor_models);
    save_in_json($refactor_models);
    write_to_table($refactor_models, $non_applicable_checks);
    $app->post_run();
}

sub check_model {
    my ($app, $mod, $model, $manual_check) = @_;
    my $package = $model->{'package'};
    # eval для того что бы не ловить 'Abstract method "get_product_name" must be defined'
    $model->{'description'} = eval {$package->get_product_name()} // '';

    for my $checker (keys %$CHECKERS) {
        $model->{$checker} = $CHECKERS->{$checker}->{'check'}->($app, $model, $manual_check);
        if (!ref($model->{$checker}) && $model->{$checker} eq '?') {
            $model->{$checker} = get_from_manual_check($manual_check, $model->{'accessor'}, $checker);
        }
    }

    for my $role (@$ROLES_TO_CHECK) {
        my $type = ($role =~ m/Application::Model::Role::(\w+)/)[0];
        if ($type eq $mod || $type eq 'Has') {
            $model->{$role} = check_role($app, $model->{'accessor'}, $role);
            if (!ref($model->{$role}) && $model->{$role} eq '?') {
                if (exists($manual_check->{'added_correctly_roles'}->{$role})) {
                    $model->{$role} = '-';
                } else {
                    $model->{$role} = get_from_manual_check($manual_check, $model->{'accessor'}, $role);
                }
            }
        }
    }
}

sub _get_arg {
    my $accessor = '';
    my $help     = 0;
    Getopt::Long::GetOptions(
        'accessor=s' => \$accessor,
        'help|?|h'   => \$help,
    ) or pod2usage(1);

    pod2usage(-verbose => 2, -noperldoc => 1) if $help;
    return $accessor;
}

sub get_short_name {
    my ($check_name) = @_;
    if ($check_name =~ m/Role/) {
        return ($check_name =~ m/(\w+)$/)[0];
    }
    return $check_name;
}

sub save_in_json {
    my ($refactor_models) = @_;
    my $content = eval {to_json($refactor_models, pretty => 1)};
    write_file('models.json', {binmode => ':utf8'}, \$content);
}

sub get_data_from_file {
    my ($file_path) = @_;
    my $content = eval {read_file($file_path, binmode => ':utf8')};
    my $data = eval {from_json($content)} // {};
    return $data;
}

sub get_from_manual_check {
    # получение результатов ручной проверки. файл - exceptions.json
    my ($manual_check, $model_accessor, $checker) = @_;
    if (exists($manual_check->{'checks'}->{$model_accessor}->{$checker})) {
        return $manual_check->{'checks'}->{$model_accessor}->{$checker};
    }
    return '?';
}

sub get_role_fields {
    # получение полей, которые добавляет роль
    my ($app, $roles, $models) = @_;
    for my $role (@$roles) {
        for my $model (@$models) {
            my $fields = {};
            try {
                if ($role->can('get_structure_model_fields')) {
                    no strict 'refs';
                    my $sub = \&{"$role\::get_structure_model_fields"};
                    $fields = $sub->($app->{$model});
                }
            };
            for my $field (keys %$fields) {
                $ROLE_FIELDS->{$role}->{$field} = TRUE;
            }
        }
    }
}

sub clear_consumed_fields {
    # удаление полей, которые добавлены в роль через consume другой роли
    my ($roles) = @_;
    my $roles_relations = {%Role::Tiny::APPLIED_TO};
    for my $role (@$roles) {
        my $relative_roles = $roles_relations->{$role};
        delete $relative_roles->{$role};
        for my $rel_role (keys %$relative_roles) {
            delete @{$ROLE_FIELDS->{$role}}{keys %{$ROLE_FIELDS->{$rel_role}}} if ($ROLE_FIELDS->{$rel_role});
        }
    }
}

sub check_role {
    my ($app, $model_accessor, $role) = @_;
    if (!$role->can('get_structure_model_fields')) {
        return 'YES' if ($app->{$model_accessor}->DOES($role));
        return "?";
    }
    return 'YES' if ($app->{$model_accessor}->DOES($role));

    my $role_fields  = $ROLE_FIELDS->{$role};
    my $model_fields = $app->$model_accessor->get_model_fields();
    my $count        = 0;
    for my $field_in_role (keys %$role_fields) {
        if (!exists($model_fields->{$field_in_role}) || $field_in_role eq 'product_type') {
            $count++;
        }
    }
    return "-" if ($count == keys %$role_fields);
    return "NO";
}

sub remove_consumed_roles {
    # Если к модели применена роль (общая роль), которая консюмит другие роли,
    # у "маленьких" ролей выставляется имя общей роли.
    my ($checker_results) = @_;
    my $role_relations = {%Role::Tiny::APPLIED_TO};
    for my $mod (keys %$checker_results) {
        for my $model (@{$checker_results->{$mod}}) {
            for my $role (@$ROLES_TO_CHECK) {
                my ($type) = ($role =~ m/Application::Model::Role::(\w+)/);
                if ($model->{$role} && ($type eq $mod || $type eq 'Has')) {

                    my $relative_roles = $role_relations->{$role};
                    next unless (%$relative_roles);

                    my $count_of_roles       = 0;
                    my $count_of_added_roles = 0;
                    for my $relative_role (keys %$relative_roles) {
                        if ($model->{$relative_role}) {
                            $count_of_roles++;
                            $count_of_added_roles++ if ($model->{$relative_role} eq 'YES');
                        }
                    }
                    if ($count_of_added_roles == $count_of_roles && $model->{$role} eq 'YES') {
                        for my $relative_role (keys %$relative_roles) {
                            if ($model->{$relative_role} && $role ne $relative_roles) {
                                $model->{$relative_role} = ($role =~ m/(\w+)$/)[0];
                            }
                        }
                    }
                }
            }
        }
    }
}

sub get_non_applicable_checks {
    # Получения неприменимых ролей для пейджей и блоков
    my ($refactor_models) = @_;
    my $non_applicable_checks = {};
    for my $type (keys %$refactor_models) {
        my $checkers = [keys %{$refactor_models->{$type}->[0]}];
        for my $checker (@$checkers) {
            my $dash_count           = 0;
            my $question_marks_count = 0;
            for my $model (@{$refactor_models->{$type}}) {
                if ($model->{$checker} eq '-') {
                    $dash_count++;
                } elsif ($model->{$checker} eq '?') {
                    $question_marks_count++;
                }
            }
            if (   $dash_count == scalar @{$refactor_models->{$type}}
                or $question_marks_count == scalar @{$refactor_models->{$type}})
            {
                $non_applicable_checks->{$type}->{$checker} = TRUE;
            }
        }
    }
    $non_applicable_checks->{$PAGE}->{'Application::Model::Role::Has::Block::DesignSettings'} = TRUE;
    return $non_applicable_checks;
}

sub write_to_table {
    my ($data, $non_applicable_checks) = @_;
    for my $type (keys %$data) {
        my $filename = "refactoring_matrix_for_$type.txt";
        open(my $fh, '>:encoding(utf8)', $filename) or die "'$filename' $!";

        my $fixed_columns = ['Продукт', '%', 'validation', 'api', 'get_structure_*', 'hooks'];
        my $columns = [
            sort {get_short_name($a) cmp get_short_name($b)} grep {
                     !in_array($_, [qw(package file description accessor)])
                  && !in_array($_, $fixed_columns)
                  && !exists($non_applicable_checks->{$type}->{$_})
              }
              keys %{$data->{$type}->[0]}
        ];

        push(@$fixed_columns, @$columns);
        my $columns_names = [map {get_short_name($_)} @$fixed_columns];
        # форматирование для таблицы в wiki
        print $fh $TABLE_START;
        print $fh "**||" . join("|", map sprintf("%-10s", $_), @$columns_names) . "||**\n";
        for my $result (@{$data->{$type}}) {
            my $row           = [];
            my $count         = 0;
            my $success_count = 0;
            for my $column (@$fixed_columns) {
                next if ($column eq '%');
                if ($column eq 'Продукт') {
                    push($row, $result->{'description'} . ": " . $result->{'accessor'});
                } else {
                    if (ref($result->{$column}) eq 'ARRAY') {
                        push($row, join(", ", @{$result->{$column}}));
                        $count++;
                    } else {
                        push($row, $result->{$column});
                        if ($result->{$column} =~ m/^YES|NO|(\d{1,3}%)$/) {
                            $count++;
                        }
                        if ($result->{$column} eq 'YES' || $result->{$column} eq '100%') {
                            $success_count++;
                        }
                    }
                }
            }
            splice(@$row, 1, 0, int($success_count * 100 / $count) . "%");
            print $fh "||" . join("|", map sprintf("%-10s", $_), @$row) . "||\n";
        }
        print $fh $TABLE_END;
        close $fh or die "$filename: $!";
    }
}
