#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

    Скрипт создает дамп editable_fields и available_fields для продуктовых моделей

=head1 USAGE

./bin/PI-14350_get_editable_fields_for_refactoring.pl --force

=head1 OPTIONS

    --force - перезаписать дамп

=cut

# common modules
use Carp;
use warnings;
use strict;
use File::Slurp qw(read_file write_file);
use File::Path qw(make_path);
use Getopt::Long qw();

# project modules
use lib::abs qw(
  ./../../lib
  );
use qbit;
use Application;
use Utils::Logger qw(INFOF);

my @EXTRA_ACCESSORS = qw(users);
my $STASH           = {};

main();

sub main {
    my $force = _get_arg();
    my $app   = Application->new();
    $app->pre_run();

    {
        no warnings 'redefine';
        no strict 'refs';

        *{'QBit::Application::check_rights'} = sub {TRUE};
    }
    my $yndx_developer_user_id = 1022;
    $app->set_cur_user({id => $yndx_developer_user_id});

    my $models = {
        'pages'  => $app->product_manager->get_page_model_accessors(),
        'blocks' => $app->product_manager->get_block_model_names(),
    };

    push(@{$models->{'pages'}}, @EXTRA_ACCESSORS);

    INFOF "!!!FORCE!!!" if ($force);
    for my $type (keys $models) {
        for my $model (@{$models->{$type}}) {
            INFOF "Model: %s", $model;
            my $full_path = lib::abs::path("model_fields") . "/$model";
            if (defined $full_path && -d $full_path && !$force) {
                next;
            }
            my $data;
            my $page_id_field_name;
            if ($model eq 'users') {
                $data = $app->$model->get_all(fields => ['id'],);
            } else {
                $page_id_field_name = $app->$model->get_page_id_field_name();
                $data               = $app->$model->get_all(
                    fields => ['id', $page_id_field_name],
                    filter => {multistate => 'not deleted'},
                );
            }
            my $count = scalar @$data;
            INFOF "rows count: %d", $count;
            my $offset;
            if ($count < 500) {
                $offset = 1;
            } else {
                $offset = int($count / 500);
            }
            my $i               = 0;
            my $id_from_db      = [];
            my $page_id_from_db = [];
            while ($i * $offset < $count) {
                push(@$id_from_db, $data->[$i * $offset]->{'id'});
                push(@$page_id_from_db, $data->[$i * $offset]->{$page_id_field_name}) if ($model ne 'users');
                $i++;
            }

            if (@$id_from_db) {
                my $all_fields;
                if ($model ne 'users') {
                    $all_fields = $app->$model->get_all(
                        fields => [qw(editable_fields available_fields)],
                        filter =>
                          ['AND', [['id' => 'IN' => $id_from_db], [$page_id_field_name => 'IN' => $page_id_from_db]]],
                    );
                } else {
                    $all_fields = $app->$model->get_all(
                        fields => [qw(editable_fields available_fields)],
                        filter => ['id' => 'IN' => $id_from_db],
                    );
                }
                for my $res (@$all_fields) {
                    for my $field_name (keys $res) {
                        map {$STASH->{$model}->{$field_name}->{$_} = TRUE} keys $res->{$field_name};
                    }
                }
                if (!$STASH->{$model}->{'editable_fields'}) {
                    map {$STASH->{$model}->{'editable_fields'}->{$_} = TRUE} keys $app->$model->get_add_fields();
                }
            } else {
                $STASH->{$model}->{editable_fields}  = {};
                $STASH->{$model}->{available_fields} = {};
            }
        }
    }
    save_stash();
    $app->post_run();
}

sub _get_arg {
    my $help  = 0;
    my $force = 0;
    Getopt::Long::GetOptions(
        'force'    => \$force,
        'help|?|h' => \$help,

    ) or pod2usage(1);

    pod2usage(-verbose => 2, -noperldoc => 1) if $help;
    return $force;
}

sub save_stash {
    for my $model (keys $STASH) {
        my $path = lib::abs::path("model_fields") . "/$model";
        make_path($path);
        for my $field_type (keys $STASH->{$model}) {
            my $content = eval {to_json($STASH->{$model}->{$field_type}, pretty => TRUE)};
            my $file = "$path/$field_type.json";
            write_file($file, {binmode => ':utf8'}, \$content);
        }
    }

}

sub get_file_name_from_package {
    my ($package) = @_;
    my $file_name = $package;
    $file_name =~ s/::/\//g;
    $file_name = sprintf 'lib/%s.pm', $file_name unless $file_name eq 'root';
    return $file_name;
}
