#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

Скрипт показывает поля статистики и откуда ини берутся

=head1 PARAMS

 level      - фильтрует по уровню статистики.
 type       - фильтрует по типу - [dimension;entity]
 model_type - фильтрует по типу таблицы - [stat;page;block;other;unknown]
 field      - фильтрует по имени поля
 data_only  - lfyyst без форматирования
 tsv        - TSV

=head1 USAGE

perl ./bin/explain_stat_fields.pl  --level=advnet_context_on_site_market_api  --type=dimension --field=campaign

=cut

use strict;
use warnings;

# project modules
use lib::abs '../lib';
use qbit;
use Application;

# common modules
use Pod::Usage;
use Getopt::Long qw();
use Term::ANSIColor qw(colored);
use Data::Dumper;

######
main();
######

sub main {

    my ($filter_level, $filter_type, $filter_field, $filter_model_type, $data_only, $tsv) = _get_args();

    my $app = _get_app();

    my ($stat_fields, $field_get_subs) =
      get_stat($app, $filter_level, $filter_type, $filter_field, $filter_model_type, $data_only);

    _print_data($stat_fields, $field_get_subs, $data_only, $tsv);
}

sub get_stat {
    my ($app, $filter_level, $filter_type, $filter_field, $filter_model_type, $data_only) = @_;

    my $deparse = B::Deparse->new;

    my $stat_acc = $app->product_manager->get_statistics_accessors();

    my $stat_fields = {
        # <entity_fields | dimension_fields> => {
        #     <ID> => [
        #         {...},
        #         ...
        #     ],
        #     ...
        # }
    };

    my $level_data = {};

    my $field_get_subs = {};

    foreach my $level (@$stat_acc) {
        next if $level =~ /_publisher$/;
        next if $filter_level && $level !~ /$filter_level/;

        my $stat_table_fields = {};
        unless ($app->$level->isa('Application::Model::Statistics::Hierarchy')) {
            $stat_table_fields = _get_table_fields($app, $app->$level->db_table_name);
        }

        my ($prod_accessor, $prod_table, $fields, $prod_table_fields);
        if ($app->$level->can('product')) {
            $prod_accessor     = $app->$level->product()->accessor;
            $prod_table        = $app->$prod_accessor->db_table_name();
            $fields            = $app->$prod_accessor->get_model_fields();
            $prod_table_fields = _get_table_fields($app, $prod_table);
        }

        foreach my $fields_type (sort qw( entity_fields  dimension_fields  )) {
            next if $filter_type && $fields_type !~ /^$filter_type/;

            my $stat_type_fields =
                $fields_type eq 'entity_fields'
              ? $app->$level->get_entity_fields()
              : $app->$level->get_dimension_fields('raw' => 1);

            foreach my $def (sort {$a->{id} cmp $b->{id}} @$stat_type_fields) {
                my $id = $def->{id};

                next if $filter_field && $id !~ /$filter_field/;

                my $id_fields = $stat_fields->{$fields_type}->{$id} //= [];

                my $field_prod_accessor     = $prod_accessor;
                my $field_prod_table_fields = $prod_table_fields;

                my $field_db_fields = $def->{db_fields};

                my $db_field          = $id;
                my $db_tables         = [];
                my $is_stat_field     = 0;
                my $field_get_sub_key = '';
                if ($fields_type eq 'dimension_fields') {

                    unless ($def->{model}) {
                        $is_stat_field = 1;
                        $db_tables     = $level;
                        $db_field      = $field_db_fields && @$field_db_fields ? $field_db_fields->[0] : '{UNKNOWN}';

                        $field_prod_accessor     = $level;
                        $field_prod_table_fields = $stat_table_fields;
                    } else {
                        my $prod_accessors = $def->{model} // [];
                        $prod_accessors = [$prod_accessors] unless ref($prod_accessors);

                        $db_field = $def->{caption_field} // 'caption';
                        $db_tables = [map {$app->$_->db_table_name()} @$prod_accessors];

                        $field_prod_accessor = $prod_accessors->[0];
                        my $field_prod_table = $app->$field_prod_accessor->db_table_name();
                        $field_prod_table_fields = _get_table_fields($app, $field_prod_table);
                    }
                } else {
                    $db_field = $fields->{$id}->{db_expr}
                      if $field_prod_table_fields->{$id} && $field_prod_table_fields->{$id}->{db_expr};

                    if ($prod_table_fields->{$id} || $field_prod_table_fields->{$id}->{db}) {
                        $db_tables = $prod_table;
                    } else {
                        $db_field = '{PRE_PROC}';
                        $db_tables = $prod_table // '';

                        my $sub_get = $fields->{$id}->{get};
                        if ($sub_get) {
                            local $Data::Dumper::Deparse = 1;
                            my $dump_str = Data::Dumper->Dump([$sub_get], ['xx']);

                            my ($package) = ($dump_str =~ m/^\s+package ([^;]+)/m);
                            my $key = $id . '_' . $package;

                            unless ($field_get_subs->{$key}) {
                                $dump_str =~ s/^\$xx = //;
                                $dump_str =~ s/^      //mg;
                                $dump_str =~ s/^/    /mg;
                                $dump_str =~ s/.*WARNING_BITS.*\n//;
                                $dump_str =~ s/.*use strict.*\n//;
                                $dump_str =~ s/^\s+package.*\n//m;

                                (my $path = $package) =~ s|::|/|g;
                                $dump_str =~ s/^(\s+sub \{)/$1    # ${path}.pm/g;

                                $field_get_subs->{$key} = $dump_str;
                                $field_get_sub_key = $key;
                            }
                        }
                    }
                }

                $db_tables = [$db_tables] unless ref($db_tables);

                my $type =
                  $is_stat_field
                  ? 'stat'
                  : $field_prod_accessor ? $app->$field_prod_accessor->isa('Application::Model::Page')
                      ? 'page'
                      : $app->$field_prod_accessor->isa('Application::Model::Block') ? 'block'
                    : 'other'
                  : 'unknown';

                next if $filter_model_type && $type ne $filter_model_type;

                my $db_field_json =
                  !(grep {$db_field eq $_} qw( {UNKNOWN} {PRE_PROC} ))
                  && @$db_tables == 1
                  ? to_json($field_prod_table_fields->{$db_field})
                  : '--';

                push @$id_fields,
                  {
                    type       => $fields_type,
                    level      => $level,
                    id         => $id,
                    label      => $def->{label} // $def->{title} // '',
                    stat_fk    => $def->{db_fields},
                    model_type => $type,
                    db_field   => $db_field,
                    db_field_json => $db_field_json // '',
                    db_tables     => $db_tables,
                    raw           => $def,
                    field_get_sub_key => $field_get_sub_key,
                  };
            }
        }

        # Filters
        my $filters = $app->$level->entity_filter_simple_fields();
        foreach my $bag (@$filters) {
            my $bag = [$bag] unless ref($bag) eq 'ARRAY';
            foreach my $field (@$bag) {
                my ($label, $name) = @{$field}{qw( lable  name )};

                my $type =
                    ($name =~ /^campaign/) ? 'page'
                  : ($name =~ /^block/)    ? 'block'
                  :                          'unknown';

                push @{$stat_fields->{filter_fields}->{$name}},
                  {
                    'db_field'          => '',
                    'db_field_json'     => '--',
                    'db_tables'         => [],
                    'field_get_sub_key' => '',
                    'id'                => $name,
                    'label'             => $label,
                    'level'             => $level,
                    'model_type'        => $type,
                    'raw'               => undef,
                    'stat_fk'           => undef,
                    'type'              => 'filter_fields'
                  };
            }
        }
    }

    return ($stat_fields, $field_get_subs);
}

sub _get_table_fields {
    my ($app, $table) = @_;

    my $db_fields = {
        map {
            my $field = $_;
            $_->name => {
                map {$_ => $field->{$_}}
                  grep {$_ ne 'db' && $_ ne 'table'}
                  keys %$field
              }
          } @{$app->partner_db->$table->fields()}
    };
    return $db_fields;
}

sub _print_data {
    my ($stat_fields, $field_get_subs, $data_only, $tsv) = @_;

    my @headers = qw(id  type  model_type  db_field  description  level  db_tables  stat_key_field  mysql_field);

    my $row_pattern = qq[  %-40s %-12s %-10s %-45s %-22s %-40s %-25s %-20s %s\n];

    if ($tsv) {
        print ' ' x 2, join("\t", @headers), "\n";
    } else {
        printf $row_pattern, @headers;
    }

    foreach my $fields_type (sort keys %$stat_fields) {
        print "####", uc($fields_type), "\n" unless $data_only;

        my $last_id = '';
        foreach my $id (sort keys %{$stat_fields->{$fields_type}}) {
            my $level_fields = $stat_fields->{$fields_type}->{$id};

            print "---\n" if $last_id ne $id && !$data_only;

            my $last_field_get_sub_key = 0;
            foreach my $row (sort {$a->{level} cmp $b->{level}} @$level_fields) {

                (my $stat_level = $row->{level}) =~ s/^statistics_//;

                my $label = $row->{label} // '';
                $label = $label->() if ref($label);
                $label =~ s/&nbsp;/ /g;

                my $db_tables = $row->{db_tables};

                my $db_field =
                  (grep {$row->{db_field} eq $_} qw( {UNKNOWN} {PRE_PROC} )) || @$db_tables > 1
                  ? $row->{db_field}
                  : join('.', grep {$_} $db_tables->[0], $row->{db_field});

                my $db_tables_str =
                  @$db_tables > 1
                  ? join(', ', @$db_tables)
                  : '';

                $db_field      =~ s/$row->{level}/<level>/g;
                $db_tables_str =~ s/$row->{level}/<level>/g;

                (my $type = $row->{type}) =~ s/_fields//;

                my @row_data = (
                    $id,
                    $type,
                    $row->{model_type},
                    $db_field,
                    $label,
                    $stat_level,
                    $db_tables_str,
                    join(', ', @{$row->{stat_fk} // []}),

                    $row->{db_field_json}
                );

                if ($tsv) {
                    print ' ' x 2, join("\t", @row_data), "\n";
                } else {
                    printf $row_pattern, @row_data;
                }

                print $field_get_subs->{$row->{field_get_sub_key}}, "\n"
                  if $row->{field_get_sub_key} && $row->{field_get_sub_key} ne $last_field_get_sub_key && !$data_only;
                $last_field_get_sub_key = $row->{field_get_sub_key};
            }
            $last_id = $id;
        }

        print "\n";
    }

    return 1;
}

sub _get_app {

    my $app = Application->new();
    $app->pre_run();

    $app->set_cur_user({id => 0, login => 'system-cron'});    # $stat_fields->{$fields_type}->{$id}
    my $curuser = $app->get_option('cur_user');
    $curuser->{can_view_adfox_fields} = TRUE;

    no strict 'refs';
    no warnings 'redefine';
    *{'QBit::Application::check_rights'} = sub {TRUE};

    return $app;
}

sub _get_args {

    my $level      = '';
    my $type       = '';
    my $field      = '';
    my $model_type = '';
    my $data_only  = '';
    my $tsv        = '';

    my $help = 0;
    Getopt::Long::GetOptions(
        #--- Obligatory
        'level:s'      => \$level,
        'type:s'       => \$type,
        'field:s'      => \$field,
        'model_type:s' => \$model_type,
        'data_only!'   => \$data_only,
        'tsv!'         => \$tsv,
        #---
        'help|?|h' => \$help,
    ) or pod2usage(1);

    pod2usage(-verbose => 2, -noperldoc => 1) if $help;

    my $errors = [];
    push @$errors, qq[Wront "type"] if $type && !grep {$type eq $_} qw( entity dimension  );
    push @$errors, qq[Wront "model_type"]
      if $model_type && !grep {$model_type eq $_} qw( stat   page  block  other  unknown);

    if (@$errors) {
        print join("\n", @$errors, ''), "\n";
        pod2usage(-verbose => 2, -noperldoc => 1);
        exit(0);
    }

    return ($level, $type, $field, $model_type, $data_only, $tsv);
}
