#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

Скрипт для получения информации о ПИ юзере его договорах
и его связках с Adfox, а также о юзере Adfox и его договорах и связках

=head1 USAGE

perl bin/explain_user.pl --login=vsheff

perl bin/explain_user.pl --login=vsheff  --adfox_login=xx --adfox_psswd=xx

=head1 OPTIONS

  login       - логин в ПИ
  adfox_login - логин в Adfox
  adfox_psswd - пароль Adfox логина

=cut

use strict;
use warnings;

# project modules
use lib::abs '../lib';
use qbit;
use Application;

# common modules
use Pod::Usage;
use Getopt::Long qw();
use JSON::XS qw();
use Time::HiRes qw(gettimeofday tv_interval);
use Term::ANSIColor qw(colored);

$ENV{'LAZY_LOAD'} = TRUE;

main();

sub main {

    my $t0 = [gettimeofday()];
    my $app = my $self = _get_app();

    my ($login, $adfox_login, $adfox_psswd) = _get_args();

    my $data = {};

    ####### users
    $data->{user} = $self->users->get_all(
        filter => ['login' => 'IN' => [$login]],
        fields => [qw(id login client_id)]
    )->[0];

    unless ($data->{user}) {
        printf('PI login "%s" not found', $login);
    } else {

        ####### common_offer_allowed_users
        $data->{allow} = $self->partner_db->query->select(
            table  => $self->partner_db->common_offer_allowed_users,
            fields => [qw(inn_ignore accept_date)],
            filter => ['user_id' => '=' => \$data->{user}->{id}]
        )->get_all()->[0];

        ######### get_client_id_by_uid
        $data->{balance_client_id} = $self->api_balance->get_client_id_by_uid($data->{user}->{id});

        ######### pi_contract
        $data->{pi_contract} = _get_client_contract($self, $data->{balance_client_id});

        ######### user_adfox
        $data->{pi_adfox_links} = $self->partner_db->query->select(
            table  => $self->partner_db->user_adfox,
            fields => [qw(adfox_id adfox_login)],
          )->join(
            table   => $self->partner_db->users,
            fields  => ['login'],
            join_on => [id => '=' => {user_id => $self->partner_db->user_adfox}],
            filter => {login => $login},
          )->get_all();
    }

    if ($adfox_login) {

        ######### user_adfox
        $data->{adfox_user} = $self->partner_db->query->select(
            table  => $self->partner_db->user_adfox,
            fields => [qw(user_id  adfox_id  adfox_login )],
            filter => {adfox_login => $adfox_login}
          )->join(
            table   => $self->partner_db->users,
            fields  => ['login'],
            join_on => [id => '=' => {user_id => $self->partner_db->user_adfox}],
          )->get_all();

        unless ($data->{adfox_user}) {
            printf 'Adfox "%" user not found', $adfox_login;
        } else {

            ######### api_adfox_graphq->get_billing_client_info
            $data->{adfox_billing_client_info} = $self->api_adfox_graphql->get_billing_client_info(
                adfox_login => $adfox_login,
                adfox_psw   => $adfox_psswd,
            );

            ######### adfox_contract
            if ($data->{adfox_billing_client_info}->{billingClientId}) {
                $data->{adfox_contract} = _get_client_contract(
                    $self,
                    $data->{adfox_billing_client_info}->{billingClientId},
                    $data->{adfox_billing_client_info}->{billingContractId}
                );
            }
        }
    }

    print to_json($data);
}

sub _get_app {

    my $app = Application->new();
    $app->pre_run();

    no strict 'refs';
    no warnings 'redefine';
    *{'QBit::Application::check_rights'} = sub {TRUE};

    return $app;
}

sub _get_client_contract {
    my ($self, $client_id, $contract_id) = @_;

    my $all_raw_contracts = $self->api_balance->get_partner_contracts(ClientID => $client_id);

    my $active_raw_contract;
    if ($contract_id) {
        ($active_raw_contract) = grep {$_->{Contract}{contract2_id} eq $contract_id} @$all_raw_contracts;
    } else {
        my $pi_active_contract = $self->documents->get_active_contract({client_id => $client_id});

        ($active_raw_contract) =
          grep {$_->{Contract}{external_id} eq $pi_active_contract->{Contract}{external_id}} @$all_raw_contracts;
    }

    my $contract_data = {
        Collaterals => $active_raw_contract->{Collaterals},
        Contract    => {
            dates => {
                map {$_ => $active_raw_contract->{Contract}->{$_}}
                  qw(
                  dt
                  create_dt
                  service_start_dt
                  update_dt
                  )
            },
            map {$_ => $active_raw_contract->{Contract}->{$_}}
              qw(
              client_id
              collateral_type_id
              contract2_id
              contract_type
              external_id
              is_cancelled
              memo
              payment_type
              services
              type
              ),
        },
        Person => {map {$_ => $active_raw_contract->{Person}->{$_}} qw( client_id  signer_person_name )},
    };

    return $contract_data;
}

sub _get_args {

    my $login       = 0;
    my $adfox_login = undef;
    my $adfox_psswd = undef;
    my $help        = 0;

    Getopt::Long::GetOptions(
        #--- Obligatory
        'login=s'       => \$login,
        'adfox_login:s' => \$adfox_login,
        'adfox_psswd:s' => \$adfox_psswd,
        #---
        'help|?|h' => \$help,
    ) or pod2usage(1);

    pod2usage(-verbose => 2, -noperldoc => 1) if $help;

    my $errors = [];

    push @$errors, 'Нужно указать "login"' unless ($login);

    if ($adfox_login && !defined $adfox_psswd) {
        push @$errors, 'Нужно указать пароль для пользователя Adfox';
    }

    if (@$errors) {
        print join("\n", map {colored('ERROR! ' . $_, 'red')} @$errors), "\n\n";
        pod2usage(-verbose => 2, -noperldoc => 1);
        exit(0);
    }

    return ($login, $adfox_login, $adfox_psswd);
}
