#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

Скрипт для работы с фичами пользователей

=head1 USAGE

perl bin/features.pl --list

perl bin/features.pl --stat

perl bin/features.pl --add=feature --roles=9,27

perl bin/features.pl --del=feature --roles=37

=head1 OPTIONS

  list    - список фич
  stat    - небольшая статистика по фичам
  add     - добавление фичи
  del     - удаление фичи
  roles   - список ролей для ограничения
  limit   - числовое ограничение на количество
  percent - процентное ограничение на количество

=cut

use lib::abs qw(
  ../lib
  );

use qbit;

use List::Util;

use Utils::ScriptWrapper;

my $CHUNK = 100;

my $app;
my $opts;

sub args {
    my ($opts) = @_;

    return (
        'list'      => \$opts->{list},
        'stat'      => \$opts->{stat},
        'add=s@'    => \$opts->{add},
        'del=s@'    => \$opts->{del},
        'roles=s@'  => \$opts->{roles},
        'limit=i'   => \$opts->{limit},
        'percent=f' => \$opts->{percent},
    );
}

run(
    sub {
        ($app, $opts) = @_;

        foreach (qw(add del roles)) {
            $opts->{$_} = [split /\s+,\s+/, join ',', @{$opts->{$_}}] if ($opts->{$_});
        }

        print_list()       if ($opts->{list});
        print_stat()       if ($opts->{stat});
        process_features() if ($opts->{add} || $opts->{del});
        print_line();
    }
   );

sub process_features {
    print_line();

    my $features = $app->user_features->get_all_features();

    my @unknown = grep {!$features->{$_}} map {@{$opts->{$_} // []}} qw(add del);
    if (scalar @unknown) {
        print logstr 'Unknown features in args:', [sort @{array_uniq(@unknown)}];
        return;
    }

    my $filter = $app->partner_db->filter();
    foreach (qw(add del)) {
        if ($opts->{$_}) {
            $filter->or(
                [
                    id => ($_ eq 'add' ? 'NOT IN' : 'IN') => $app->partner_db->query->select(
                        fields => [qw(user_id)],
                        filter => [feature => IN => \$opts->{$_}],
                        table  => $app->user_features->partner_db_table(),
                    ),
                ]
            );
        }
    }
    $filter->and([id => IN => get_role_id_subquery()]) if $opts->{roles};

    my $users;
    my $offset          = 0;
    my $limit           = $opts->{limit};
    my @features_to_add = @{$opts->{add} // []};
    my %features_to_del = map {$_ => TRUE} @{$opts->{del} // []};
    do {
        $users = $app->users->get_all(
            fields   => [qw(id features)],
            filter   => $filter,
            limit    => (!defined($limit) || $limit > $CHUNK ? $CHUNK : $limit),
            offset   => $offset,
            order_by => [qw(create_date id)],
        );
        foreach my $user (@$users) {
            if ($opts->{percent} && $opts->{percent} < rand(100)) {
                print logstr 'SKIP:', $user;
                $offset += 1;
                next;
            }
            print logstr 'BEFORE:', $user;
            $user->{features} = array_uniq(@features_to_add, grep {!$features_to_del{$_}} @{$user->{features}});
            try {
                $app->users->do_action($user, edit => features => $user->{features}) unless $opts->{dry_run};
                print logstr 'AFTER:', $user;
            }
            catch {
                my ($e) = @_;

                print logstr 'ERROR:', "$e";
                $offset += 1;
            };
            $limit -= 1 if defined $limit;
        }
    } while (@$users && (!defined($limit) || $limit > 0));

    print_stat() if ($opts->{stat});
}

sub print_stat {
    print_line();

    my @features = keys %{$app->user_features->get_all_features()};
    my $features_length = List::Util::max(0, map {length $_} @features);

    my $all_counts = get_count();
    my $all_length = List::Util::max(0, map {length $_} values %$all_counts);

    if ($opts->{roles}) {
        my $by_roles_counts = get_count(filter => [user_id => IN => get_role_id_subquery()],);
        my $by_roles_length = List::Util::max(0, map {length $_} values %$by_roles_counts);
        print logstr sprintf('%*s : %*s / %*s',
            $features_length, $_, $by_roles_length, (delete($by_roles_counts->{$_}) // 0),
            $all_length, (delete($all_counts->{$_}) // 0))
          foreach sort @features;
        if (scalar keys %$all_counts) {
            print_line();
            print logstr 'There are extra features in DB:', $by_roles_counts, $all_counts;
        }
    } else {
        print logstr sprintf('%*s : %*s', $features_length, $_, $all_length, (delete($all_counts->{$_}) // 0))
          foreach sort @features;
        if (scalar keys %$all_counts) {
            print_line();
            print logstr 'There are extra features in DB:', $all_counts;
        }
    }
}

sub get_count {
    my (%filter) = @_;

    my %counts = map {
        $_->{count} =~ s/(?<=\d)(?=(?:\d{3})+$)/ /g;
        $_->{feature} => $_->{count};
      } @{
        $app->partner_db->query->select(
            fields => {
                feature => '',
                count   => {COUNT => [qw(user_id)]},
            },
            table => $app->user_features->partner_db_table(),
            %filter,
          )->group_by(qw(feature))->get_all()
      };

    return \%counts;
}

sub get_role_id_subquery {
    return $app->partner_db->query->select(
        fields => [qw(user_id)],
        filter => [role_id => IN => \$opts->{roles}],
        table  => $app->partner_db->user_role,
    );
}

sub print_list {
    print_line();

    my %features = map {ref $_ ? $_->{name} : $_} %{$app->user_features->get_all_features()};
    my $features_length = List::Util::max(0, map {length $_} keys %features);
    print logstr sprintf('%*s : %s', $features_length, $_, $features{$_}) foreach sort keys %features;
}

sub print_line {
    print logstr '-' x 80;
}
