#!/usr/bin/perl

=encoding UTF-8
=cut

=head1 DESCRIPTION

    ./bin/fix_page_id.pl --page_ids=190134

Для работы скрипта нужно подключить нужную базу. Другие внешние системы подключать не нужно.

Цель скрипта - привести данные в базе ПИ2 в такое же состояние как в БК.

Скрипт идемпотентный.

L<https://wiki.yandex-team.ru/partner/w/remove-protected/>

=cut

# common modules
use feature 'say';
use Carp;

# project modules
use lib::abs qw(
  ../lib
  );
use qbit;
use Application;
use SQL::Abstract;
use Pod::Usage;
use Term::ANSIColor qw(colored);

use PiConstants qw($TECHNICAL_RTB_BLOCK_ID);

# global vars

# subs

# copy-paste из lib/Cron/Methods/System.pm
sub _get_field_names_to_ignore {
    my ($app, $model_name) = @_;

    my @names_to_ignore;

    my $model  = $app->$model_name;
    my $fields = [keys(%{$model->get_model_fields})];

    try {
        $model->get_template($fields),;
    }
    catch Exception::Validation::BadArguments with {
        my $error_message = $_[0]->{'text'};

        if ($error_message =~ /Неизвестные поля: (.*)/) {
            @names_to_ignore = split(/, /, $1);
        }

    };

    return @names_to_ignore;
}

sub run_qbit_validator_for_blocks {
    my ($app, $page_id) = @_;

    my @block_ids =
      map {$_->{id}}
      @{$app->partner_db->_get_all('select id from context_on_site_rtb where campaign_id = ?', $page_id,)};

    my $model        = $app->context_on_site_rtb();
    my $model_fields = $model->get_model_fields;
    my %pk           = map {$_ => TRUE} grep {$model_fields->{$_}{'pk'}} keys(%$model_fields);

    my @ignore_fields = _get_field_names_to_ignore($app, 'context_on_site_rtb');
    my $fields = arrays_difference([keys(%$model_fields)], \@ignore_fields);

    foreach my $block_id (@block_ids) {

        next if $block_id == $TECHNICAL_RTB_BLOCK_ID;

        my $pk_value = {
            campaign_id => $page_id,
            id          => $block_id,
        };

        my $block_data = $model->get_all(
            fields => array_uniq(@$fields, keys(%pk)),
            filter => $pk_value,
        )->[0];

        my $qv = QBit::Validator->new(
            data => $block_data,
            app  => $model,
            $model->get_template($fields, %$pk_value),
        );

        if ($qv->has_errors) {
            my @fields_with_errors = $qv->get_fields_with_error();

            #pk поля как правило проверяются при добавлении get_all_campaigns_for_adding
            #затем площадка может быть удалена или находится в другом статусе.
            my @error_fields = grep {!$pk{$_}} map {$_->{'path'}[0] // 'ERROR_IN_ROOT'} @fields_with_errors;

            if (@error_fields) {
                say "$block_id - fail:";
                say "direct_block: $block_data->{direct_block}";
                foreach my $field (@error_fields) {
                    my $field_value;
                    if (not defined $block_data->{$field}) {
                        $field_value = 'undef';
                    } elsif (ref $block_data->{$field} eq '') {
                        $field_value = $block_data->{$field};
                    } else {
                        $field_value = Dumper $block_data->{$field};
                    }
                    say "$field: $field_value";
                }
                ldump \@fields_with_errors;
                say '';
            } else {
                croak 'this should never happen';
            }
        } else {
            say "$block_id - ok";
        }

    }
}

sub _get_args {

    # Скрипт принимает параметр --page_ids, но может работать только с один Page ID
    # Название --page_ids использовано для консистентности со скриптом ./bin/resend_to_bk.pl

    my $page_id_str = '';
    my $help        = 0;

    Getopt::Long::GetOptions(
        #--- Obligatory
        'page_ids:s' => \$page_id_str,
        #---
        'help|?|h' => \$help,
    ) or pod2usage(1);

    pod2usage(-verbose => 2, -noperldoc => 1) if $help;

    my @page_ids = ();

    my $errors = [];
    if ($page_id_str) {
        @page_ids = split /,/, $page_id_str;
    } else {
        push @$errors, 'You must specify "page_ids"';
    }

    foreach my $page_id (@page_ids) {
        if ($page_id !~ /^\d+\z/) {
            push @$errors, 'Wrong page_id="' . $page_id . '"';
        }
    }

    if (@page_ids != 1) {
        push @$errors, 'Script works only with 1 Page ID';
    }

    if (@$errors) {
        print join("\n", map {colored('ERROR! ' . $_, 'red')} @$errors), "\n\n";
        pod2usage(-verbose => 2, -noperldoc => 1);
        exit(0);
    }

    return $page_ids[0];
}

sub get_campaign_table {
    my ($data) = @_;

    my $table;

    if (exists($data->{context_on_site_campaign}) && exists($data->{search_on_site_campaign})) {
        die "can't work with both tables 'context_on_site_campaign' & 'search_on_site_campaign'";
    } elsif (exists($data->{context_on_site_campaign})) {
        $table = 'context_on_site_campaign';
    } elsif (exists($data->{search_on_site_campaign})) {
        $table = 'search_on_site_campaign';
    } else {
        die "Must be info about table 'context_on_site_campaign' or 'search_on_site_campaign'";
    }

    return $table;
}

# main
sub main {
    my $app = Application->new();
    $app->pre_run();
    $app->set_app_locale('ru');

    my $sa = SQL::Abstract->new(quote_char => '`');

    $app->set_cur_user({id => 0});

    no strict 'refs';
    no warnings 'redefine';
    *{'QBit::Application::check_rights'} = sub {TRUE};

    my $page_id = _get_args();

    my $data = from_json readfile lib::abs::path("$page_id.json");

    my $campaign_table = get_campaign_table($data);

    my $campaigns = $app->partner_db->_get_all("select * from $campaign_table where page_id = ?", $page_id,);

    if (@{$campaigns} == 1) {
        my $campaign_data = $campaigns->[0];

        my $multistate_without_protected_bit = $campaign_data->{multistate} & 0b1111111101111111111;

        $app->partner_db->_do($sa->update($campaign_table, $data->{$campaign_table}, {page_id => $page_id,},));

    } else {
        $app->partner_db->transaction(
            sub {
                $data->{$campaign_table}->{id} = $app->$campaign_table->get_next_id_for_campaign();

                $app->partner_db->_do($sa->insert($campaign_table, $data->{$campaign_table},));
            }
        );

        $campaigns = $app->partner_db->_get_all("select * from $campaign_table where page_id = ?", $page_id,);
    }

    # было бы хорошо прогнать новые данные по площадке через QBit::Validator,
    # но в моделе Application::Model::Product::AN::ContextOnSite::Campaign нет сабы get_template()
    say "$campaign_table - ok";

    foreach my $table (keys %{$data}) {
        if ($table eq 'context_on_site_rtb') {
            foreach my $block_id (sort {$a <=> $b} keys %{$data->{$table}}) {
                my $block_id_number = $block_id + 0;

                say "updating $table $block_id_number";

                $app->partner_db->_do(
                    $sa->update(
                        $table,
                        $data->{$table}->{$block_id},
                        {
                            campaign_id => $page_id,
                            id          => $block_id_number,
                        },
                    )
                );
            }
        } elsif ($table eq 'media_sizes') {
            foreach my $block_id (sort {$a <=> $b} keys %{$data->{$table}}) {
                my $block_id_number = $block_id + 0;

                say "updating $table $block_id_number";

                $app->partner_db->_do('delete from media_sizes where page_id = ? and block_id = ?',
                    $page_id, $block_id_number,);

                foreach my $el (@{$data->{$table}->{$block_id}}) {
                    $app->partner_db->_do(
                        $sa->insert(
                            $table,
                            {
                                page_id  => $page_id,
                                block_id => $block_id_number,
                                type     => $el,
                            },
                        )
                    );
                }

            }
        } elsif ($table eq 'context_on_site_dsps') {
            foreach my $block_id (sort {$a <=> $b} keys %{$data->{$table}}) {
                my $block_id_number = $block_id + 0;

                say "updating $table $block_id_number";

                $app->partner_db->_do('delete from context_on_site_dsps where campaign_id = ? and block_id = ?',
                    $page_id, $block_id_number,);

                foreach my $el (@{$data->{$table}->{$block_id}}) {
                    $app->partner_db->_do(
                        $sa->insert(
                            $table,
                            {
                                campaign_id => $page_id,
                                block_id    => $block_id_number,
                                dsp_id      => $el,
                                is_deleted  => 0,
                            },
                        )
                    );
                }

            }
        } elsif ($table eq 'search_on_site_campaign_bk_language') {
            say "updating $table";

            my %bk_language2id =
              map {$_->{language} => $_->{id}} @{$app->partner_db->_get_all('select id, language from bk_language',)};

            $app->partner_db->_do(
                'delete from search_on_site_campaign_bk_language where campaign_id = ?',
                $campaigns->[0]{id},
            );

            foreach my $lang (@{$data->{search_on_site_campaign_bk_language}}) {
                $app->partner_db->_do(
                    $sa->insert(
                        $table,
                        {
                            campaign_id => $campaigns->[0]{id},
                            language_id => $bk_language2id{$lang},
                        },
                    )
                );
            }

        } elsif ($table eq 'search_on_site_campaign') {
            # ничего не делаем, так как это сделали раньше
        } else {
            die "Unknown table $table";
        }
    }

    say '';

    say 'running qbit validation';
    run_qbit_validator_for_blocks($app, $page_id);

    $app->post_run();
}
main();
__END__
