#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION
Скрипт возвращает запрос на изменение справочников в DBaaS

Как настроить yc написано тут
https://doc.yandex-team.ru/cloud/managed-clickhouse/quickstart.html

Получить токен для выполнения http запроса
yc iam create-token

=head1 USAGE

perl bin/generate_request_to_mdb.pl --test --format=api
perl bin/generate_request_to_mdb.pl --prod --format=api

=head1 OPTIONS

  test   - генерирует для тестового кластера
  prod   - генерирует для продакшен кластера
  format - Формат запроса (Доступные значения: yc/api)

=cut

use lib::abs qw(../lib);

use qbit;

use Pod::Usage;
use Getopt::Long qw();
use File::Find;

use PiSecrets;

my $CONFIG = {
    db   => 'partner',
    type => 'mysql',
    port => 3306,
};

my $API_HOST              = 'gw.db.yandex-team.ru';
my $CLUSTER_ID_PRESTABLE  = '6bb674ba-9a67-4681-a031-32771653ebe3';
my $CLUSTER_ID_PRODUCTION = 'f082a02f-55cb-4ba4-8262-e86bb97de27b';

main();

sub main {
    my ($stage, $format) = _get_args();

    $format //= '';

    my ($cluster_id, $config, $hosts);
    if ($stage eq 'test') {
        $cluster_id = $CLUSTER_ID_PRESTABLE;

        $config = get_secret('connection-to-partner2-testing-database');

        $hosts = _get_hosts("/etc/haproxy/haproxy-mysql-db-test.conf");
    } elsif ($stage eq 'prod') {
        $cluster_id = $CLUSTER_ID_PRODUCTION;

        $config = get_secret('connection-to-partner2-prod-database-ro');

        $hosts = _get_hosts("/etc/haproxy/haproxy-mysql-db-prod.conf");
    }

    $CONFIG->{$_} = $config->{$_} foreach (qw(user password));
    $CONFIG->{'replicas'} = [map {{priority => 1, host => $_}} @$hosts];

    my $body = {};
    my $dictionaries = $body->{'clickhouse'}{'dictionaries'} = [];

    find(
        {
            wanted => sub {
                return if $File::Find::fullname !~ /\.json$/;

                my $content = readfile $File::Find::fullname;

                my $dictionary = from_json($content);

                #check
                throw 'expected key ".source.table"' unless exists($dictionary->{'source'}{'table'});

                #patch
                $dictionary->{'source'} = {%{$dictionary->{'source'}}, %$CONFIG};

                push(@$dictionaries, $dictionary);
            },
            follow => 1,
        },
        lib::abs::path('../dictionaries')
    );

    if ($format eq 'yc') {
        foreach my $dictionary (@{$body->{'clickhouse'}{'dictionaries'}}) {
            _print_yc_format_dictionary($cluster_id, $dictionary);
        }
    } elsif ($format eq 'api') {
        _print_api_format_dictionary($cluster_id, $dictionaries);
    } else {
        printf "yc mdb cluster UpdateOptions --clusterId %s --databaseOptions '%s'\n\n", $cluster_id, to_json($body);
    }
}

sub _print_yc_format_dictionary {
    my ($cluster_id, $dictionary) = @_;

    printf("%s\n\n", uc($dictionary->{'name'}));

    printf("yc managed-clickhouse cluster remove-external-dictionary %s --dict-name %s\n",
        $cluster_id, $dictionary->{'name'});

    my $cmd = sprintf('yc managed-clickhouse cluster add-external-dictionary %s --dict-name %s',
        $cluster_id, $dictionary->{'name'});

    my $structure = $dictionary->{'structure'};
    if (exists($structure->{'id'})) {
        $cmd .= " --structure-id $structure->{'id'}{'name'}";
    } elsif (exists($structure->{'key'})) {
        $cmd .= _get_params_by_attributes('--structure-key', $structure->{'key'}{'attributes'});
    }

    if (exists($structure->{'attributes'})) {
        $cmd .= _get_params_by_attributes('--structure-attribute', $structure->{'attributes'});
    }

    if (my $lifetime = $dictionary->{'lifetime'}) {
        $cmd .= " --lifetime-range min=$lifetime->{'min'},max=$lifetime->{'max'}";
    }

    if (my $layout_type = $dictionary->{'layout'}{'type'}) {
        $cmd .= " --layout-type $layout_type";
    }

    my $source = $dictionary->{'source'};
    if (defined($source)) {
        if ($source->{'type'} eq 'mysql') {
            $cmd .= sprintf(" --mysql-source db=%s,table=%s,port=%d,user=%s,password=%s",
                @$source{qw(db table port user password)});

            if (my $replicas = $source->{'replicas'}) {
                foreach my $replica (@$replicas) {
                    $cmd .= sprintf(
                        " --mysql-replica host=%s,priority=%s,port=%d,user=%s,password=%s",
                        @$replica{qw(host priority)},
                        @$source{qw(port user password)}
                    );
                }
            }
        } else {
            printf("Dictionary: %s, expected type = mysql\n", $dictionary->{'name'});
            exit;
        }
    }

    print "$cmd\n\n";
}

sub _get_params_by_attributes {
    my ($param_name, $attributes) = @_;

    my $params_str = '';
    foreach my $attribute (@$attributes) {
        my %params = hash_transform($attribute, [qw(name type)], {null_value => 'null-value'});
        $params_str .= " $param_name " . join(',', map {"$_=$params{$_}"} sort keys(%params));
    }

    return return $params_str;
}

sub _print_api_format_dictionary {
    my ($cluster_id, $dictionaries) = @_;

    my $body = {update_mask => 'configSpec.clickhouse.config.dictionaries'};
    my $api_format = $body->{'configSpec'}{'clickhouse'}{'config'}{'dictionaries'} = [];

    foreach my $dictionary (@$dictionaries) {
        my $api_dictionary = {};

        foreach (qw(name layout)) {
            $api_dictionary->{$_} = $dictionary->{$_};
        }

        $api_dictionary->{'lifetimeRange'} = $dictionary->{'lifetime'};

        my $source = $dictionary->{'source'};

        if ($source->{'type'} eq 'mysql') {
            foreach (qw(db table port user password)) {
                $api_dictionary->{'mysqlSource'}{$_} = $source->{$_};
            }

            foreach (@{$source->{'replicas'}}) {
                push(
                    @{$api_dictionary->{'mysqlSource'}{'replicas'}},
                    {
                        %$_,
                        port     => $source->{'port'},
                        user     => $source->{'user'},
                        password => $source->{'password'},
                    }
                );
            }
        } else {
            printf("Dictionary: %s, expected type = mysql\n", $dictionary->{'name'});
            exit;
        }

        my $structure = $dictionary->{'structure'};

        foreach (@{$structure->{'attributes'}}) {
            $_->{'nullValue'} = delete($_->{'null_value'});

            push(@{$api_dictionary->{'structure'}{'attributes'}}, $_);
        }

        if (exists($structure->{'id'})) {
            $api_dictionary->{'structure'}{'id'} = $structure->{'id'};
        } elsif (exists($structure->{'key'})) {
            foreach (@{$structure->{'key'}{'attributes'}}) {
                $_->{'nullValue'} = delete($_->{'null_value'});

                push(@{$api_dictionary->{'structure'}{'key'}{'attributes'}}, $_);
            }
        } else {
            printf("Dictionary: %s, expected structure.id or structure.key\n", $dictionary->{'name'});
            exit;
        }

        push(@{$api_format}, $api_dictionary);
    }

    printf(
qq[curl -k -X PATCH 'https://%s/managed-clickhouse/v1/clusters/%s/' -d '%s' -H 'Content-Type: application/json' -H 'Authorization: Bearer <TOKEN>'\n\n],
        $API_HOST, $cluster_id, to_json($body));
}

sub _get_args {
    my ($stage, $test, $prod, $format, $help);

    Getopt::Long::GetOptions(
        #--- Obligatory
        'test'     => \$test,
        'prod'     => \$prod,
        'format=s' => \$format,

        #---
        'help|?|h' => \$help,
    ) or pod2usage(1);

    if ($test && $prod) {
        $help = TRUE;
    } elsif ($test) {
        $stage = 'test';
    } elsif ($prod) {
        $stage = 'prod';
    } else {
        $help = TRUE;
    }

    if ($help) {
        pod2usage(-verbose => 2, -noperldoc => 1);

        exit;
    }

    return ($stage, $format);
}

sub _get_hosts {
    my ($path) = @_;

    my $content = readfile($path);

    my @hosts = ();

    while ($content =~ /([^\s]+?\.db\.yandex\.net)\:3306/msig) {
        push(@hosts, $1);
    }

    return \@hosts;
}
