#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

Скрипт для просмотра DIFF логов отправки пейджа за период

=head1 USAGE

 ./bin/get_bk_data.pl --page_id=136342
 ./bin/get_bk_data.pl --format=dumper      --block_public_id=R-IM-132438-2
 ./bin/get_bk_data.pl --format=json_pretty --block_public_id=132438-2
 ./bin/get_bk_data.pl --dsp_id=136342

=head1 OPTIONS

  page_id         - ID площадки
  page_only       - не ходить за блоками (по дефоту 0)
  dsp_id          - DSP id
  for_logbroker   - посмотерть что уезжает в logbroker (пока тольок для DSP)
  block_public_id - ID блока
  format          - [dumper; json; json_pretty] (json by dеfault)

=cut

BEGIN {
    $ENV{LAZY_LOAD} = 1;
    $ENV{mock_http} = 0;
}

# common modules
use feature 'say';
#use Carp;

# project modules
use Getopt::Long qw();
use Pod::Usage;

use lib::abs qw( ../lib  ../t/lib ../t_lib);

use Test::Partner::Utils qw( get_bk_data );
use Utils::PublicID qw( split_block_public_id );

use qbit;

use Application;

#####
main();
#####

sub main {

    my $args = _get_args();

    my ($page_id, $block_public_id, $dsp_id) = @$args{qw( page_id  block_public_id  dsp_id )};

    my $app = setup($args->{page_only});

    my $bk_data = undef;
    if ($page_id) {

        my $object = {};
        if ($args->{page_accessor}) {
            $object = {
                id    => $page_id,
                model => $args->{page_accessor}
            };
        } else {
            $object = $app->all_pages->get_all(
                fields => ['id', 'model'],
                filter => {page_id => $page_id}
            )->[0];
            die 'page not found' unless $object;
        }

        my $accessor        = $object->{model};
        my $page_field_name = $app->$accessor->get_page_id_field_name();

        $bk_data = get_bk_data(
            $app->$accessor,
            {$page_field_name => $page_id},
            _get_bk_data_opts(blocks_from => $args->{blocks_from})
        );

        $bk_data = $args->{asis} ? $bk_data : {$app->api_bk->_get_bk_data([$bk_data])};

    } elsif ($block_public_id) {
        my $block_id = '';
        if ($block_public_id =~ /^\d+-\d+$/) {
            ($page_id, $block_id) = split(/-/, $block_public_id);
        } else {
            (undef, $page_id, $block_id) = split_block_public_id($block_public_id);
        }

        my $accessor        = $app->product_manager->get_model_data_from_public_id($block_public_id)->{accessor};
        my $page_field_name = $app->$accessor->get_page_id_field_name();

        my $data = get_bk_data($app->$accessor, {$page_field_name => $page_id, id => $block_id});

        $bk_data = $data->{$block_id};
    } elsif ($dsp_id) {
        $bk_data = get_bk_data($app->dsp, {id => $dsp_id});

        if ($args->{for_logbroker}) {
            $bk_data = $app->api_http_bk->_prepare_dsp_object_for_logbroker($bk_data);
        }
    }

    $app->post_run();

    output($bk_data, $args->{format});

    return 1;
}

sub output {
    my ($bk_data, $format) = @_;

    if ($format eq 'json_pretty') {
        say to_json($bk_data, pretty => TRUE);
    } elsif ($format eq 'json') {
        say to_json($bk_data, canonical => TRUE);
    } elsif ($format eq 'dumper') {
        no warnings;
        local $Data::Dumper::Indent     = 2;
        local $Data::Dumper::Sortkeys   = 1;
        local $Data::Dumper::Terse      = 1;
        local $Data::Dumper::Sparseseen = 1;
        print Data::Dumper::Dumper($bk_data);
    }

    return 1;
}

sub setup {
    my ($page_only) = @_;

    my $app = Application->new();
    $app->pre_run();

    $app->set_cur_user({id => 0, login => 'system-cron'});

    no strict 'refs';
    no warnings 'redefine';
    *{'QBit::Application::check_rights'} = sub {TRUE};

    if ($page_only) {
        $app->context_on_site_campaign;
        *{'Application::Model::Page::_add_blocks_data_for_bk'} = sub {1};
    }

    return $app;
}

sub _get_args {
    my $args = {format => 'json'};
    my $help = 0;
    Getopt::Long::GetOptions(
        #--- Obligatory
        'page_id=i'         => \$args->{page_id},
        'page_accessor=s'   => \$args->{page_accessor},
        'page_only!'        => \$args->{page_only},
        'block_public_id=s' => \$args->{block_public_id},
        'dsp_id=s'          => \$args->{dsp_id},
        'for_logbroker!'    => \$args->{for_logbroker},
        'format=s'          => \$args->{format},
        'blocks_from=s'     => \$args->{blocks_from},
        'asis!'             => \$args->{asis},
        #---
        'help|?|h' => \$help,
    ) or pod2usage(1);

    pod2usage(-verbose => 2, -noperldoc => 1) if $help;

    die 'You have to specify "page_id" of "block_public_id" or "dsp_id"'
      unless grep {$args->{$_}} qw(page_id  block_public_id  dsp_id);

    return $args;
}

sub _get_bk_data_opts {
    my (%opts) = @_;
    return (
        is_from_java   => 'java' eq ($opts{blocks_from} // ''),
        without_blocks => 'none' eq ($opts{blocks_from} // ''),
    );
}
