#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

  perl -Ilib bin/get_stat_for_blocks.pl --fields=partner_wo_nds --dates=2021-10-15,2021-10-17 --input=block.list-short.txt --compare=7 --output=stat_block.json
  perl -Ilib bin/get_stat_for_blocks.pl --fields=all_wo_nds,shows,hits,ctr_direct --dates=2021-10-15,2021-10-17 --input=PI-25162_fix_native_design.out.txt --compare=7

=cut

use lib::abs qw(../../lib);

use LWP::UserAgent;

use qbit;
use Pod::Usage;
use Utils::ScriptWrapper 'util';
run(\&main);

sub args {
    my ($opts) = @_;
    return (
        'input:s'   => \$opts->{input},
        'output:s'  => \$opts->{output},
        'fields:s'  => \$opts->{fields},
        'dates:s'   => \$opts->{dates},
        'compare:i' => \$opts->{compare},
        'part:i'    => \$opts->{part},
        're:s'      => \$opts->{re},
        'tsv!'      => \$opts->{tsv},
    );
}

sub prepare_args {
    my ($opts) = @_;

    $opts->{re}   //= '(\w+-\w+-\d+-\d+)';
    $opts->{part} //= 100;

    if ($opts->{input}) {
        my $re = qr/$opts->{re}/;
        my @list;
        for my $file (split /[\s,]+/, $opts->{input}) {
            open F, $file;
            while (my $str = <F>) {
                if ($str =~ $re) {
                    push @list, $1;
                }
            }
            close F;
        }
        $opts->{blocks} = \@list;
    } else {
        die "need --input=result.txt,result2,txt\n";
    }

    unless ($opts->{fields}) {
        die "need --fields=partner_wo_nds,show\n";
    }
    $opts->{fields} = [split /,/, $opts->{fields}];

    unless ($opts->{dates}) {
        die "need --dates=2021-01-01,2021-01-31\n";
    } else {
        my @d = split /,/, $opts->{dates};
        die "bad date format $opts->{dates}\n"
          unless check_date($d[0], iformat => 'db') && check_date($d[1], iformat => 'db');
        if ($opts->{compare}) {
            my $d0 = date_sub($d[0], day => $opts->{compare}, iformat => 'db', oformat => 'db');
            my $d1 = date_sub($d[1], day => $opts->{compare}, iformat => 'db', oformat => 'db');
            $d[1] = [$d[0], $d[1]];
            $d[0] = [$d0, $d1];
        }
        $opts->{dates} = \@d;
    }

    $opts->{type} //= 'main';
    $opts->{output} //= 'stat_block_' . curdate(oformat => 'date_only_numbers') . ($opts->{tsv} ? '.txt' : '.json');
}

sub main {
    my ($app, $opts) = @_;

    silent();
    my $result;
    my $total = @{$opts->{blocks}};
    my $count = 0;
    while (my @part = splice @{$opts->{blocks}}, 0, $opts->{part}) {
        $count += scalar(@part);
        print logstr('FETCH', $count, $total);
        my $data = $app->bk_statistics->get_statistics2(
            fields        => $opts->{fields},
            period        => $opts->{dates},
            stat_type     => $opts->{type},
            entity_fields => ['complex_block_id'],
            levels        => [{filter => ["AND", [["complex_block_id", "IN", \@part]]]}]
        );
        unless ($result) {
            $result = $data;
            delete $result->{totals};
        } else {
            push @{$result->{points}}, @{$data->{points}};
            $result->{total_rows} += $data->{total_rows};
        }
    }

    open F, ">", $opts->{output};
    if ($opts->{tsv}) {
        print F format_result($result, $opts);
    } else {
        print F to_json($result, pretty => TRUE);
    }
    close F;
}

sub silent {

    no warnings 'redefine';
    no strict 'refs';

    require QBit::Application::Model::API::HTTP;
    require Application::Model::API::Yandex::HTTPMOL;

    *{'QBit::Application::Model::API::HTTP::INFO'}      = sub { };
    *{'Application::Model::API::Yandex::HTTPMOL::INFO'} = sub { };
}

sub format_result {
    my ($data, $opts) = @_;
    my %idx      = map {$_ => undef} @{$opts->{blocks}};
    my @header   = ('public_id');
    my @measures = sort keys %{$data->{measures}};
    for my $m (@measures) {
        if ($opts->{compare}) {
            push @header, "was $m";
            push @header, "became $m";
            push @header, "diff $m";
            push @header, "diff_p $m";
        } else {
            push @header, $m;
        }
    }
    my $result = join("\t", @header) . "\n";
    for my $row (sort {$b->{measures}[0]{$measures[0]} <=> $a->{measures}[0]{$measures[0]}} @{$data->{points}}) {
        my @row;
        my $public_id = $row->{dimensions}{complex_block_id};
        push @row, $public_id;
        delete $idx{$public_id};
        for my $m (@measures) {
            if ($opts->{compare}) {
                my $became = $row->{measures}[0]{$m};
                my $was    = $row->{measures}[1]{$m};
                push @row, $was;
                push @row, $became;
                push @row, sprintf("%.2f", $became - $was);
                if ($was) {
                    push @row, sprintf("%.0f", ($became - $was) / $was * 100);
                } else {
                    push @row, "";
                }
            } else {
                push @row, $row->{measures}[0]{$m};
            }
        }
        $result .= join("\t", @row) . "\n";
    }

    return $result;
}
