#!/usr/bin/perl

=encoding UTF-8
=cut

=head1 DESCRIPTION

=head1 USAGE

 ./bin/inheritance.pl -main   -show_file_name
 ./bin/inheritance.pl -page   -show_mixins -short -norestapi
 ./bin/inheritance.pl -block  -show_roles  --sub=get_temlate

=head1 OPTIONS

 main           выводит иерархию общих модулей (Cron, IntAPI, WebInterface, etc)
 page           выводит иерархию Пэйджей
 block          выводит иерархию Блоков
 show_mixins    выводжит миксины
 show_roles     выводжит роли
 short          отезает общий путь у ролей и миксин
 show_file_name вместо имен пакетов, выводить пути к ним
 norestapi      не выводит RestApi:: миксины
 sub=<NAME>     рядом с названием выведет "*", если в модуле есть саба с таким именем


=cut

# common modules
use strict;
use warnings FATAL => 'all';
use feature 'say';
use utf8;
use open qw(:std :utf8);

use Data::Printer {use_prototypes => 0,};
use Carp;
use Pod::Usage;
use Getopt::Long;
use File::Slurp;
use Term::ANSIColor qw(colored);
use Tree::Simple;

use lib::abs qw(
  ../lib
  );
use Application;
use Partner2::Code;

# global vars
my $SUB;
my $IS_SHORT       = 0;
my $SHOW_MIXINS    = 0;
my $SHOW_ROLES     = 0;
my $SHOW_FILE_NAME = 0;
my $RESTAPI        = 1;

main();

sub main {

    my $args = _get_args();
    my $app  = _get_app();

    if ($args->{main}) {
        show_main_inheritance();
    } else {
        my $packages = [];
        if ($args->{page}) {
            $packages = get_page_inheritance($app);
        } elsif ($args->{block}) {
            $packages = get_block_inheritance($app);
        }
        _show_inheritance($packages);
    }

    $app->post_run();
}

sub get_block_inheritance {
    my ($app) = @_;
    my @accessors = @{$app->product_manager->get_block_model_names()};
    my @packages = map {ref($app->$_)} @accessors;
    _add_isa(\@packages);
    return \@packages;
}

sub get_page_inheritance {
    my ($app) = @_;
    my $accessors = $app->product_manager->get_page_model_accessors();
    my @packages = map {ref($app->$_)} @$accessors;
    _add_isa(\@packages);
    return \@packages;
}

sub _add_isa {
    my ($packages) = @_;

    my $except_modules = {
        map {$_ => 1}
          qw(
          Application::Model::Product
          Application::Model::ValidatableMixin
          Exporter
          )
    };

    my $extra_isa = {};
    my @stack     = @$packages;
    while (my $package = pop @stack) {
        no strict 'refs';
        my @isa = grep {$_ !~ /^(QBit|RestApi)::/ && $_ !~ /::MIXIN::/ && !$except_modules->{$_}} @{$package . '::ISA'};
        map {$extra_isa->{$_} = 1} @isa;
        push @stack, @isa;
    }
    push @$packages, sort keys %$extra_isa;

    1;
}

sub get_file_name_from_package {
    my ($package) = @_;

    my $file_name = $package;

    $file_name =~ s/::/\//g;
    $file_name = sprintf 'lib/%s.pm', $file_name unless $file_name eq 'root';

    return $file_name;
}

sub output_tree {
    my ($tree, $package_struct, $ident) = @_;

    $ident //= 1;

    my $has_sub = $SUB && $package_struct->{$tree->getNodeValue()->{package}}->{has_sub} ? 1 : 0;

    my $addition = '';

    my $types = {
        roles  => $SHOW_ROLES,
        mixins => $SHOW_MIXINS
    };
    foreach my $type (keys %$types) {
        my $items = $tree->getNodeValue()->{$type};
        if ($types->{$type} && $items && @$items) {

            my @items_strs = ();
            foreach my $item (@$items) {
                my $item_str = $item;
                my $color = $type eq 'roles' ? 'magenta' : 'cyan';
                if ($SHOW_FILE_NAME) {
                    $item_str = get_file_name_from_package($item);
                    $item_str =~ s|/([^/.]+)\.pm$|'/' . colored($1, $color). '.pm'|e;
                    $item_str =~ s|lib/Application/Model/(Role/)?|| if $IS_SHORT;
                } else {
                    $item_str =~ s/::([^:]+)$/'::' . colored($1, $color)/e;
                    $item_str =~ s/Application::Model::(Role::)?// if $IS_SHORT;
                }
                push @items_strs, $item_str;
            }
            $addition .= sprintf '%s: %s; ', uc($type), join(', ', @items_strs);
        }
    }

    my $name = $tree->getNodeValue()->{package};

    # eval для того что бы не ловить 'Abstract method "get_product_name" must be defined'
    my $descr = eval {$name->get_product_name()} // '';

    if ($SHOW_FILE_NAME) {
        $name = get_file_name_from_package($name);
    }

    if (-t STDOUT) {
        my $color = $has_sub ? 'red' : 'green';
        if ($SHOW_FILE_NAME) {
            $name =~ s|/([^/.]+)(?=\.)|'/' . colored($1, $color)|e;
        } else {
            $name =~ s/::([^:]+)$/'::' . colored($1, $color)/e;
        }
    }

    my $start = $has_sub ? '* ' : '  ';

    my $indent_str = $start . ' ' x (($ident - 1) * 2);
    my $indent2 = 95 - length($indent_str);
    say sprintf("%s %-${indent2}s %-32s %s", $indent_str, $name, $descr, $addition);

    $ident++;
    foreach my $child ($tree->getAllChildren()) {
        output_tree($child, $package_struct, $ident);
    }
    $ident--;

    return 1;
}

sub get_package_structure {
    my ($packages) = @_;

    my $package_struct = {};

    my @files = ();
    if ($packages) {
        @files = map {my $name = $_; $name =~ s|::|/|g; $INC{$name . '.pm'}} @$packages;
    } else {
        @files = get_pm_files();
    }

    foreach my $file_name (@files) {
        my $content = read_file $file_name, {binmode => ':utf8',};

        if ($content =~ /^package\s+([^ ;]+);/) {
            my $package = $1;
            my @base_packages;
            my @roles;
            if ($content =~ /use base\s+qw\((.*?)\)/ms) {
                @base_packages = grep {$_} map {s/\s+//g; $_} split /\s+/, $1;
            }
            if ($content =~ /consume\s+qw\((.*?)\)/ms) {
                @roles = grep {$_} map {s/\s+//g; $_} split /\s+/, $1;
            }

            $package_struct->{$package} = {
                base  => {map {$_ => 1} @base_packages},
                roles => {map {$_ => 1} @roles},
                ($SUB && $content =~ /sub\s+$SUB/ ? (has_sub => 1) : ()),
            };
        }
    }

    return $package_struct;
}

sub show_main_inheritance {

    my $package_struct = get_package_structure();

    my $base = 'Application';

    my $tree = Tree::Simple->new({package => $base}, Tree::Simple->ROOT);

    foreach my $package (sort keys %$package_struct) {
        if ($package_struct->{$package}{'base'}{'Application'}) {
            $tree->addChild(Tree::Simple->new({package => $package}));
        }
    }

    say '';
    output_tree($tree, $package_struct);
    say '';

}

sub _show_inheritance {
    my ($packages) = @_;

    my $package_struct = get_package_structure($packages);

    my %data;

    my $base = 'root';

    my %objects;

    my $tree = Tree::Simple->new({package => $base}, Tree::Simple->ROOT);

    my @extra_packages = qw(
      RestApi::Model
      RestApi::MultistateModel
      Application::Model::Product::Stripe
      Application::Model::Block::Content
      );

    foreach my $package (sort @$packages) {

        if (!$objects{$package}->{done}) {
            my @parent;
            my @mixins;
            foreach my $el (keys %{$package_struct->{$package}->{base}}) {
                if ($RESTAPI || $el !~ /^RestApi::/) {
                    if ($el =~ /mixin/i || grep {$el eq $_} @extra_packages) {
                        push @mixins, $el;
                    } else {
                        push @parent, $el;
                    }
                }
            }

            if (@parent > 1) {
                p $package;
                p \@parent;
                die;
            }

            $objects{$package} = Tree::Simple->new(
                {
                    package => $package,
                    parent  => $parent[0],
                    mixins  => \@mixins,
                    roles   => [keys %{$package_struct->{$package}->{roles}}],
                    done    => 1,
                }
            );

            if (@parent) {
                if (!$objects{$parent[0]}) {
                    $objects{$parent[0]} = Tree::Simple->new({package => $parent[0],});
                }
            }
        }
    }

    foreach my $package (sort keys %objects) {
        my $parent = $objects{$package}->getNodeValue()->{parent};

        if (defined($parent) && $objects{$parent}) {
            $objects{$parent}->addChild($objects{$package});
        } else {
            $tree->addChild($objects{$package});
        }
    }

    say '';
    output_tree($tree, $package_struct);
    say '';
}

sub _get_app {

    my $app = Application->new();
    $app->pre_run();

    $app->set_cur_user({id => 0, login => 'system-cron'});

    no strict 'refs';
    no warnings 'redefine';
    *{'QBit::Application::check_rights'} = sub {1};

    return $app;
}

sub _get_args {
    my %args;

    GetOptions(
        \%args,       'help|h|?',       'main',  'page',     'block', 'show_mixins',
        'show_roles', 'show_file_name', 'sub=s', 'restapi!', 'short'
    ) or die "Error. $0 is run incorrectly.\n";

    if ($args{help} || !%args) {
        pod2usage(1);
        exit;
    }

    ($SHOW_MIXINS, $SHOW_ROLES, $SUB, $SHOW_FILE_NAME, $RESTAPI, $IS_SHORT) =
      @args{qw( show_mixins  show_roles  sub  show_file_name  restapi  short)};

    $RESTAPI //= 1;

    return \%args;
}
