#!/usr/bin/perl -w

=encoding utf8

=head1 DESCRIPTION

Cкрипт принимает параметры запроса в IntAPI и выполняет его как если бы это был настоящий вызов сервиса

=head1 PARAMS

 url             - URL (обязятельно в кавычках!)
 http_method     - [GET; POST; PUT; PATCH]
 content-type    - application/json, application/xml, text/plain
 accept-language - [ru; en]
 body            - тело POST запроса

=head1 USAGE

 intapi_call --url='/intapi/partner1/user.json?&uid=0&pretty=1'

=cut

use strict;

BEGIN {
    $ENV{mock_http} = 0;
}

use lib::abs qw(
  ../lib
  ../local/lib/perl5
  ../t/lib
  ../t_lib
  );

use qbit;
use IntAPI;
use Test::Partner::Utils;
use Utils::Safe;

use PiConstants qw($AUTHORIZATION_USER_FIELDS);

use Pod::Usage;
use Getopt::Long qw();

use feature 'say';
use Carp;

####
_run();
#####

sub _run {

    confirm_connect_on_production();
    my $args = _get_args();
    my $app  = _get_app();

    my ($uri, $query_string) = split /\?/, $args->{url}, 2;
    my ($path, $method, $ext) = $uri =~ m|intapi/([\w-]+)/([\w\-]+)\.?(\w+)?|;

    if (exists $args->{tvm_source}) {
        no warnings 'redefine';
        no warnings 'once';
        no strict 'refs';
        *IntAPI::Method::get_tvm_source = sub {
            return $args->{tvm_source};
        };
    }

    my ($header, $out) = request_to_intapi(
        app             => $app,
        path            => $path,
        method          => $method,
        ext             => $ext,
        query_string    => $query_string,
        request_method  => $args->{'http_method'},
        content_type    => $args->{'content-type'},
        accept_language => $args->{'accept-language'},
        body            => $args->{'body'},
        mode            => 'full_answer',
    );
}

sub _get_app {

    my $app = IntAPI->new();

    return $app;
}

sub _get_args {

    my $args = {};
    Getopt::Long::GetOptions(
        $args,
        'url=s',
        'http_method:s',
        'content-type:s',
        'accept-language:s',
        'body:s',
        'tvm_source:s',
        #---
        'help|h|?'
    ) or pod2usage(1);

    pod2usage(-verbose => 2, -noperldoc => 1) if $args->{help} || !%$args;

    #-- Проверяем зн-ия входных параметров
    my $errors = [];
    unless ($args->{url}) {
        push @$errors, 'url is empty';
    }

    if (@$errors) {
        print join("\n", @$errors), "\n";
        pod2usage(-verbose => 2, -noperldoc => 1);
        exit(0);
    }

    return $args;
}
