#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

  Скрипт делает запрос в jsonapi локально или в хост. Описание JsonApi тут https://wiki.yandex-team.ru/partner/w/partner2-api/rest-json-api/
  Токен и куки сохраняются в файлы ~/.jsonapi_token и ~/.pi2_cookie как есть, один в один.

=head1 USAGE

  perl ./bin/jsonapi_call.pl --url=/v1/context_on_site_rtb --login=yasaint --beta_port=8031

=head1 OPTIONS

  url       - урл запроса. Пример: --url=/v1/resources
  token     - использовать токен для авторизации (~/.jsonapi_token). Пример: --token
  beta_port - порт беты. Если опция не указана, а папка содержит порт, то будет использоваться он. Пример: --beta_port=8031
  method    - метод http запроса. Пример: --method=POST
  data      - данные для POST и PATCH запросов. --data='{"data":{"attributes":{"caption":"test"},"id":"R-A-1212-12","type":"context_on_site_rtb"}}'
  login     - логин пользователя под которым запрашиваются данные. Пример: --login=yndx-blizzard
  host      - хост. Пример: --host=partner2.yandex.ru или --host=partner2.yandex.ru:443
  restapi   - запрос в web или api. Пример: --restapi
  cookie    - использовать куки для авторизации (~/.pi2_cookie). Обязательно если указана опция --restapi. Пример: --cookie
  schema    - протокол запроса. Пример: --schema=http
  pretty    - pretty вывод. Пример: --pretty
  debug     - Выводит объект response. Пример: --debug

=cut

use lib::abs qw(
  ../lib
  );

use qbit;
use LWP::UserAgent;
use Utils::ScriptWrapper 'util';

my $LOCALHOST         = '127.0.0.1';
my $TOKEN_FILE        = '~/.jsonapi_token';
my $COOKIE_FILE       = '~/.pi2_cookie';
my $X_REAL_IP         = '2a02:6b8:c1b:229e:0:4627:6207:0';
my $JSONAPI_TEST_HOST = 'jsonapi.partner2-test.yandex.com';
my $TEST_HOST         = 'partner2-test.yandex.ru';

sub args {
    my ($opts) = @_;

    return (
        'url:s'       => \$opts->{'url'},
        'token'       => \$opts->{'token'},
        'beta_port:i' => \$opts->{'beta_port'},
        'method:s'    => \$opts->{'method'},
        'data:s'      => \$opts->{'data'},
        'login:s'     => \$opts->{'login'},
        'host:s'      => \$opts->{'host'},
        'restapi'     => \$opts->{'restapi'},
        'cookie'      => \$opts->{'cookie'},
        'schema:s'    => \$opts->{'schema'},
        'pretty'      => \$opts->{'pretty'},
        'debug'       => \$opts->{'debug'},
    );
}

run(
    sub {
        my ($app, $opts) = @_;

        my $schema = $opts->{'schema'} // 'https';
        my $method = $opts->{'method'} // 'GET';
        my $url    = $opts->{'url'}    // die 'Expected --url';
        my $data   = $opts->{'data'};
        my $debug  = $opts->{'debug'};

        my $login = $opts->{'login'};

        my $pretty = $opts->{'pretty'};

        my $token;
        if ($opts->{'token'}) {
            $token = `cat $TOKEN_FILE`;
        }

        my $cookie;
        if ($opts->{'cookie'}) {
            $cookie = `cat $COOKIE_FILE`;
        }

        my $env_host = $JSONAPI_TEST_HOST;
        if ($opts->{'restapi'}) {
            $env_host = $TEST_HOST;

            unless ($opts->{'token'}) {
                $cookie = `cat $COOKIE_FILE`;
                chomp($cookie);

                unless (length($cookie)) {
                    die "Expected --cookie or check $COOKIE_FILE";
                }
            }

            $url =~ s/^\/v1/\/restapi\/v1/;
        } elsif (!$opts->{'cookie'}) {
            $token = `cat $TOKEN_FILE`;
            chomp($token);

            unless (length($token)) {
                die "Expected --token or check $TOKEN_FILE";
            }
        }

        if ($opts->{'beta_port'} && $opts->{'host'}) {
            die 'Expected only beta_port or host';
        }

        my $host;
        my $port = 400;
        if ($opts->{'beta_port'}) {
            $host = $LOCALHOST;
            $port = $port + $opts->{'beta_port'};
        } elsif (`pwd` =~ /\.(\d+)$/) {
            $host = $LOCALHOST;
            $port = $port + $1;
        }

        if ($opts->{'host'}) {
            $host     = $opts->{'host'};
            $env_host = undef;

            ($port) = $host =~ /^.*?\:(\d+)?$/;

            unless (defined($port)) {
                $port = $schema eq 'https' ? 443 : 80;
            }
        }

        $url =~ s/^\///;
        my $uri = URI->new(sprintf("%s://%s:%d/%s", $schema, $host, $port, $url));

        my $request = HTTP::Request->new($method => $uri);

        if (defined($data)) {
            $request->content($data);
        }

        my $response = get_response($request, $token, $cookie, $env_host, $login, $debug);

        my $content_type = $response->header('content-type') // '';
        printf("ANSWERED: %s\n", $content_type =~ /charset/ ? 'Java' : 'Perl');

        my $content = fix_utf($response->decoded_content);

        if ($pretty) {
            print to_json(eval {from_json($content)}, pretty => TRUE) . "\n";
        } else {
            print "$content\n";
        }
    }
   );

sub get_response {
    my ($request, $token, $cookie, $env_host, $login, $debug) = @_;

    my $env_cookie = $cookie // '';
    if (defined($login)) {
        $env_cookie .= (defined($env_cookie) ? ';' : '') . "fakelogin=$login";
    }

    my $frontend_token;
    my $response;
    my $count = 1;
    while ($count <= 3) {
        printf("Request: %d\n", $count);

        my $lwp = LWP::UserAgent->new(
            default_headers => HTTP::Headers->new(
                Content_Type      => 'application/vnd.api+json',
                Accept            => 'application/vnd.api+json',
                'Accept-Language' => 'ru-RU,ru;q=0.5',
                (
                    defined($env_host)
                    ? (
                        HOST             => $env_host,
                        'HTTP_X_REAL_IP' => $X_REAL_IP,
                      )
                    : ()
                ),
                (defined($token)          ? (Authorization              => "token $token")  : ()),
                (defined($env_cookie)     ? (Cookie                     => $env_cookie)     : ()),
                (defined($frontend_token) ? ('X-Frontend-Authorization' => $frontend_token) : ()),
            ),
            ssl_opts => {verify_hostname => 0,},
        );

        $response = $lwp->request($request);

        if ($debug) {
            utf8::decode($response->{'_content'});
            ldump($response);
        }

        if ($response->code == 401) {
            my $res = eval {from_json($response->decoded_content())};

            if (exists($res->{'errors'}[0]{'detail'}) && $res->{'errors'}[0]{'detail'} =~ /CSRF/) {
                print "WRONG CSRF\n";
                $frontend_token = $response->header('x-frontend-authorization');

                $count++;

                next;
            }
        }

        return $response;
    }
}
