package GMDiffBkData;

use qbit;
use PiConstants qw(
  $MAX_CPM $CPM_MULTIPLIER $BLOCKED_CPM_VALUE
  $MIN_CPM_STRATEGY_ID
  $MAX_REVENUE_STRATEGY_ID
  $YAN_TRAFFIC_STRATEGY_ID
  $SEPARATE_CPM_STRATEGY_ID
  );

use GMUtils;

use base qw(Exporter);
our @EXPORT_OK = qw(
  diff_bk_data
  );

my $size_check_key = '-SizesForCheck';
my $pio_check_key  = '-PageImpOptionsForCheck';

sub diff_bk_data {
    my ($model, $orig, $new) = @_;

    my @result;
    $orig = sort_dsp($orig);
    $new  = sort_dsp($new);
    $orig = sort_pio($orig);
    $new  = sort_pio($new);
    $orig = sort_sizes($orig);
    $new  = sort_sizes($new);

    compare_data(\@result, '', $orig, $new);

    my %result;
    for my $d (@result) {
        while (my ($k, $v) = each %$d) {
            push @{$result{$k}}, $v;
        }
    }

    clear_unsigned(\%result);
    clear_unsigned_by_model($model, \%result);

    clear_legacy(\%result, $model, $orig, $new);
    delete $orig->{$size_check_key};
    delete $new->{$size_check_key};
    delete $orig->{$pio_check_key};
    delete $new->{$pio_check_key};

    clear_legacy_by_model($model, \%result, $orig, $new);

    return %result ? \%result : undef;
}

sub compare_data {
    my ($result, $path, $src, $dst) = @_;

    unless (ref $src) {
        compare_as_scalar($result, $path, $src, $dst);
    } elsif (ref $src eq 'ARRAY') {
        compare_as_array($result, $path, $src, $dst);
    } elsif (ref $src eq 'HASH') {
        compare_as_hash($result, $path, $src, $dst);
    } elsif (ref $src eq 'JSON::XS::Boolean') {
        compare_as_json($result, $path, $src, $dst);
    } else {
        die "uncomparable value: '$path' ref=" . ref($src) . "\n";
    }
}

sub compare_as_hash {
    my ($result, $path, $src, $dst) = @_;

    unless (ref $dst eq 'HASH') {
        push @$result, {$path => [ref $src, ref $dst]};
    } else {
        my %src = %$src;
        my %dst = %$dst;
        for my $k (keys %src) {
            compare_data($result, "$path.$k", $src{$k}, delete $dst{$k});
        }
        for my $k (keys %dst) {
            compare_data($result, "$path.$k", $src{$k}, $dst{$k});
        }
    }
}

sub compare_as_array {
    my ($result, $path, $src, $dst) = @_;

    unless (ref $dst eq 'ARRAY') {
        push @$result, {$path => [ref $src, ref $dst]};
    } else {
        my @src = @$src;
        my @dst = @$dst;
        my $cnt = @src < @dst ? @dst : @src;
        for (my $i = 0; $i < $cnt; $i++) {
            compare_data($result, "$path.[$i]", $src[$i], $dst[$i]);
        }
    }
}

sub compare_as_scalar {
    my ($result, $path, $src, $dst) = @_;

    if (ref $dst) {
        if (ref $dst eq 'JSON::XS::Boolean') {
            unless (defined $src) {
                push @$result, {$path => [$src, $dst]};
            } elsif (
                not(   ($src eq 'true' and $dst)
                    or ($src eq 'false' and not $dst))
              )
            {
                push @$result, {$path => [$src, $dst]};
            }
        } else {
            push @$result, {$path => [$src, ref $dst]};
        }
    } elsif (defined $src xor defined $dst) {
        push @$result, {$path => [$src, $dst]};
    } elsif (defined $src and defined $dst) {
        unless ($src eq $dst) {
            push @$result, {$path => [$src, $dst]};
        }
    }
}

sub compare_as_json {
    my ($result, $path, $src, $dst) = @_;

    if (ref $dst ne 'JSON::XS::Boolean') {
        push @$result, {$path => [$src, $dst]};
    } elsif ($src != $dst) {
        push @$result, {$path => [$src, $dst]};
    }
}

sub clear_unsigned {
    my ($data) = @_;

    delete @{$data}{
        (
            '.MultiState', '.CustomBlockData.WidgetInfo.NewsJsonOptionId',
            '.BlockCaption', '.BlockModel', '.AdFoxBlock', '.campaign_id', '.application_id', '.page_id'
        )
      };
}

sub clear_unsigned_by_model {
    my ($model, $data) = @_;

    my @keys;
    unless ($model =~ /_rtb$/) {
        # для не rtb блоков поле не имеет смысла
        @keys = grep {$_ =~ /^\.RtbDesign/} keys %$data;
    } elsif ($model =~ /context_on_site_rtb$/) {
        # 0-й дизайн не настраивается
        @keys = grep {$_ =~ /^\.RtbDesign\.0/} keys %$data;
    }
    delete @{$data}{@keys};
}

sub clear_legacy {
    my ($data, $model, $orig, $new) = @_;

    for my $key (keys %$data) {
        # старые значения блокировочных данных
        if ($key =~ /^\.Brand\.\d+\.value$/) {
            if ($data->{$key}[0][1] == $BLOCKED_CPM_VALUE and $data->{$key}[0][0] >= $MAX_CPM * $CPM_MULTIPLIER) {
                delete $data->{$key};
            }
        }
        # старые значения блокировочных данных
        elsif ($key =~ /^\.Article\.-?\d+\.value$/) {
            if ($data->{$key}[0][1] >= $MAX_CPM * $CPM_MULTIPLIER and $data->{$key}[0][0] >= $MAX_CPM * $CPM_MULTIPLIER)
            {
                delete $data->{$key};
            }
        } elsif ($key =~ /^\.DSPInfo\.\[\d+\]\.CPM$/) {
            if ($data->{$key}[0][1] >= $MAX_CPM * $CPM_MULTIPLIER and $data->{$key}[0][0] >= $MAX_CPM * $CPM_MULTIPLIER)
            {
                delete $data->{$key};
            }
        }
        # ошибочные данные
        elsif ($key =~ /^\.PICategoryIAB\.\d+\./) {
            unless ($key =~ /^\.PICategoryIAB\.\d+\.(?:MediaImageReach|MediaCreativeReach|VideoCreativeReach)$/) {
                delete $data->{$key};
            }
        }
        # проверяем в ключе $pio_check_key
        elsif ($key =~ /^\.PageImpOptions\.(?:Disable|Enable)/) {
            delete $data->{$key};
        }

        # новая версия сайта
        elsif ($key =~ /^\.$pio_check_key\.turbo-desktop$/o) {
            if (   !defined $data->{$key}[0][0]
                and defined $data->{$key}[0][1]
                and $data->{$key}[0][1] == 0)
            {
                delete $data->{$key};
            }
        } elsif ($key =~ /^\.$pio_check_key\.turbo$/o) {
            if (   !defined $data->{$key}[0][0]
                and defined $data->{$key}[0][1]
                and $data->{$key}[0][1] == 0)
            {
                delete $data->{$key};
            }
        }
        # цвета в верхнем регистре
        elsif ($key =~ /^\.RtbDesign\.\d+\.design\.\w+Color$/) {
            if (    defined $data->{$key}[0][0]
                and defined $data->{$key}[0][1]
                and normalize_color($data->{$key}[0][0]) eq $data->{$key}[0][1])
            {
                delete $data->{$key};
            } elsif ($key =~ /^\.RtbDesign\.(\d+)\.design\.borderColor$/) {
                my $dn = $1;
                if (defined $data->{$key}[0][0]
                    and !defined $data->{$key}[0][1])
                {
                    my $bt = $new->{RtbDesign}{$dn}{design}{borderType};
                    if (defined $bt and $bt eq 'none') {
                        delete $data->{$key};
                    }
                }
            }
        }
        # исправления после валидации
        elsif ($key =~ /^\.RtbDesign\.\d+\.design\.borderRadius$/) {
            if (    defined $data->{$key}[0][0]
                and !defined $data->{$key}[0][1]
                and !$data->{$key}[0][0])
            {
                delete $data->{$key};
            }
        }
        # имя дизайна могло поменятся
        elsif ($key =~ /^\.RtbDesign\.\d+\.name$/) {
            delete $data->{$key};
        }
        # тип дизайна новое поле
        elsif ($key =~ /^\.RtbDesign\.\d+\.type$/) {
            if (   !defined $data->{$key}[0][0]
                and defined $data->{$key}[0][1])
            {
                delete $data->{$key};
            }
        }
        # ["1",true]
        elsif ($key =~ /^\.RtbDesign\.\d+\.design\.(?:fullscreen|interscroller)$/) {
            if (    defined $data->{$key}[0][0]
                and $data->{$key}[0][0] == 1
                and defined $data->{$key}[0][1]
                and $data->{$key}[0][1])
            {
                delete $data->{$key};
            }
        }
        # маппинг формата
        elsif ($key =~ /^\.RtbDesign\.\d+\.design\.name$/) {
            if (    defined $data->{$key}[0][0]
                and defined $data->{$key}[0][1]
                and format_mapping($data->{$key}[0][0]) eq $data->{$key}[0][1])
            {
                delete $data->{$key};
            }
        }
        # наследие копипасты
        elsif ($key =~ /^\.RtbDesign.\d+\.design\.blockId$/) {
            delete $data->{$key};
        }
        # sizes  проверяется в ключе $size_check_key
        elsif ($key =~ /^\.Sizes.\[\d+\].(?:Height|Width)$/) {
            delete $data->{$key};
        } elsif ($key =~ /^\.Sizes.\[\d+\]$/) {
            delete $data->{$key};
        } elsif ($key =~ /^\.$size_check_key\.(\d+)x(\d+)$/o) {
            my $w = $1;
            my $h = $2;
            if (    defined $orig->{Width}
                and $w == $orig->{Width}
                and defined $orig->{Height}
                and $h == $orig->{Height})
            {
                delete $data->{$key};
            }
            # заготовка для пропуска появившегося 0x0
            # elsif ($w == 0 and $h == 0
            #     and  (
            #         !defined $data->{$key}[0][0]
            #         and defined $data->{$key}[0][1]
            #         and $data->{$key}[0][1] == 1
            #     )
            # )
            # {
            #     delete $data->{$key};
            # }
        }
    }

    # [0,null] or [null,0] считается допустимым отличием
    if (my $v = $data->{'.PrivateAuction'}) {
        if (   (defined $v->[0][0] and $v->[0][0] == 0 and not defined $v->[0][1])
            or (!defined $v->[0][0] and defined $v->[0][1] and $v->[0][1] == 0))
        {
            delete $data->{'.PrivateAuction'};
        }
    }
    if (my $v = $data->{'.AltHeight'}) {
        if (defined $v->[0][0] and $v->[0][0] == 0 and not defined $v->[0][1]) {
            delete $data->{'.AltHeight'};
        }
    }
    if (my $v = $data->{'.AltWidth'}) {
        if (defined $v->[0][0] and $v->[0][0] == 0 and not defined $v->[0][1]) {
            delete $data->{'.AltWidth'};
        }
    }
    if (my $v = $data->{'.AlternativeCode'}) {
        if (defined $v->[0][0] and $v->[0][0] eq "" and not defined $v->[0][1]) {
            delete $data->{'.AlternativeCode'};
        }
    }
    if (my $v = $data->{'.OptimizeType'}) {
        # допустимое оличие отсутствие поля и 0
        if (!defined $v->[0][0] and defined $v->[0][1] and $v->[0][1] == $MIN_CPM_STRATEGY_ID) {
            delete $data->{'.OptimizeType'};
        } elsif ($model =~ /context_on_site_rtb$/
            and (defined $v->[0][0] and $v->[0][0] == $MAX_REVENUE_STRATEGY_ID)
            and (defined $v->[0][1] and $v->[0][1] == $MIN_CPM_STRATEGY_ID)
            and ($orig->{AdTypeSet} and !$orig->{AdTypeSet}{media} and !$orig->{AdTypeSet}{'media-performance'}))
        {
            # помяняли стратегию на раздельный cpm для блокировки медийки
            delete $data->{'.OptimizeType'};
        }
    }
    # допустимое оличие отсутствие поля и 0
    if (my $v = $data->{'.PremiumVideoFormat'}) {
        if (!defined $v->[0][0] and defined $v->[0][1] and $v->[0][1] == 0) {
            delete $data->{'.PremiumVideoFormat'};
        }
    }
    # допустимое оличие отсутствие поля и пустой массив
    for my $key (qw(TargetTags OrderTags)) {
        if (my $v = $data->{".$key"}) {
            if (   !defined $v->[0][0]
                and defined $v->[0][1]
                and $v->[0][1] eq 'ARRAY'
                and @{$new->{$key}} == 0)
            {
                delete $data->{".$key"};
            }
        }
    }
    # допустимое оличие отсутствие поля и пустой хэш
    for my $key (qw(AdType)) {
        if (my $v = $data->{".$key"}) {
            if (   !defined $v->[0][0]
                and defined $v->[0][1]
                and $v->[0][1] eq 'HASH'
                and scalar(keys %{$new->{$key}}) == 0)
            {
                delete $data->{".$key"};
            }
        }
    }
    # новые значения которых небыло раньше
    for my $key (('.AdTypeSet.video-motion', '.AdTypeSet.video', '.AdTypeSet.video-performance')) {
        if (my $v = $data->{$key}) {
            if (   !defined $v->[0][0]
                and defined $v->[0][1])
            {
                delete $data->{$key};
            }
        }
    }
}

sub sort_dsp {
    my ($data) = @_;

    if ($data->{DSPInfo}) {
        my @sorted = sort {$a->{DSPID} <=> $b->{DSPID}} @{$data->{DSPInfo}};
        $data->{DSPInfo} = \@sorted;
    }

    return $data;
}

sub sort_pio {
    my ($data) = @_;

    if ($data->{PageImpOptions}) {
        $data->{$pio_check_key} =
          {(map {$_ => 1} @{$data->{PageImpOptions}{Enable}}), (map {$_ => 0} @{$data->{PageImpOptions}{Disable}}),};
    }

    return $data;
}

sub sort_sizes {
    my ($data) = @_;

    if ($data->{Sizes}) {
        $data->{$size_check_key} = {map {$_->{Width} . 'x' . $_->{Height} => 1} @{$data->{Sizes}}};
    }

    return $data;
}

sub clear_legacy_by_model {
    my ($model, $data, $orig, $new) = @_;

    if ($model =~ /_natural$/) {
        # вычисляемое поле
        delete $data->{'.DirectLimit'};
    } elsif ($model =~ /context_on_site_rtb$/) {
        # вычисляемое поле
        delete $data->{'.DirectLimit'};
    }
}

1;
