package GMUtils;

use qbit;
use Application;

use base qw(Exporter);
our @EXPORT = qw(
  prepare_path
  get_opts
  get_app
  hash_to_array
  to_csv
  is_old_bk_data
  get_list_hashes
  get_models
  get_real_db_fields
  check_dsp_exists
  normalize_color
  format_mapping
  filter_brand_exists
  from_csv
  );

use Getopt::Long;
use File::Path qw(make_path remove_tree);
use Time::HiRes qw(gettimeofday);

my $app;

sub get_app {

    unless ($app) {
        $app = Application->new();

        $app->pre_run();
        $app->set_cur_user({id => 0});
        {
            no warnings 'redefine';
            *QBit::Application::check_rights = sub {1};
        }
    }

    return $app;
}

sub get_models {

    my $app          = get_app();
    my $block_models = $app->product_manager->get_block_model_names;
    my @model_with_godmode;
    for my $accessor (grep {!/^ssp/} @$block_models) {
        my $model_fields = $app->$accessor->get_model_fields;
        push @model_with_godmode, $accessor if $model_fields->{is_custom_bk_data};
    }

    return \@model_with_godmode;
}

sub get_opts {
    my (@strict) = @_;

    my %opts = (bkdata => 1);
    Getopt::Long::GetOptions(
        "progress=n"        => \$opts{progress},
        "split=n"           => \$opts{split},
        "limit=n"           => \$opts{limit},
        "root=s"            => \$opts{root},
        "source|src=s"      => \$opts{source},
        "destination|dst=s" => \$opts{destination},
        "input=s"           => \$opts{input},
        "input_list|il=s"   => \$opts{input_list},
        "input_json|ij=s"   => \$opts{input_json},
        "list=s"            => \$opts{list},
        "date=s"            => \$opts{date},
        "commit!"           => \$opts{commit},
        "validate!"         => \$opts{validate},
    );

    unless ($opts{root}) {
        die "need root";
    }

    if ($opts{input}) {
        $opts{input} = readfile($opts{input});
    }
    if ($opts{input_list}) {
        my $list = readfile($opts{input_list});
        $opts{input_list} = [map {/^(\w+-\w+-\d+-\d+)/} split /\n/, $list];
    }
    if ($opts{input_json}) {
        my $json = readfile($opts{input_json});
        $opts{input_json} = from_json($json);
    }

    if ($opts{list}) {
        $opts{list} = [split /,/, $opts{list}];
    }

    for my $key (@strict) {
        unless (defined $opts{$key}) {
            die "need $key";
        }
    }

    return \%opts;
}

sub prepare_path {
    my (@path) = @_;

    my $path = join('/', @path);
    (my $root = $path) =~ s|/[^/]+$||;
    unless (-e $root) {
        make_path($root);
    }
    if (-e $path) {
        my @time = localtime;
        my $ts   = sprintf "%04d%02d%02d-%02d%02d%02d",
          $time[5] + 1900,
          $time[4] + 1,
          $time[3],
          $time[2],
          $time[1],
          $time[0],
          ;
        rename $path, "$path.$ts.backup";
    }

    return $path;
}

sub is_old_bk_data {
    my ($data) = @_;

    for my $field (qw(RtbDesign Design)) {
        if (exists $data->{$field} and not ref $data->{$field}) {
            return 1;
        }
    }

    return;
}

sub hash_to_array {
    my ($data) = @_;

    my @result;
    for my $key (sort {$data->{$b} <=> $data->{$a}} keys %$data) {
        push @result, [$key, $data->{$key}];
    }

    return \@result;
}

sub to_csv {
    my ($header, $data) = @_;

    my $separator = "\t";
    my $out = join($separator, @$header) . "\n";
    for my $row (@$data) {
        if (@$header < @$row) {
            splice @$row, @$header;
        }
        $out .= join($separator, map {$_ // ''} @$row) . "\n";
    }
    return $out;
}

sub from_csv {
    my ($header, $filename) = @_;

    my @result;
    return \@result unless -e $filename;

    open F, $filename;
    while (my $str = <F>) {
        $str =~ s/\s+$//;
        my %row;
        @row{@$header} = split /\t/, $str;
        push @result, \%row;
    }
    close F;

    return \@result;
}

sub get_list_hashes {
    my ($opts, $filename) = @_;

    return [] unless -e "$opts->{root}/$filename";
    open F, "$opts->{root}/$filename";
    my $data = [];
    my $part;
    while (my $str = <F>) {
        if ($part) {
            $part .= $str;
            if ($str =~ /^}/) {
                push @$data, from_json($part);
                undef $part;
            }
        } elsif ($str =~ /^{/) {
            $part = $str;
        }
    }
    close F;
    if ($part) {
        warn "part has data\n";
    }

    return $data;
}

my %real_db_fields;

sub get_real_db_fields {
    my ($app, $accessor) = @_;

    unless ($real_db_fields{$accessor}) {
        $real_db_fields{$accessor} =
          {map {$_->name => 1} grep {!$_->{generated}} @{$app->partner_db->$accessor->fields}};
    }

    return $real_db_fields{$accessor};
}

my $exists_dsp;

sub check_dsp_exists {
    my ($app, $dsp) = @_;

    unless ($exists_dsp) {
        $exists_dsp = {};
        for my $row (@{$app->dsp->get_all(fields => ['id'], filter => {multistate => 'not deleted'})}) {
            $exists_dsp->{$row->{id}} = 1;
        }
    }

    return $exists_dsp->{$dsp};
}

sub normalize_color {
    my ($color) = @_;

    $color =~ s/^#//;
    $color .= "0" x (6 - length($color));

    return uc($color);
}

my $format_mapping = {
    # PI-17516
    "160x450"        => "modernAdaptive",
    "180x150"        => "modernAdaptive",
    "234x60"         => "modernAdaptive",
    "468x60"         => "modernAdaptive",
    "600x60"         => "modernAdaptive",
    "auto"           => "modernAdaptive",
    "flat"           => "modernAdaptive",
    "flat, 300x300," => "300x300",
    "Horizontal"     => "horizontal",
    "mediaTouch"     => "modernAdaptive",
    "mobile"         => "modernAdaptive",
    # PI-17738
    "newPosterVertical"   => "posterVertical",
    "newPosterHorizontal" => "posterHorizontal",
    "oldHorizontal"       => "horizontal",
    "horizontal0318"      => "horizontal",
    "vertical0318"        => "vertical",
    # PI-19972
    "adaptive0418" => "modernAdaptive",
};

sub format_mapping {
    my ($name) = @_;
    return $format_mapping->{$name} // $name;
}

sub filter_brand_exists {
    my ($app, $list) = @_;

    my @need;
    if (ref $list eq 'ARRAY') {
        @need = map {$_->{bid}} @$list;
    } elsif (ref $list eq 'HASH') {
        @need = keys %$list;
    } else {
        die "invalid call filter_brand_exists(" . ref($list) . ")\n";
    }
    my $data = $app->partner_db->tns_dict_brand->get_all(fields => ['bid'], filter => [bid => 'IN' => \\@need]);
    my %has;
    @has{map {$_->{bid}} @$data} = ();
    if (ref $list eq 'ARRAY') {
        return [grep {exists $has{$_->{bid}}} @$list];
    } else {
        for my $bid (@need) {
            delete $list->{$bid} unless exists $has{$bid};
        }
        return $list;
    }
}

1;
