#!/usr/bin/perl

=head1 пример вызова

   # получаем данные и работаем по ним
   bin/oneshots/GM/fix_godmod.pl --root=/home/ie2018/tmp/fix_bk_data
   # получаем данные и работаем по ним, выполняем валидацию
   bin/oneshots/GM/fix_godmod.pl --root=/home/ie2018/tmp/fix_bk_data --validate
   # получаем данные и работаем по ним, выполняем валидацию, применяем изменения
   bin/oneshots/GM/fix_godmod.pl --root=/home/ie2018/tmp/fix_bk_data --validate --commit
   # получаем данные из БД и после сохранения останавливаемся
   bin/oneshots/GM/fix_godmod.pl --root=/home/ie2018/tmp/fix_bk_data --destination=result.json
   # получаем данные из файла ы выполняем работу только для блока R-A-19417-10(список через запятую)
   bin/oneshots/GM/fix_godmod.pl --root=/home/ie2018/tmp/fix_bk_data --source=result.json --list=R-A-19417-10
   #  получаем данные из файла ы выполняем работу только для блоков в файле wo_iab.txt(один блок на строку)
   bin/oneshots/GM/fix_godmod.pl --root=/home/ie2018/tmp/fix_bk_data.step04 --validate --source=result.json --list=R-VI-195290-1
   bin/oneshots/GM/fix_godmod.pl --root=/home/ie2018/tmp/fix_bk_data.step07 --validate --source=result.json --list=R-I-239082-31

   bin/oneshots/GM/fix_godmod.pl --root=/home/ie2018/tmp/fix_bk_data.step07 --validate --source=result.json --input_list=list_cbd.txt --split=10 --progress=10
   bin/oneshots/GM/fix_godmod.pl --root=/home/ie2018/tmp/fix_bk_data.PI-19674 --validate --source=result.json --split=3 --progress=100

=cut

use lib::abs qw(../../../lib .);

use Time::HiRes qw(gettimeofday);

use qbit;

use GMUtils;
use GMHasDiff;
use GMFixBkData qw(fix_bk_data);
use GMFixBlockData qw(fix_block_data);
use GMDiffBkData qw(diff_bk_data);

my $file_pages;
my $file_fblock;
my $file_fdata;
my $file_diff;
my $file_error;
my $file_check;

main();

sub main {
    my $opts = get_opts();

    unless ($opts->{destination}) {
        $file_pages  = prepare_path($opts->{root}, 'pages.txt');
        $file_fblock = prepare_path($opts->{root}, 'fixed_block.txt');
        $file_fdata  = prepare_path($opts->{root}, 'fixed_data.txt');
        $file_diff   = prepare_path($opts->{root}, 'has_diff.txt');
        $file_error  = prepare_path($opts->{root}, 'errors.txt');
        $file_check  = prepare_path($opts->{root}, 'validation.txt');
    }

    my $result;
    if ($opts->{source}) {
        $result = from_json(readfile("$opts->{root}/$opts->{source}"));
        if (my $block_list = $opts->{list} || $opts->{input_list}) {
            my %filter = map {$_ => 1} @$block_list;
            my @result = grep {$filter{$_->{public_id}}} @$result;
            $result = \@result;
        }

        if (my $split = $opts->{split}) {
            my $size = @$result / $split;
            unless ($size == int($size)) {
                $size = int($size) + 1;
            }
            for my $job (0 .. $split - 1) {
                unless (fork) {
                    $opts->{job} = $job;
                    my @part = splice @$result, $job * $size, $size;
                    process_data($opts, \@part);
                    return;
                }
            }
            1 while wait != -1;
            return;
        }
    } else {
        my $data = get_custom_bk_data($opts);

        writefile(prepare_path($opts->{root}, 'skip_no_data.txt'),   join("\n", @{$data->{no_data}}));
        writefile(prepare_path($opts->{root}, 'skip_old_data.txt'),  join("\n", @{$data->{old_data}}));
        writefile(prepare_path($opts->{root}, 'skip_protected.txt'), join("\n", @{$data->{protected}}));
        $result = $data->{result};

        if ($opts->{destination}) {
            writefile(prepare_path($opts->{root}, $opts->{destination}), to_json($result, pretty => TRUE));
            warn "Stop after store working data\n";
            return;
        }
    }

    process_data($opts, $result);
}

sub process_data {
    my ($opts, $data) = @_;

    my $app = get_app();
    my @diff;
    my @process;
    my %pages;
    my @errors;
    my %counts;

    my $start    = gettimeofday;
    my $last     = $start;
    my $progress = $opts->{progress} // 100;
    my $job      = $opts->{split} && defined $opts->{job} ? "($opts->{job}/$opts->{split})" : "";
    for my $block (@$data) {
        $counts{total}++;
        unless ($counts{total} % $progress) {
            my $now = gettimeofday;
            warn sprintf("progress%s: %d/%d - %.03f\n", $job, $counts{total}, scalar(@$data), $now - $last);
            $last = $now;
        }
        # для отладочных целей ограничиваем количество обрабатываемых блоков для каждой модели
        $counts{$block->{model}}++;
        if ($opts->{limit} and $opts->{limit} < $counts{$block->{model}}) {
            next;
        }
        try {
            $app->partner_db->transaction(
                sub {
                    my $orig_bk_data = fix_bk_data($block);
                    my ($new_bk_data, $table_data) = fix_block_data($block);
                    if (my $diff = diff_bk_data($block->{model}, $orig_bk_data, $new_bk_data)) {
                        writefile(
                            $file_diff,
                            to_json(
                                {
                                    block => $block,
                                    orig  => $orig_bk_data,
                                    new   => $new_bk_data,
                                    diff  => $diff,
                                    btd   => $table_data,
                                },
                                pretty => TRUE
                            ),
                            append => TRUE,
                            lock   => TRUE
                        );
                        throw GMHasDiff;
                    }

                    if ($opts->{validate} and !$block->{is_deleted}) {
                        my $accessor = $block->{model};
                        try {
                            $app->$accessor->do_action($block->{public_id}, 'edit',
                                caption => $block->{table_data}{caption});
                        }
                        catch {
                            my ($e) = @_;
                            my $message = $e->message;
                            eval {$message = from_json($message);};
                            writefile(
                                $file_check,
                                to_json(
                                    {
                                        block     => $block,
                                        orig      => $orig_bk_data,
                                        new       => $new_bk_data,
                                        validator => $message,
                                        btd       => $table_data,
                                    },
                                    pretty => TRUE
                                ),
                                append => TRUE,
                                lock   => TRUE
                            );
                            throw GMHasDiff;
                        };
                    }

                    $pages{$block->{page_id}} = undef;
                    writefile($file_fblock, $block->{public_id} . "\n", append => TRUE, lock => TRUE);
                    writefile(
                        $file_fdata,
                        to_json(
                            {
                                block => $block,
                                orig  => $orig_bk_data,
                                new   => $new_bk_data,
                                btd   => $table_data,
                            },
                            pretty => TRUE
                        ),
                        append => TRUE,
                        lock   => TRUE
                    );

                    unless ($opts->{commit}) {
                        throw GMHasDiff;
                    }
                }
            );
        }
        catch GMHasDiff with {} catch {
            my ($e) = @_;
            writefile(
                $file_error, to_json({%$block, error => $e->message}, pretty => TRUE),
                append => TRUE,
                lock   => TRUE
            );
        };
    }

    my $done = gettimeofday;
    warn sprintf("done: %0.3f\n", $done - $start);

    writefile($file_pages, join("\n", keys %pages), append => TRUE, lock => TRUE);
}

sub get_custom_bk_data {
    my ($opts) = @_;

    my $start = gettimeofday;
    my $app   = get_app();

    my %filter_by_model;
    if (my $block_list = $opts->{list} || $opts->{input_list}) {
        require Utils::PublicID;
        my $r = Utils::PublicID::group_public_ids_by_model($app, $block_list);
        for my $model (keys %$r) {
            if (@{$r->{$model}}) {
                $filter_by_model{$model} = [{public_id => $r->{$model}}];
            }
        }
    } else {
        %filter_by_model = map {$_ => []} @{get_models()};
    }

    my @result;
    my @skip_no_data;
    my @skip_old_data;
    my @skip_protected;

    for my $accessor (sort keys %filter_by_model) {
        my @db_fields = keys %{get_real_db_fields($app, $accessor)};
        my $model = $app->$accessor;

        my $fields = [keys %{$model->get_model_fields}];
        my $mw     = $model->get_multistate_by_name('working');
        my $md     = $model->get_multistate_by_name('deleted');
        my $mr     = $model->get_campaign_model->get_multistate_by_name('read_only') // 0;

        my $r0 = gettimeofday;
        my @date_filter;
        if ($model->get_model_fields->{create_date} and $opts->{date}) {
            push @date_filter, ['create_date', '<', $opts->{date}];
        }
        my $list = $model->get_all(
            fields => ['public_id', 'id', 'page_id', 'bk_data', 'multistate', @$fields],
            filter => [AND => [{is_custom_bk_data => 1}, @{$filter_by_model{$accessor}}, @date_filter]],
        );

        my $r1 = gettimeofday;
        for my $row (@$list) {
            if ($row->{is_protected} or $row->{page}{multistate} & $mr) {
                push @skip_protected, $row->{public_id};
                next;
            }

            my $bk_data_custom = delete $row->{bk_data};
            unless ($bk_data_custom) {
                push @skip_no_data, $row->{public_id};
                next;
            }
            $bk_data_custom = from_json($bk_data_custom);
            if (is_old_bk_data($bk_data_custom)) {
                push @skip_old_data, $row->{public_id};
                next;
            }

            my %table_data;
            @table_data{@db_fields} = @{$row}{@db_fields};
            my %data = (
                model          => $accessor,
                public_id      => $row->{public_id},
                id             => $row->{id},
                page_id        => $row->{page_id},
                bk_data_custom => $bk_data_custom,
                is_deleted     => $row->{is_deleted} ? 1 : 0,
                table_data     => \%table_data,
            );
            push @result, \%data;
        }
        my $r2 = gettimeofday;
        warn sprintf("%s: %d %0.3f %0.3f\n", $accessor, scalar(@$list), $r1 - $r0, $r2 - $r1);
    }

    my $get_data = gettimeofday();
    warn sprintf("get data: %0.3f\n", $get_data - $start);

    return +{
        result    => \@result,
        no_data   => \@skip_no_data,
        old_data  => \@skip_old_data,
        protected => \@skip_protected,
    };
}
