#!/usr/bin/perl
use strict;
use warnings FATAL => 'all';

=head1 пример вызова

   скрипт для генерации отчета

   bin/oneshots/GM/report_result.pl --root=/home/ie2018/tmp/fix_bk_data.PI-19674

=cut

use lib::abs qw(../../../lib .);

use Time::HiRes qw(gettimeofday);

use qbit;

use GMUtils;
use GMHasDiff;
use GMFixBkData qw(fix_bk_data);
use GMFixBlockData qw(fix_block_data);
use GMDiffBkData qw(diff_bk_data);

main_report();

sub main_report {
    my $opts = get_opts();

    my $diff  = get_list_hashes($opts, 'has_diff.txt');
    my $valid = get_list_hashes($opts, 'validation.txt');
    my $error = get_list_hashes($opts, 'errors.txt');
    my $fixed = get_list_hashes($opts, 'fixed_data.txt');
    my $input = from_json(readfile("$opts->{root}/result.json"));

    my %result;
    my @filter_data;
    my %dict;
    for my $row (@$input) {
        $result{$row->{public_id}} = {
            public_id  => $row->{public_id},
            model      => $row->{model},
            is_deleted => $row->{is_deleted},
            money      => 0,
            domain     => '<nodata>',
            login      => '<nodata>',
            fixed      => '<nodata>',
            c_date     => $row->{table_data}{create_date} // '<nodata>',
        };
        push @filter_data, sprintf("(%d,%d)", $row->{page_id}, $row->{id});
        $dict{$row->{page_id}}{$row->{id}} = $row->{public_id};
    }
    for my $row (@$valid) {
        next unless exists $result{$row->{block}{public_id}};
        $result{$row->{block}{public_id}}{fixed} = 'validation';
    }
    for my $row (@$error) {
        next unless exists $result{$row->{public_id}};
        $result{$row->{public_id}}{fixed} = 'error';
    }
    for my $row (@$fixed) {
        next unless exists $result{$row->{block}{public_id}};
        $result{$row->{block}{public_id}}{fixed} = 'ok';
    }
    for my $row (@$diff) {
        next unless exists $result{$row->{block}{public_id}};
        $result{$row->{block}{public_id}}{fixed} = 'diff_' . scalar(keys %{$row->{diff}});
    }

    my $money = get_money(\@filter_data, '2019-07-01');
    for my $row (@$money) {
        $result{$dict{$row->{page_id}}{$row->{block_id}}}{money} = $row->{money};
    }

    my $pages = get_site_and_login([keys %dict]);
    my %cnt_by_login;
    for my $row (@$pages) {
        for my $id (keys %{$dict{$row->{page_id}}}) {
            my $elem = $result{$dict{$row->{page_id}}{$id}};
            $elem->{login}  = $row->{login};
            $elem->{domain} = $row->{domain};

            my $cnt = $cnt_by_login{$row->{login}} //= {
                login => $row->{login},
                total => 0,
                ok    => 0,
                fail  => 0,
                valid => 0,
                error => 0,
                diff  => 0,
            };

            $cnt->{total}++;
            if ($elem->{fixed} eq 'ok') {
                $cnt->{ok}++;
            } elsif ($elem->{fixed} eq 'validation') {
                $cnt->{valid}++;
                $cnt->{fail}++;
            } elsif ($elem->{fixed} eq 'error') {
                $cnt->{error}++;
                $cnt->{fail}++;
            } else {
                $cnt->{diff}++;
                $cnt->{fail}++;
            }
        }
    }

    my @fields = qw(public_id model login domain money is_deleted c_date fixed);
    writefile(prepare_path($opts->{root}, "full_block_data.tsv"),
        to_csv(\@fields, hash_to_list_array(\%result, \@fields)));

    @fields = qw(login total ok fail valid error diff);
    writefile(prepare_path($opts->{root}, "block_by_login.tsv"),
        to_csv(\@fields, hash_to_list_array(\%cnt_by_login, \@fields)));
}

sub hash_to_list_array {
    my ($data, $fields) = @_;

    my @result;
    for my $row (values %$data) {
        push @result, [@{$row}{@$fields}];
    }

    return \@result;
}

sub get_money {
    my ($filter_data, $date) = @_;

    my @money;
    while (my @part = splice @$filter_data, 0, 1000) {
        get_money_data(\@money, \@part, $date);
    }

    return \@money;
}

sub get_money_data {
    my ($money, $filter_data, $date) = @_;

    my $filter     = join ",", @$filter_data;
    my $scale      = 10_000_000_000;
    my $ch_request = "
select
    page_id,
    block_id,
    round(sum(all_wo_nds)/$scale) as money
from
    statistics
where
    dt >= '$date'
    and (page_id,block_id) in ($filter)
group by
    page_id, block_id;
";
    $ch_request =~ s/\s+/ /g;
    my $ch_db = "clickhouse_mdb --server=prod";
    my $ch    = `echo "$ch_request" | $ch_db`;

    for my $row (split /\n/, $ch) {
        $row =~ s/\s+$//;
        my @row = split /\t/, $row;
        next unless $row[0] and $row[1];
        push @$money,
          {
            page_id  => $row[0],
            block_id => $row[1],
            money    => $row[2],
          };
    }

    return $money;
}

sub get_site_and_login {
    my ($pages) = @_;

    my $app = get_app;
    return $app->all_pages->get_all(filter => {page_id => $pages}, fields => [qw(page_id login domain)]);
}
