#!/usr/bin/perl
package main;

use lib::abs qw(
  ../../lib
  );
use feature 'say';
use qbit;

use Application;
use File::Temp qw(tempfile);
use List::Util qw(min);
use Text::CSV;
use Data::Dumper;

use DDP;

my $TRAFFIC_RATIO_TSV_PATH = lib::abs::path('../../mobile_traffic.tsv');
my $CHUNK_SIZE             = 1000;
my $CONVERSION             = {
    'everything_ok'                => {site_version => 'mobile', form_factor => undef},
    'poster_horizontal_with_limit' => {site_version => 'mobile', form_factor => undef, limit => undef},
};

my $app = Application->new();
$app->pre_run();
$app->set_cur_user({id => 0});

{
    no warnings 'redefine';
    *QBit::Application::check_rights = sub {1};
}

my %users = map {$_->{id} => $_->{login}} @{$app->users->get_all(fields => [qw(id login)])};

main();

sub main {
    my $blocks = get_blocks_with_high_mobile_traffic_ratio();

    say sprintf("HIGH MOBILE TRAFFIC BLOCKS: %s", scalar(@$blocks));

    my $conversion_result = try_conversion_to_mobile($blocks);

    report_conversion_result($conversion_result);

    say 'UPDATE BLOCKS';

    update_blocks($conversion_result->{blocks});

    say 'UPDATE PAGES';

    update_pages($conversion_result->{pages});

    say 'END';
}

sub get_blocks_with_high_mobile_traffic_ratio {
    my $iter = MobileTrafficRatio::tsv_iter($TRAFFIC_RATIO_TSV_PATH);

    my @blocks_with_big_ratio;

    while (my $ratio = $iter->()) {
        push @blocks_with_big_ratio, $ratio->block_public_id if $ratio->is_big_enough();
    }

    return \@blocks_with_big_ratio;
}

sub array_to_chunks_iter {
    my ($array) = @_;

    my $i = -1;
    return sub {
        $i += 1;
        my $start = $i * $CHUNK_SIZE;
        my $end   = min($#$array, ($i + 1) * $CHUNK_SIZE);
        my $chunk = [@{$array}[$start .. $end]];
        return $chunk if @$chunk;
        return undef;
    };
}

sub try_conversion_to_mobile {
    my ($ids) = @_;

    my $iter = array_to_chunks_iter($ids);

    my %pages_to_update;
    my %blocks_to_update = map {$_ => []} keys %$CONVERSION;
    my @conflicts;

    while (my $chunk = $iter->()) {

        my $blocks_chunk = $app->context_on_site_rtb->get_all(
            fields => $RtbBlock::FIELDS,
            filter => [AND => [{id => $chunk}, [site_version => '=' => 'desktop']]],
        );

        for my $block (map {RtbBlock->new(%$_)} @$blocks_chunk) {
            next if $block->is_100500;

            if (my $c = $block->conversion_to_mobile) {
                $pages_to_update{$block->page->{id}} //= ContextPage->new(%{$block->page}, blocks => []);
                $pages_to_update{$block->page->{id}}->add_block($block);
                push @{$blocks_to_update{$c}}, $block;
            } else {
                push @conflicts, $block;
            }
        }
    }

    return {
        pages     => \%pages_to_update,
        blocks    => \%blocks_to_update,
        conflicts => \@conflicts,
    };
}

sub update_pages {
    my ($pages) = @_;

    for my $id (keys %$pages) {
        my $page = $pages->{$id};
        $page->set_need_update if $page->multistate_allows_update;
    }
}

sub update_blocks {
    my ($blocks) = @_;

    for my $conversion_type (keys %$blocks) {

        my $iter = array_to_chunks_iter($blocks->{$conversion_type});

        while (my $chunk = $iter->()) {
            $app->partner_db->context_on_site_rtb->edit(
                $app->partner_db->filter(
                    [OR => [map {[AND => [[campaign_id => '=' => \$_->campaign_id], [id => '=' => \$_->id]]]} @$chunk]]
                ),
                $CONVERSION->{$conversion_type},
            );
        }
    }
}

sub report_conversion_result {
    my ($result) = @_;

    say sprintf("CONFLICTS: %s", scalar(@{$result->{conflicts}}));

    for my $conversion_type (keys %{$result->{blocks}}) {
        my ($blocks_fh, $blocks_filename) = tempfile("PI_10337_${conversion_type}_XXXXXX", DIR => './');
        binmode($blocks_fh, ':utf8');

        for my $block (@{$result->{blocks}{$conversion_type}}) {
            $blocks_fh->print(sprintf("%s %s\n", $block->campaign_id, $block->id));
        }

        close($blocks_fh) or die "close: $!";
    }

    my ($conflicts_fh, $conflicts_filename) = tempfile('PI_10337_conflicts_XXXXXX', DIR => './');
    binmode($conflicts_fh, ':utf8');

    my $tsv =
      Text::CSV_XS->new({binary => 1, sep_char => "\t", eol => "\n", escape_char => undef, quote_char => undef});
    $tsv->print($conflicts_fh, $RtbBlock::ARRAY_FIELDS);
    $tsv->column_names($RtbBlock::ARRAY_FIELDS);

    for my $conflict (@{$result->{conflicts}}) {
        $tsv->print($conflicts_fh, $conflict->to_array);
    }

    close($conflicts_fh) or die "close: $!";
}

package MobileTrafficRatio;

use Class::Accessor::Lite (
    new => 1,
    rw  => [qw(page_id block_id ratio)],
);

sub block_public_id {
    my ($self) = @_;

    return sprintf('R-A-%s-%s', $self->page_id, $self->block_id);
}

sub is_big_enough {
    my ($self) = @_;

    return $self->ratio >= 20 || $self->ratio == 0;
}

sub tsv_iter {
    my ($path) = @_;

    my $tsv = Text::CSV->new({binary => 1, sep_char => "\t", eol => "\n", escape_char => undef, quote_char => undef});

    open(my $fh, '<', $path) or die "open: $!";

    my %name_mapping = (
        PageID => 'page_id',
        ImpID  => 'block_id',
        DIF    => 'ratio',
    );

    $tsv->column_names(map {$name_mapping{$_}} @{$tsv->getline($fh)});

    return sub {
        MobileTrafficRatio->new(%{$tsv->getline_hr($fh) // return undef});
    };
}

package RtbBlock;

our $FIELDS;
our $ARRAY_FIELDS;

BEGIN {
    $FIELDS =
      [qw(campaign_id id site_version caption domain direct_block limit is_custom_format_direct page is_custom_bk_data)
      ];
    $ARRAY_FIELDS = [qw(page_id id login direct_block limit caption domain is_custom_format_direct is_custom_bk_data)];
}

use Class::Accessor::Lite (
    new => 1,
    rw  => [
        qw(campaign_id id site_version caption domain direct_block limit is_custom_format_direct page is_custom_bk_data)
    ],
);

sub conversion_to_mobile {
    my ($self) = @_;

    if ($self->direct_block eq 'posterHorizontal' && defined($self->limit) && $self->limit <= 1) {
        return 'poster_horizontal_with_limit';
    } elsif (
        $self->is_custom_format_direct
        || $self->direct_block eq 'posterHorizontal' && !defined($self->limit)
        || $self->direct_block eq 'posterVertical'   && $self->limit <= 2
        || $self->direct_block eq 'horizontal'       && $self->limit <= 2
        || $self->direct_block eq 'vertical'         && $self->limit <= 2
        || grep {
            $self->direct_block eq $_
        } (
            qw(
              240x400
              200x300
              250x250
              240x400
              300x300
              300x250
              320x50
              320x100
              336x280
              grid
              adaptive
              modernAdaptive
              extensibleMobile
              )
          )
      )
    {
        return 'everything_ok';
    } else {
        return undef;
    }
}

sub is_100500 {
    my ($self) = @_;

    return $self->id == 100500;
}

sub to_array {
    my ($self) = @_;

    return [
        $self->{campaign_id}, $self->{id}, $users{$self->{page}{owner_id}},
        $self->{direct_block}, $self->{limit} // 'undef', $self->{caption},
        $self->{domain}, $self->{is_custom_format_direct}, $self->{is_custom_bk_data},
    ];
}

1;

package ContextPage;

use Class::Accessor::Lite (
    new => 1,
    rw  => [qw(id page_id owner_id multistate domain allowed_amp allowed_turbo blocks)],
);

sub add_block {
    my ($self, $block) = @_;

    push @{$self->blocks}, $block;
}

sub multistate_allows_update {
    my ($self) = @_;

    return !($app->context_on_site_campaign->check_multistate_flag($self->multistate, 'need_approve')
        || $app->context_on_site_campaign->check_multistate_flag($self->multistate, 'rejected')
        || $app->context_on_site_campaign->check_multistate_flag($self->multistate, 'deleted'));
}

sub set_need_update {
    my ($self) = @_;

    my $status = 'OK';

    eval {$app->context_on_site_campaign->do_action($self->id, 'set_need_update');} or do {
        my $e = $@;
        $status = $e->message;
    };

    if ($status ne 'OK') {
        say sprintf("SET_NEED_UPDATE FAIL: %s %s", $self->page_id, $status);
    }

    return 1;
}

1;
