#!/usr/bin/perl

use strict;
use warnings FATAL => 'all';

use utf8;

use Data::Dumper;
$Data::Dumper::Sortkeys = 1;

use lib::abs '../../lib';
use qbit;
use Application;
use Utils::Logger qw( ERROR ERRORF INFO INFOF TRACE DEBUG );

my $page_id = 251316;

my $brands_data = <<BRANDS;
ТТК ЮФО (ВГРД)
ТТК (ОМСК)
ТТК (УФА)
ТТК
INTERNET ТЕЛЕФОНИЯ (ВЛД)
MANGO TELE.COM
MANGO-OFFICE
NEW-TEL
TELE 2
TELPHIN
UFANET
VEON
ZADARMA
АВАНГАРД (ИНТЕРНЕТ-ТЕЛЕФОНИЯ)
АКАДО ТЕЛЕКОМ
БИЛАЙН
ВОКРУГ СВЕТА С TELE2
ЕНИСЕЙ ТЕЛЕКОМ
КИНОАКАДЕМИЯ TELE2
КРУГОСВЕТКА С TELE2
МГТС
МЕЖРЕГИОНАЛЬНЫЙ ТРАНЗИТ ТЕЛЕКОМ
МОЙ TELE 2
МТС
НА СВЯЗИ TELE 2
ПЕТЕРБУРГСКАЯ ТЕЛЕФОННАЯ СЕТЬ
РОСТЕЛЕКОМ
РОСТЕЛЕКОМ-ВОЛГА
ТАКСИ 4G ОТ МТС
ЦИФРОВЫЕ ТЕЛЕФОННЫЕ СЕТИ (РНД)
ЯРТЕЛЕКОМ (ЯР)
HOT WIFI
Clever Wi-Fi
CIAWiFi
Wi-Fi System
Global Hotspot
BRANDS

my $only_check;
my $app;
my $tmp_all_rights;
my @bids;
my @blocks;

sub pre_run {
    $only_check = grep {/^only_?check$/} @ARGV;
    $app = Application->new();
    INFO 'START';
    $app->pre_run();
    $tmp_all_rights = $app->add_all_tmp_rights();
}

sub find_bids {
    TRACE 'find_bids START';
    my %bids;
    foreach my $brand (split /\n/, $brands_data) {
        my $result = $app->tns_dict_brand->get_all(
            'fields' => ['bid'],
            'filter' => {
                'name'   => $brand,
                'status' => 'A',
            },
        );
        if (@$result > 1) {
            ERRORF 'Found more than one brand with name "%s": %s', $brand, join(', ', map {$$_{'bid'}} @$result);
        } elsif (@$result) {
            $bids{$$result[0]{'bid'}} = $brand;
            INFOF "\t%d => %s", $$result[0]{'bid'}, $brand;
        } else {
            ERRORF 'Brand "%s" was not found', $brand;
        }
    }
    @bids = sort {$a <=> $b} keys %bids;
    TRACE 'find_bids END';
}

sub find_blocks {
    TRACE 'find_blocks START';
    my $result = $app->context_on_site_rtb->get_all(
        'fields' => ['id'],
        'filter' => {'page_id' => $page_id,},
    );
    @blocks = sort {$a <=> $b} map {$$_{'id'}} @$result;
    INFOF 'Blocks: [%s]', join ', ', @blocks;
    TRACE 'find_blocks END';
}

sub check {
    TRACE 'check START';
    my $result = $app->brands->get_all(
        'fields' => [qw( bid block_id cpm )],
        'filter' => {
            'page_id' => $page_id,
            'blocked' => FALSE,
            'bid'     => \@bids,
        },
    );
    if (@$result) {
        ERRORF('( page_id = %d, block_id = %d, bid = %d, cpm = %s ) already in table',
            $page_id, $$_{'block_id'}, $$_{'bid'}, ($$_{'cpm'} // 'NULL'))
          foreach @$result;
        return 0;
    } else {
        INFO 'There is no suspicious brands';
        return 1;
    }
    TRACE 'check END';
}

sub block {
    TRACE 'block START';
    foreach my $block_id (@blocks) {
        my $result = $app->brands->add(
            {
                'page_id'  => $page_id,
                'block_id' => $block_id,
            },
            [map {+{'bid' => $_, 'blocked' => TRUE, 'cpm' => undef,}} @bids]
        );
        INFOF 'Block %d brands for (page_id = %d, block_id = %d)', $result, $page_id, $block_id;
    }
    TRACE 'block END';
}

sub run {
    # Найти все идентификаторы брендов, соответствующие именам $brands_data
    find_bids();
    return unless @bids;    # Дальше не продолжаем, если ничего не нашли

    # Найти все блоки для заданного $page_id
    find_blocks();
    return unless @blocks;    # Дальше не продолжаем, если ничего не нашли

    # Дальше не продолжаем, если нашли не заблокированные бренды с cpm
    return unless check();

    # Дальше не продолжаем, если запуск был с целью проверки
    return if $only_check;

    block();                  # Блокируем
}

sub post_run {
    undef $tmp_all_rights;
    $app->post_run();
    INFO 'END';
    undef $app;
}

sub main {
    pre_run();
    run();
    post_run();
}

main();
