#!/usr/bin/perl

=encoding UTF-8

=head1 DESCRIPTION

  Скрипт устанавливает стратегию и добавляет DSP-Директ для видеоблоков имеющих категорию VH

  NOTE! Нужно только подключить прод базу

=head1 USAGE

    script.pl --blocks=instream:293335:1,inpage:276130:41,fullscreen:276130:42,inpage:222283:2

=head1 OPTIONS

  --no-debug   - запустить реальное изменение, по умолчанию только показывает подходящие блоки

=head1 RESULT

  Создаются файлы с результатами
    PI_13202-process.txt - измененные блоки
    PI_13202-skip.txt    - блоки с режимом "Редактировать транспорт"
    PI_13202-empty.txt   - страницы не имеющие измененных блоков(РТ)

=cut

use feature 'say';
use lib::abs qw(
  ../../lib
  );

use qbit;
use Application;
use PiConstants qw($SEPARATE_CPM_STRATEGY_ID);

use Getopt::Long qw();
use Pod::Usage;

my $DEBUG          = 1;
my $DEBUG_PAGE_IDS = [];

my $pages_filter;

my @strategy_kp_team = (
    strategy                  => $SEPARATE_CPM_STRATEGY_ID,
    video_cpm                 => 50,
    video_active              => 1,
    video_blocked             => 0,
    video_performance_cpm     => 25,
    video_performance_active  => 1,
    video_performance_blocked => 0,
    mincpm                    => undef,
);

my @strategy_other = (
    strategy                  => $SEPARATE_CPM_STRATEGY_ID,
    video_cpm                 => 100,
    video_active              => 1,
    video_blocked             => 0,
    video_performance_cpm     => 50,
    video_performance_active  => 1,
    video_performance_blocked => 0,
    mincpm                    => undef,
);

my %vpaid = (
    vpaid_enabled => 1,
    vpaid_timeout => 2000,
);

my @result_ok;
my @result_skip;
my @empty_page;

sub main {
    my $app = _get_app();

    my $blocks = get_blocks_for_category($app, $app->video_an_site_categories->get_category_ids_for_yahosting);

    my $pages = get_pages_by_blocks($app, $blocks);

    my ($count_pages, $count_blocks) = (0, 0);
    for my $page (@$pages) {
        my $strategy = $page->{login} eq 'kp-team' ? \@strategy_kp_team : \@strategy_other;
        $count_blocks += update_blocks($app, $page, $blocks->{$page->{id}}, $strategy);
        $count_pages++;
    }
    say "pages=$count_pages blocks=$count_blocks";
    writefile("PI_13202-process.txt", join("\n", @result_ok));
    writefile("PI_13202-skip.txt",    join("\n", @result_skip));
    writefile("PI_13202-empty.txt",   join("\n", @empty_page));
}

sub update_blocks {
    my ($app, $page, $blocks, $strategy) = @_;

    my $count = 0;
    for my $model (keys %$blocks) {
        for my $block (@{$blocks->{$model}}) {
            my $dsps = $block->{dsps};
            my %dsps;
            unless (grep {$_->{dsp_id} == 1} @$dsps) {
                %dsps = (dsps => [@$dsps, {dsp_id => 1}]);
            }

            (my $mn = $block->{multistate_name}) =~ s/[\n\r]+/ /g;
            if ($block->{is_custom_bk_data}) {
                push @result_skip, "$model:$page->{id}:$block->{id} - $mn";
            } else {
                push @result_ok, "$model:$page->{id}:$block->{id} -  $mn";
                $count++;
            }

            unless ($DEBUG || $block->{is_custom_bk_data}) {
                if ($app->$model->check_multistate_flag($block->{multistate}, 'deleted')) {
                    $app->partner_db->$model->edit($block, {@$strategy});
                    if (%dsps) {
                        $app->video_an_site_block_dsps->save_dsps(
                            {page_id => $block->{page_id}, block_id => $block->{id}},
                            $dsps{dsps}, product => $model);
                    }
                } else {
                    unless (eval {$app->$model->do_action($block, 'edit', %dsps, @$strategy); 1}) {
                        say "error in $model:$page->{id}:$block->{id}: $@";
                    }
                }
            }
        }
    }
    unless ($count || $page->{vpaid_enabled}) {
        (my $mn = $page->{multistate_name}) =~ s/[\n\r]+/ /g;
        push @empty_page, "$page->{id} - $mn";
    } elsif (!$DEBUG && !$page->{vpaid_enabled}) {
        if ($app->video_an_site->check_multistate_flag($page->{multistate}, 'deleted')) {
            $app->partner_db->video_an_site->edit($page, \%vpaid);
        } else {
            $app->video_an_site->do_action($page, 'edit', %vpaid);
        }
    }

    return $count;
}

sub get_blocks_for_category {
    my ($app, @categories) = @_;

    my %result;
    for my $model (qw(video_an_site_instream video_an_site_inpage video_an_site_fullscreen)) {
        my $filter = {category_id => \@categories};
        if ($pages_filter) {
            unless ($pages_filter->{$model}) {
                next;
            }
            $filter = ['AND', [$filter, @{$pages_filter->{$model}}]];
        }

        my $blocks = $app->$model->get_all(
            fields => [qw(page_id id strategy dsps multistate multistate_name is_custom_bk_data)],
            filter => $filter,
        );

        for my $block (@$blocks) {
            push @{$result{$block->{page_id}}{$model}}, $block;
        }
    }

    return \%result;
}

sub get_pages_by_blocks {
    my ($app, $blocks) = @_;

    my $pages = $app->video_an_site->get_all(
        fields => [qw(id multistate login vpaid_enabled multistate_name)],
        filter => [id => IN => [keys %$blocks]],
    );

    return $pages;
}

sub _get_app {

    _get_args();
    $ENV{LAZY_LOAD} = 1;

    my $app = Application->new();
    $app->pre_run();

    $app->set_cur_user({id => 0});

    no strict 'refs';
    no warnings 'redefine';
    *QBit::Application::check_rights = sub {1};

    return $app;
}

sub _get_args {
    my $blocks = '';
    my $help   = 0;

    Getopt::Long::GetOptions(
        #--- Obligatory
        'blocks:s' => \$blocks,
        'debug!'   => \$DEBUG,
        #---
        'help|?|h' => \$help,
    ) or pod2usage(1);

    pod2usage(-verbose => 2, -noperldoc => 1) if $help;

    if ($blocks) {
        my $pages = [map {[split /:/]} split /,/, $blocks];
        $pages_filter = {};
        for my $page (@$pages) {
            my $model = 'video_an_site_' . $page->[0];
            push @{$pages_filter->{$model}}, {'page_id' => $page->[1], 'id' => $page->[2]};
        }
        for my $model (keys %$pages_filter) {
            unless (@{$pages_filter->{$model}} == 1) {
                $pages_filter->{$model} = [['OR', $pages_filter->{$model}]];
            }
        }
    }
}

main();
