#!/usr/bin/perl

use qbit;

use Net::INET6Glue::INET_is_INET6;
use Config::IniFiles;
use File::Find;
use QBit::Gettext::Extract;
use Yandex::Tanker;

use File::Temp qw(tempfile);

use lib::abs qw(../../lib/);

use PiSecrets;

my $startrek_base_url = 'https://st.yandex-team.ru/';

sub fatal {
    print "$_[0]\n";
    exit(1);
}

sub get_ticket_link {
    my $cur_branch_name = `git rev-parse --abbrev-ref HEAD`;
    my ($ticket) = ($cur_branch_name =~ /^([A-Z]{2,}-[0-9]{3,})(\D|$)/);
    return $ticket;
}

sub to_file {
    my ($keysets, $ticket_link) = @_;

    my ($tmp_fh, $tmp_filename) = tempfile('tankerXXXXXX', SUFFIX => '.tjson', TMPDIR => TRUE, UNLINK => TRUE);

    binmode($tmp_fh, ":utf8");
    print $tmp_fh to_json({keysets => $keysets});
    close($tmp_fh);

    return $tmp_filename;
}

sub get_config {

    fatal('File ".tanker" does not exists.') unless -f '.tanker';

    my $config = Config::IniFiles->new(-file => '.tanker', -fallback => 'GENERAL', -nocase => TRUE)
      // fatal("File \".tanker\" is invalid:\n" . join("\n", @Config::IniFiles::errors));

    return $config;
}

sub get_tanker {
    my ($config) = @_;

    my $tanker_url;
    if (defined($config->val('GENERAL', 'server')) && $config->val('GENERAL', 'server') eq 'production') {
        $tanker_url = $Yandex::Tanker::PROD_URL;
    } elsif (defined($config->val('GENERAL', 'server')) && $config->val('GENERAL', 'server') eq 'test') {
        $tanker_url = $Yandex::Tanker::TEST_URL;
    } else {
        fatal('No "server" option in config file ".tanker". Expected it to be "production" or "test".');
    }

    my $tanker = Yandex::Tanker->new(
        {
            project => scalar($config->val('GENERAL', 'project-id')),
            url     => $tanker_url,
            token   => get_secret('tanker-pi2-token'),
        }
    );

    return $tanker;
}

sub make_change {
    my ($keysets) = @_;

    my $count   = 0;
    my $changed = {};
    for my $keyset (keys(%$keysets)) {
        for my $key (keys(%{$keysets->{$keyset}{'keys'}})) {
            if (my $m = $keysets->{$keyset}{'keys'}{$key}{translations}{ru}{form}) {
                next unless $m =~ /\bfullscreen\b/i;
                next unless $m =~ /\sfullscreen/i or $m =~ /fullscreen\s/i;
                my $row = $changed->{$keyset}{'keys'}{$key} = $keysets->{$keyset}{'keys'}{$key};
                (my $ru = $row->{translations}{ru}{form}) =~ s/\bfullscreen\b/Interstitial/ig;
                (my $en = $row->{translations}{en}{form}) =~ s/\bfullscreen\b/Interstitial/ig;
                $row->{translations} = {
                    ru => {form => $ru},
                    en => {form => $en},
                };
                $count++;
            }
        }
    }

    return ($changed, $count);
}

sub update_tanker {
    my ($tanker, $changed, $ticket_link, $branch) = @_;

    fatal('need branch') unless $branch;

    $tanker->_action_on_keyset(
        file   => to_file($changed, $ticket_link),
        keyset => [keys %$changed],
        format => 'tjson',
        action => 'merge',
        branch => $branch,
    );
}

sub main {
    my ($branch) = @_;

    my $config  = get_config();
    my $tanker  = get_tanker($config);
    my $keysets = from_json(Encode::decode_utf8($tanker->get_project_tjson(status => 'unapproved')))->{'keysets'};
    my ($changed, $count) = make_change($keysets);

    update_tanker($tanker, $changed, get_ticket_link(), $branch);

    warn "make $count\n";
}

my $branch = shift || 'master';

main($branch);
